package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnActiveParticipantsChanged
import io.dyte.core.controllers.DyteEventType.OnPeerAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerLeft
import io.dyte.core.controllers.DyteEventType.OnPeerPageUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerPinned
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerUnpinned
import io.dyte.core.controllers.DyteEventType.OnPeerVideoUpdate
import io.dyte.core.controllers.PageViewMode.GRID
import io.dyte.core.controllers.PageViewMode.PAGINATED
import io.dyte.core.models.DyteMeetingParticipant
import io.dyte.core.models.DyteParticipant
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.ParticipantFlags
import io.dyte.core.models.WaitListStatus.ACCEPTED
import io.dyte.core.models.WaitListStatus.REJECTED
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketGetPageModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerPinnedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAccepted
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAdded
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerClosed
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerRejected
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.delay
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlin.math.ceil

internal class ParticipantController(
  controllerContainer: IControllerContainer
) : IParticipantController, BaseController(controllerContainer) {

  private val waitlistedParticipants = arrayListOf<DyteMeetingParticipant>()
  private val joinedParticipants = arrayListOf<DyteMeetingParticipant>()
  private val activeParticipants = arrayListOf<DyteMeetingParticipant>()
  private val screenshareParticipants = arrayListOf<DyteMeetingParticipant>()

  private val hiddenParticipants = arrayListOf<DyteMeetingParticipant>()

  override val meetingRoomParticipants = DyteRoomParticipants(
    waitlistedParticipants,
    joinedParticipants,
    activeParticipants,
    screenshareParticipants,
    controllerContainer
  )

  private var _pinnedParticipant: DyteMeetingParticipant? = null

  override val pinnedParticipant: DyteMeetingParticipant?
    get() = _pinnedParticipant

  private var pageNumber: Int = 0
  private var maxVideoCount = 6
  private var viewMode: PageViewMode = GRID

  private var isLoadingPage = false

  private var canGoNextPage = false
  private var canGoPreviousPage = false
  private var pageCount: Int = 0

  private lateinit var gridPagesInfo: GridPagesInfo

  private var requestedPageNumber = 0

  override fun getPageNumber(): Int {
    return pageNumber
  }

  override fun getViewMode(): PageViewMode {
    return viewMode
  }

  override fun setViewMode(pageViewMode: PageViewMode) {
    viewMode = pageViewMode
    when (viewMode) {
      GRID -> {
        setPage(0)
      }
      PAGINATED -> {
        setPage(1)
      }
    }
  }

  override fun init() {
    maxVideoCount =
      controllerContainer.presetController.getMaxVideoCount()
  }

  /**
   * On peer joined
   *
   * when a remote peer joins, this does not contain audio track or video track
   *
   * @see onParticipantStreamConnected
   *
   * @param meetingPeerUser
   */
  override fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser) {
    println("DyteMobileClient | ParticipantController onPeerJoined ${meetingPeerUser.name}")
    val participant = getDyteParticipant(meetingPeerUser)
    if (participant.flags.hiddenParticipant || participant.flags.recorder) {
      hiddenParticipants.add(participant)
    } else {
      if (controllerContainer.presetController.permissions.showParticipantList.not()) {
        throw UnsupportedOperationException("not allowed to view participant list")
      }
      if (waitlistedParticipants.find { it.id == meetingPeerUser.id } != null) {
        waitlistedParticipants.remove(waitlistedParticipants.find { it.id == meetingPeerUser.id })
      }
      joinedParticipants.add(participant)
      updatePageCount()
      controllerContainer.eventController.triggerEvent(DyteEventType.OnPeerJoin(participant))
    }
  }

  override fun onPeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel) {
    var isHiddentParticipantLeft = false
    var peerToRemove = joinedParticipants.find { it.id == webSocketPeerLeftModel.peerId }
    if (peerToRemove == null) {
      peerToRemove = hiddenParticipants.find { it.id == webSocketPeerLeftModel.peerId }
      hiddenParticipants.remove(peerToRemove)
      isHiddentParticipantLeft = true
    }

    if (!isHiddentParticipantLeft) {
      peerToRemove?.let {
        println("DyteMobileClient | ParticipantController onPeerLeft ${it.name}")
        val screenSharePeer = screenshareParticipants.find { it.id == peerToRemove.id }
        if (screenSharePeer != null) {
          screenshareParticipants.remove(screenSharePeer)
          controllerContainer.eventController.triggerEvent(OnPeerScreenShareUpdate)
        }

        joinedParticipants.remove(peerToRemove)
        val removedFromActive = activeParticipants.removeAll { it.id == peerToRemove.id }
        println("DyteMobileClient | ParticipantController onPeerLeft removedFromActive $removedFromActive")
        if (removedFromActive) {
          updatePageCount()
          controllerContainer.eventController.triggerEvent(OnPeerPageUpdate(gridPagesInfo))
          controllerContainer.eventController.triggerEvent(OnActiveParticipantsChanged(activeParticipants))
        }
        if (peerToRemove.id == pinnedParticipant?.id) {
          println("DyteMobileClient | ParticipantController onPeerLeft this is pinned")
          _pinnedParticipant = null
          controllerContainer.eventController.triggerEvent(OnPeerUnpinned)
        }
        controllerContainer.eventController.triggerEvent(OnPeerLeft(peerToRemove))
      }
    }
  }

  private fun updatePageCount() {
    // always for active
    pageCount = 1

    // create chunks of maxVideoCount from joined
    val pages = ceil(joinedParticipants.size/maxVideoCount.toFloat()).toInt()
    // if only 1 chunk is present add nothing
    if (pages > 1) {
      pageCount += pages
    }

    // TODO: find better solution for this.
    if (screenshareParticipants.isNotEmpty() || controllerContainer.pluginsController.activePlugins.isNotEmpty()) {
      if (joinedParticipants.size in 3..6) {
         pageCount = 2
      }
    }

    canGoNextPage = pageNumber < (pageCount - 1)
    canGoPreviousPage = pageNumber > 0
    gridPagesInfo =
      GridPagesInfo(pageCount, pageNumber, canGoNextPage, canGoPreviousPage, pageCount > 1)
  }

  override fun onPeerAudioMuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
    participant?.let {
      participant._audioEnabled = false
      controllerContainer.eventController.triggerEvent(OnPeerAudioUpdate(participant))
    }
  }

  override fun onPeerAudioUnmuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
    participant?.let {
      participant._audioEnabled = true
      controllerContainer.eventController.triggerEvent(OnPeerAudioUpdate(participant))
    }
  }

  override fun onPeerPinned(webSocketPeerPinnedModel: WebSocketPeerPinnedModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerPinnedModel.peerId }
    participant?.let {
      _pinnedParticipant = participant
      controllerContainer.eventController.triggerEvent(OnPeerPinned(participant))
    }
  }

  override fun onPeerUnpinned() {
    _pinnedParticipant = null
    controllerContainer.eventController.triggerEvent(OnPeerUnpinned)
  }

  override fun onWaitlistPeerAdded(webSocketWaitlistPeerAdded: WebSocketWaitlistPeerAdded) {
    val participant = DyteParticipant(
      webSocketWaitlistPeerAdded.id,
      webSocketWaitlistPeerAdded.id,
      webSocketWaitlistPeerAdded.name,
      null,
      false,
      null,
      ParticipantFlags(false, false),
      controllerContainer
    )
    handleWaitlistedPeer(participant)
  }

  private fun handleWaitlistedPeer(participant: DyteMeetingParticipant) {
    waitlistedParticipants.add(participant)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnWaitListPeerJoined(participant))
  }

  override fun onWaitlistPeerAccepted(webSocketWaitlistPeerAccepted: WebSocketWaitlistPeerAccepted) {
    controllerContainer.selfController.getSelf()._waitListStatus = ACCEPTED
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnSelfWaitListStatusUpdate(
        ACCEPTED
      )
    )
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      controllerContainer.roomNodeController.joinRoom()
    }
  }

  override fun onWaitlistPeerRejected(webSocketWaitlistPeerRejected: WebSocketWaitlistPeerRejected) {
    controllerContainer.selfController.getSelf()._waitListStatus = REJECTED
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnSelfWaitListStatusUpdate(
        REJECTED
      )
    )
  }

  override fun onWaitlistPeerRejected(dyteMeetingParticipant: DyteMeetingParticipant) {
    waitlistedParticipants.remove(dyteMeetingParticipant)
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnWaitListPeerRejected(
        dyteMeetingParticipant
      )
    )
  }

  override fun onWaitlistPeerClosed(webSocketWaitlistPeerClosed: WebSocketWaitlistPeerClosed) {
    val closedPeer = waitlistedParticipants.find { it.id == webSocketWaitlistPeerClosed.id }
    closedPeer?.let {
      waitlistedParticipants.remove(closedPeer)
      controllerContainer.eventController.triggerEvent(DyteEventType.OnWaitListPeerClosed(closedPeer))
    }
  }

  override fun onParticipantVideoMuted(webSocketConsumerClosedModel: WebSocketConsumerClosedModel) {
    val consumerType =
      controllerContainer.mediaSoupController.getConsumerType(
        requireNotNull(webSocketConsumerClosedModel.id)
      )
    if ("audio" == consumerType) {
      return
    }
    webSocketConsumerClosedModel.id?.let {
      val appdata =
        controllerContainer.mediaSoupController.getAppDataFromConsumerId(it)
      val peerId = appdata?.peerId
      peerId?.let {
        val participant = joinedParticipants.find { it.id == peerId }
        participant?.let {
          if (appdata.screenShare == false) {
            participant._videoEnabled = false
            controllerContainer.eventController.triggerEvent(OnPeerVideoUpdate(participant))
          }
        }
      }
    }
  }

  override fun onParticipantVideoUnmuted(webSocketConsumerClosedModel: WebSocketConsumerResumedModel) {
    webSocketConsumerClosedModel.id?.let {
      val appdata =
        controllerContainer.mediaSoupController.getAppDataFromConsumerId(it)
      val peerId = appdata?.peerId
      peerId?.let {
        val participant = joinedParticipants.find { it.id == peerId }
        participant?.let {
          participant._videoEnabled = true
          controllerContainer.eventController.triggerEvent(OnPeerVideoUpdate(participant))
        } ?: {
          println("DyteMobileClient | ParticipantController onParticipantVideoUnmuted could not find peer with id $peerId")
        }
      }
      controllerContainer.platformUtilsProvider.getMediaSoupUtils().resumeConsumer(it)
    }
  }

  override fun onParticipantStreamConnected(
    participant: DyteMeetingParticipant
  ) {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnPeerUpdate(participant))
  }

  private fun getDyteParticipant(meetingPeerUser: WebSocketMeetingPeerUser): DyteMeetingParticipant {
    val flags = ParticipantFlags(
      meetingPeerUser.flags?.recordere ?: false,
      meetingPeerUser.flags?.hiddenParticipant ?: false
    )
    return DyteParticipant(
      requireNotNull(meetingPeerUser.id),
      requireNotNull(meetingPeerUser.userId),
      requireNotNull(meetingPeerUser.name),
      meetingPeerUser.picture,
      meetingPeerUser.isHost ?: false,
      meetingPeerUser.clientSpecificId
        ?: controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid(),
      flags,
      controllerContainer
    )
  }

  override fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel) {
    if (controllerContainer.presetController.permissions.showParticipantList.not()) {
      throw UnsupportedOperationException("not allowed to view participant list")
    }
    webSocketRoomStateModel.roomState?.peers?.forEach {
      onPeerJoined(it)
    }
  }

  override fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    if (controllerContainer.presetController.permissions.hostPermissions.acceptWaitingRequests) {
      webSocketJoinRoomModel.waitlistedPeers?.forEach {
        val participant = DyteParticipant(
          requireNotNull(it.id),
          requireNotNull(it.id),
          requireNotNull(it.name),
          null,
          false,
          null,
          ParticipantFlags(false, false),
          controllerContainer
        )
        handleWaitlistedPeer(participant)
      }
    }
    if (webSocketJoinRoomModel.pinnedPeerId?.isNotEmpty() == true) {
      val pinnedParticipant =
        joinedParticipants.find { it.id == webSocketJoinRoomModel.pinnedPeerId }
      pinnedParticipant?.let {
        _pinnedParticipant = pinnedParticipant
      }
    }
  }

  override fun onSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel) {
    println("DyteMobileClient | ParticipantController onSelectedPeers ")
    if (requestedPageNumber == 0) {
      val peerIds = arrayListOf<String>()
      peerIds.addAll(webSocketSelectedPeersModel.peerIds ?: emptyList())
      if (webSocketSelectedPeersModel.peerIds?.contains(controllerContainer.selfController.getSelf().id) == false && (webSocketSelectedPeersModel.peerIds?.size
          ?: 0) < maxVideoCount
      ) {
        if (controllerContainer.metaController.getRoomType() == "WEBINAR") {
          if (controllerContainer.webinarController.isPresenting()) {
            peerIds.add(controllerContainer.selfController.getSelf().id)
          }
        } else {
          println("DyteMobileClient | ParticipantController onSelectedPeers adding self")
          peerIds.add(controllerContainer.selfController.getSelf().id)
        }
      }
      processNewPage(peerIds, 0)
    }
  }

  private fun processNewPage(peerIds: List<String>, pageNumber: Int) {
    activeParticipants.clear()
    peerIds.forEach { peerId ->
      val peer = joinedParticipants.find { it.id == peerId }
      peer?.let {
        activeParticipants.add(peer)
      }
    }
    this.pageNumber = pageNumber
    updatePageCount()
    controllerContainer.eventController.triggerEvent(OnActiveParticipantsChanged(activeParticipants))
    controllerContainer.eventController.triggerEvent(OnPeerPageUpdate(gridPagesInfo))
  }

  override fun onPageLoaded(webSocketGetPageModel: WebSocketGetPageModel, newPageNumber: Int) {
    println("DyteMobileClient | ParticipantController onPageLoaded ${webSocketGetPageModel.peerIds?.size} for page $newPageNumber")
    processNewPage(webSocketGetPageModel.peerIds ?: emptyList(), newPageNumber)
  }

  override fun shouldShowPaginator(): Boolean {
    return joinedParticipants.size > maxVideoCount
  }

  override fun getMaxVideoCountPerPage(): Int {
    return maxVideoCount
  }

  override fun onPeerScreenShareStarted(participant: DyteMeetingParticipant, screenshareTrack: Any) {
    println("DyteMobileClient | ParticipantController onPeerScreenShareStarted ${participant.name}")
    val screenShareParticipant = DyteParticipant("screenshare-" + participant.id, participant.userId, participant.name + "'s screen", participant.picture, participant.isHost, participant.clientSpecificId, participant.flags, controllerContainer)
    screenShareParticipant._isScreenshareTrack = true
    screenShareParticipant._screenShareTrack = screenshareTrack
    screenshareParticipants.add(screenShareParticipant)
    updatePageCount()
    controllerContainer.eventController.triggerEvent(OnPeerPageUpdate(gridPagesInfo))
    controllerContainer.eventController.triggerEvent(OnPeerScreenShareUpdate)
  }

  override fun onPeerScreenSharedEnded(participant: DyteMeetingParticipant) {
    println("DyteMobileClient | ParticipantController onPeerScreenSharedEnded ${participant.name}")
    controllerContainer.platformUtilsProvider.getVideoUtils().destroyView(screenshareParticipants.find { it.id == "screenshare-" + participant.id }!!)
    screenshareParticipants.removeAll { it.id == "screenshare-" + participant.id }
    updatePageCount()
    controllerContainer.eventController.triggerEvent(OnPeerPageUpdate(gridPagesInfo))
    controllerContainer.eventController.triggerEvent(OnPeerScreenShareUpdate)
  }

  override fun setPage(newPageNumber: Int) {
    if (newPageNumber < 0) {
      throw IllegalArgumentException("page number cant be less than 0")
    }
    requestedPageNumber = newPageNumber
    isLoadingPage = true
    val content = HashMap<String, JsonElement>()
    content["pageNum"] = JsonPrimitive(newPageNumber)
    val prevPageResponse =
      controllerContainer.socketController.sendMessageSync(
        OutboundMeetingEventType.GET_PAGE,
        JsonObject(content)
      )
    val pageModel =
      controllerContainer.socketMessageResponseParser.parseResponse(prevPageResponse).payload as WebSocketGetPageModel
    if (pageModel.peerIds?.isEmpty() == false) {
      onPageLoaded(pageModel, newPageNumber)
    }
  }

  override fun onSelfJoined() {
    joinedParticipants.add(controllerContainer.selfController.getSelf())
    activeParticipants.add(controllerContainer.selfController.getSelf())
    updatePageCount()
    controllerContainer.selfController.onEnteredInRoom()
    controllerContainer.eventController.triggerEvent(OnActiveParticipantsChanged(activeParticipants))
    controllerContainer.eventController.triggerEvent(OnPeerPageUpdate(gridPagesInfo))
  }

  override fun canGoNextPage(): Boolean {
    return canGoNextPage
  }

  override fun canGoPreviousPage(): Boolean {
    return canGoPreviousPage
  }

  override fun getPageCount(): Int {
    return pageCount
  }

  private fun printRoomParticipants() {
    val stringBuilder = StringBuilder()
    stringBuilder.append("DyteMobileClient")
    stringBuilder.appendLine()
    stringBuilder.append("Joined participants count : ${joinedParticipants.size}")
    stringBuilder.appendLine()
    joinedParticipants.map {
      stringBuilder.append(it.name)
      stringBuilder.append("\t")
    }
    stringBuilder.appendLine()

    stringBuilder.append("Active participants count : ${activeParticipants.size}")
    stringBuilder.appendLine()
    activeParticipants.map {
      stringBuilder.append(it.name)
      stringBuilder.append("\t")
    }
    stringBuilder.appendLine()

    if(this::gridPagesInfo.isInitialized) {
      stringBuilder.append("grid info")
      stringBuilder.appendLine()
      stringBuilder.append(gridPagesInfo)
      stringBuilder.appendLine()
    }

    if (pinnedParticipant != null) {
      stringBuilder.append("pinned participant ${pinnedParticipant?.name}")
    }
    stringBuilder.appendLine()
    println("DyteMobileClient | ParticipantController printRoomParticipants")
    println("<<<<<<<<<<<<${stringBuilder}>>>>>>>>>>>>>>")
  }

  override fun getVideoView(dyteMeetingParticipant: DyteMeetingParticipant): Any {
    return controllerContainer.platformUtilsProvider.getVideoUtils().getVideoView(dyteMeetingParticipant)
  }
}

interface IParticipantController {
  val meetingRoomParticipants: DyteRoomParticipants
  val pinnedParticipant: DyteMeetingParticipant?

  fun getPageNumber(): Int
  fun setViewMode(pageViewMode: PageViewMode)
  fun getViewMode(): PageViewMode

  fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel)
  fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel)

  fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser)
  fun onPeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel)

  fun onPeerAudioMuted(webSocketPeerMuteModel: WebSocketPeerMuteModel)
  fun onPeerAudioUnmuted(webSocketPeerMuteModel: WebSocketPeerMuteModel)

  fun onPeerPinned(webSocketPeerPinnedModel: WebSocketPeerPinnedModel)
  fun onPeerUnpinned()

  fun onPeerScreenShareStarted(participant: DyteMeetingParticipant, screenshareTrack: Any)
  fun onPeerScreenSharedEnded(participant: DyteMeetingParticipant)

  fun onParticipantStreamConnected(participant: DyteMeetingParticipant)

  fun onParticipantVideoMuted(webSocketConsumerClosedModel: WebSocketConsumerClosedModel)
  fun onParticipantVideoUnmuted(webSocketConsumerClosedModel: WebSocketConsumerResumedModel)

  fun onSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel)
  fun onPageLoaded(webSocketGetPageModel: WebSocketGetPageModel, newPageNumber: Int)

  fun onWaitlistPeerAdded(webSocketWaitlistPeerAdded: WebSocketWaitlistPeerAdded)
  fun onWaitlistPeerAccepted(webSocketWaitlistPeerAccepted: WebSocketWaitlistPeerAccepted)
  fun onWaitlistPeerRejected(webSocketWaitlistPeerRejected: WebSocketWaitlistPeerRejected)
  fun onWaitlistPeerClosed(webSocketWaitlistPeerClosed: WebSocketWaitlistPeerClosed)
  fun onWaitlistPeerRejected(dyteMeetingParticipant: DyteMeetingParticipant)

  fun shouldShowPaginator(): Boolean
  fun getMaxVideoCountPerPage(): Int

  fun setPage(newPageNumber: Int)

  fun canGoNextPage(): Boolean
  fun canGoPreviousPage(): Boolean
  fun getPageCount(): Int

  fun onSelfJoined()

  fun getVideoView(dyteMeetingParticipant: DyteMeetingParticipant): Any
}

enum class PageViewMode {
  GRID,
  PAGINATED;
}

data class GridPagesInfo(
  val pageCount: Int,
  val currentPageNumber: Int,
  val isNextPagePossible: Boolean,
  val isPreviousPagePossible: Boolean,
  val shouldShowPaginator: Boolean
) {
  fun toMap(): Map<String, Any?> {
    val args = HashMap<String, Any?>()
    args["pageCount"] = pageCount
    args["currentPageNumber"] = currentPageNumber
    args["isNextPagePossible"] = isNextPagePossible
    args["isPreviousPagePossible"] = isPreviousPagePossible
    args["shouldShowPaginator"] = shouldShowPaginator
    return args
  }
}