package io.dyte.core.network.info

import io.dyte.core.network.models.ParticipantPreset
import io.dyte.core.network.models.UserDataWrapper
import io.dyte.core.network.models.UserDataWrapperV1
import io.dyte.core.network.models.UserPresetDataWrapper

data class ParticipantInfo (
  val id: String,
  val name: String,
  val email: String?,
  val picture: String?,
  val loggedIn: Boolean,
  val clientSpecificId: String,
  val organizationId: String,
  val scope: String?,
  val hiddenParticipant: Boolean,
  val isRecorder: Boolean,
  val presetInfo: PresetInfo
) {
  companion object {
    fun getParticipantInfoFromV1(userData: UserDataWrapperV1, userPresetData: UserPresetDataWrapper): ParticipantInfo {
      val presetInfo = PresetInfo.getPresetInfoFromV1(userPresetData)
      return ParticipantInfo(userData.user.id, userData.user.name, null, null, false, userData.user.clientSpecificId, userData.user.organizationId, null, false, false, presetInfo)
    }

    fun getParticipantInfoFromV2(userData: UserDataWrapper): ParticipantInfo {
      val preset = PresetInfo.getPresetInfoFromV2(userData.data.preset)
      val participant = userData.data.participant
      return ParticipantInfo(participant.id, participant.name, null, null, false, participant.clientSpecificId, participant.organizationId, null, false, false, preset)
    }
  }
}

data class PresetInfo(
  val viewType: String,
  val name: String,
  val waitingRoomType: WaitingRoomType,
  val gridInfo: GridInfo,
  val permissions: ParticipantPermissions
) {
  companion object {
    fun getPresetInfoFromV1(userPresetData: UserPresetDataWrapper) : PresetInfo {
      userPresetData.data?.preset?.let { preset ->
        val participantPermissions = ParticipantPermissions.getParticipantPermissionsFromV1(userPresetData)
        return PresetInfo(preset.permissions?.viewType ?: "", preset.presetName ?: "", WaitingRoomType.valueOf(preset.permissions?.waitingRoomType?.name ?: ""), GridInfo(preset.theme?.grid?.multi?.maxVideoCount ?: 6), participantPermissions)
      } ?: run {
        throw IllegalArgumentException("not enough data to build preset info from v1")
      }
    }

    fun getPresetInfoFromV2(participantPreset: ParticipantPreset) : PresetInfo {
      val participantPermissions = ParticipantPermissions.getParticipantPermissionsFromV2(participantPreset)
      // TODO : fix grid info
      return PresetInfo(participantPreset.config.viewType, participantPreset.name, WaitingRoomType.valueOf(participantPreset.permissions.waitingRoomType),GridInfo(6), participantPermissions)
    }
  }
}

data class GridInfo(
  val maxParticipantsPerPage: Int
)

data class ParticipantPermissions(
  val canEditDisplayName: Boolean,
  val showParticipantList: Boolean,
  val hostPermissions: HostPermissions,
  val pluginPermissions: PluginPermissions,
  val pollPermissions: PollPermissions,
  val mediaPermissions: MediaPermissions,
  val chatPermission: ChatPermissions
) {
  companion object {
    fun getParticipantPermissionsFromV1(userPresetData: UserPresetDataWrapper): ParticipantPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        val hostPermissions = HostPermissions.getHostPermissionsFromV1(userPresetData)
        val pluginPermissions = PluginPermissions.getPluginPermissionsFromV1(userPresetData)
        val pollPermissions = PollPermissions.getPollPermissionsFromV1(userPresetData)
        val mediaPermissions = MediaPermissions.getMediaPermissionsFromV1(userPresetData)
        val chatPermissions = ChatPermissions.getChatPermissionsFromV1(userPresetData)
        return ParticipantPermissions(permissions.canEditDisplayName?: false, permissions.showParticipantList ?: false,
        hostPermissions, pluginPermissions, pollPermissions, mediaPermissions, chatPermissions)
      } ?: run {
        throw IllegalArgumentException("not enough data to build participant permissions from v1")
      }
    }

    // TODO : fix can edit display name
    fun getParticipantPermissionsFromV2(participantPreset: ParticipantPreset): ParticipantPermissions {
      val hostPermissions = HostPermissions.getHostPermissionsFromV2(participantPreset)
      val pluginPermissions = PluginPermissions.getPluginPermissionsFromV2(participantPreset)
      val pollPermissions = PollPermissions.getPollPermissionsFromV2(participantPreset)
      val mediaPermissions = MediaPermissions.getMediaPermissionsFromV2(participantPreset)
      val chatPermissions = ChatPermissions.getChatPermissionsFromV2(participantPreset)
      return ParticipantPermissions(true, participantPreset.permissions.showParticipantList,
        hostPermissions, pluginPermissions, pollPermissions, mediaPermissions, chatPermissions)
    }
  }
}

data class HostPermissions(
  val canPresent: Boolean,
  val canAcceptPresentRequests: Boolean,
  val acceptWaitingRequests: Boolean,
  val canDisableParticipantAudio: Boolean,
  val canDisableParticipantVideo: Boolean,
  val canKickParticipant: Boolean,
  val canPinParticipant: Boolean,
  val canStartRecording: Boolean,
) {
  companion object {
    fun getHostPermissionsFromV1(userPresetData: UserPresetDataWrapper): HostPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return HostPermissions(permissions.canPresent?: false || permissions.viewType == "GROUP_CALL",
          permissions.acceptPresentRequests ?: false,
          permissions.acceptWaitingRequests?: false,
        permissions.canDisableParticipantAudio?: false,
        permissions.canDisableParticipantVideo?: false,
        permissions.kickParticipant?: false,
        permissions.pinParticipant?: false,
        permissions.canRecord?:false)
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build host permissions from v1 api")
      }
    }

    // TODO : fix canPresent, canAcceptPresentRequest,
    fun getHostPermissionsFromV2(participantPreset: ParticipantPreset): HostPermissions {
      return HostPermissions(false,
        false,
        participantPreset.permissions.acceptWaitingRequests,
        participantPreset.permissions.disableParticipantAudio,
        participantPreset.permissions.disableParticipantVideo,
        participantPreset.permissions.kickParticipant,
        participantPreset.permissions.pinParticipant,
        participantPreset.permissions.canRecord)
    }
  }
}

data class PluginPermissions(
  val canClose: Boolean,
  val canStart: Boolean,
  val canEditAcl: Boolean
) {
  companion object {
    fun getPluginPermissionsFromV1(userPresetData: UserPresetDataWrapper): PluginPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return PluginPermissions(permissions.plugins?.canClose?: false,
          permissions.plugins?.canStart ?: false,
          permissions.plugins?.canEditAcl?: false)
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build plugins permissions from v1 api")
      }
    }

    fun getPluginPermissionsFromV2(participantPreset: ParticipantPreset): PluginPermissions {
      return PluginPermissions(participantPreset.permissions.plugins.canClose,
        participantPreset.permissions.plugins.canStart,
        false)
    }
  }
}

data class PollPermissions(
  val canCreate: Boolean,
  val canVote: Boolean,
  val canView: Boolean
) {
  companion object {
    fun getPollPermissionsFromV1(userPresetData: UserPresetDataWrapper): PollPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return PollPermissions(permissions.polls?.canCreate?: false,
          permissions.polls?.canVote ?: false,
          permissions.polls?.canView?: false)
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build polls permissions from v1 api")
      }
    }

    fun getPollPermissionsFromV2(participantPreset: ParticipantPreset): PollPermissions {
      return PollPermissions(participantPreset.permissions.polls.canCreate,
        participantPreset.permissions.polls.canVote,
        participantPreset.permissions.polls.canView)
    }
  }
}

data class MediaPermissions(
  val video: VideoPermissions,
  val canPublishAudio: Boolean,
) {
  companion object {
    fun getMediaPermissionsFromV1(userPresetData: UserPresetDataWrapper): MediaPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return MediaPermissions(
          VideoPermissions.getVideoPermissionsFromV1(userPresetData),
          permissions.produce?.audio ?: false)
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build media permissions from v1 api")
      }
    }

    fun getMediaPermissionsFromV2(participantPreset: ParticipantPreset): MediaPermissions {
      return MediaPermissions(
        VideoPermissions.getVideoPermissionsFromV2(participantPreset),
        participantPreset.permissions.media.audio.canProduce == "ALLOWED")
    }
  }
}

data class VideoPermissions(
  val canPublish: Boolean,
  val quality: String,
  val frameRate: Int
) {
  companion object {
    fun getVideoPermissionsFromV1(userPresetData: UserPresetDataWrapper): VideoPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return VideoPermissions(permissions.produce?.video?.allow ?: false || permissions.viewType == "GROUP_CALL",
          permissions.produce?.video?.quality ?: "720p",
          permissions.produce?.video?.frameRate ?: 30)
      } ?: run {
        throw IllegalArgumentException("not sufficient data to build video permissions from v1 api")
      }
    }

    // TODO : fix quality and framerate
    fun getVideoPermissionsFromV2(participantPreset: ParticipantPreset): VideoPermissions {
      return VideoPermissions(participantPreset.permissions.media.video.canProduce == "ALLOWED" || participantPreset.config.viewType == "GROUP_CALL",
        "720p",
        30)
    }
  }
}

data class ChatPermissions(
  val canSend: Boolean,
  val text: Boolean,
  val canSendFiles: Boolean
) {
  companion object {
    fun getChatPermissionsFromV1(userPresetData: UserPresetDataWrapper): ChatPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return ChatPermissions(permissions.chat?.chatPublicProps?.canSend ?: false,
          permissions.chat?.chatPublicProps?.text ?: false,
          permissions.chat?.chatPublicProps?.files ?: false)
      } ?: run {
        throw IllegalArgumentException("not sufficent data to build chat permissions from v1 api")
      }
    }

    fun getChatPermissionsFromV2(participantPreset: ParticipantPreset): ChatPermissions {
      return ChatPermissions(participantPreset.permissions.chat.public.canSend,
        participantPreset.permissions.chat.public.text,
        participantPreset.permissions.chat.public.files)
    }
  }
}

enum class WaitingRoomType {
  SKIP,
  ON_ACCEPT,
  SKIP_ON_ACCEPT,
  SKIP_ON_PRIVILEGED_USER_ENTRY;
}
