package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnActiveParticipantsChanged
import io.dyte.core.controllers.DyteEventType.OnActiveSpeakerChanged
import io.dyte.core.controllers.DyteEventType.OnAudioDevicesChanged
import io.dyte.core.controllers.DyteEventType.OnCameraClosed
import io.dyte.core.controllers.DyteEventType.OnCameraDisconnected
import io.dyte.core.controllers.DyteEventType.OnCameraError
import io.dyte.core.controllers.DyteEventType.OnCameraFreezed
import io.dyte.core.controllers.DyteEventType.OnCameraOpening
import io.dyte.core.controllers.DyteEventType.OnFirstFrameAvailable
import io.dyte.core.controllers.DyteEventType.OnHostKicked
import io.dyte.core.controllers.DyteEventType.OnMeetingMessagesReceived
import io.dyte.core.controllers.DyteEventType.OnMeetingPollsReceived
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingError
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingStateUpdate
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingStopped
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomDisconnected
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitCompleted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoined
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinedWithoutCameraPermission
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinedWithoutMicPermission
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeave
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeaveStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingStateChanged
import io.dyte.core.controllers.DyteEventType.OnMicrophoneError
import io.dyte.core.controllers.DyteEventType.OnMicrophoneInitError
import io.dyte.core.controllers.DyteEventType.OnMicrophoneStartError
import io.dyte.core.controllers.DyteEventType.OnNewMeetingMessageReceived
import io.dyte.core.controllers.DyteEventType.OnNewMeetingPollReceived
import io.dyte.core.controllers.DyteEventType.OnNoActiveSpeaker
import io.dyte.core.controllers.DyteEventType.OnPeerAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerJoin
import io.dyte.core.controllers.DyteEventType.OnPeerLeft
import io.dyte.core.controllers.DyteEventType.OnPeerPageUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerPinned
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerUnpinned
import io.dyte.core.controllers.DyteEventType.OnPeerUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerVideoUpdate
import io.dyte.core.controllers.DyteEventType.OnPluginDisabled
import io.dyte.core.controllers.DyteEventType.OnPluginEnabled
import io.dyte.core.controllers.DyteEventType.OnPluginMessage
import io.dyte.core.controllers.DyteEventType.OnProximityChanged
import io.dyte.core.controllers.DyteEventType.OnSelfAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnSelfVideoUpdate
import io.dyte.core.controllers.DyteEventType.OnSelfWaitListStatusUpdate
import io.dyte.core.controllers.DyteEventType.OnWebinarStoppedPresenting
import io.dyte.core.controllers.DyteEventType.OnWaitListPeerAccepted
import io.dyte.core.controllers.DyteEventType.OnWaitListPeerClosed
import io.dyte.core.controllers.DyteEventType.OnWaitListPeerJoined
import io.dyte.core.controllers.DyteEventType.OnWaitListPeerRejected
import io.dyte.core.controllers.DyteEventType.OnWebinarPresentRequest
import io.dyte.core.listeners.DyteCameraEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteMicrophoneEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DyteParticipantUpdateListener
import io.dyte.core.listeners.DytePermissionEventsListener
import io.dyte.core.listeners.DytePluginEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.models.DyteChatMessage
import io.dyte.core.models.DyteMeetingParticipant
import io.dyte.core.models.DytePlugin
import io.dyte.core.models.DytePollMessage
import io.dyte.core.models.WaitListStatus
import kotlinx.serialization.json.JsonObject

internal class EventController(controllerContainer: IControllerContainer) :
  BaseController(controllerContainer), IEventController {
  private val selfEventsListeners = arrayListOf<DyteSelfEventsListener>()
  private val roomEventsListeners = arrayListOf<DyteMeetingRoomEventsListener>()
  private val participantEventsListeners = arrayListOf<DyteParticipantEventsListener>()
  private val cameraEventsListeners = arrayListOf<DyteCameraEventsListener>()
  private val microphoneEventsListeners = arrayListOf<DyteMicrophoneEventsListener>()
  private val permissionEventsListeners = arrayListOf<DytePermissionEventsListener>()
  private val pluginEventsListeners = arrayListOf<DytePluginEventsListener>()
  private val participantUpdateListeners = hashMapOf<String, ArrayList<DyteParticipantUpdateListener>>()

  override fun init() {
    // no-op
  }

  override fun triggerEvent(eventType: DyteEventType) {
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
      triggerEventInternal(eventType)
    }
  }

  private fun triggerEventInternal(eventType: DyteEventType) {
    when (eventType) {
      OnMeetingRoomInitStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitStarted()
        }
      }

      OnMeetingRoomInitCompleted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitCompleted()
        }
      }

      is OnMeetingRoomInitFailed -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitFailed(eventType.exception)
        }
      }

      OnMeetingRoomLeaveStarted -> {
        ArrayList(selfEventsListeners).forEach { selfEventListener ->
          selfEventListener.onMeetingRoomLeaveStarted()
        }
      }

      OnMeetingRoomLeave -> {
        ArrayList(selfEventsListeners).forEach { selfEventListener ->
          selfEventListener.onMeetingRoomLeft()
        }
      }

      is OnMeetingStateChanged -> {
        controllerContainer.connectionController.onSocketStateChanged(eventType.state)
      }

      OnMeetingRoomJoinStarted -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onMeetingRoomJoinStarted()
        }
      }

      is OnMeetingRoomJoined -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onMeetingRoomJoined()
          selfEventsListener.onUpdate(eventType.participant)
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantJoin(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      OnMeetingRoomJoinedWithoutMicPermission -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onMeetingRoomJoinedWithoutMicPermission()
        }
      }

      OnMeetingRoomJoinedWithoutCameraPermission -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onMeetingRoomJoinedWithoutCameraPermission()
        }
      }

      is OnMeetingRoomJoinFailed -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onMeetingRoomJoinFailed(eventType.exception)
        }
      }

      is OnSelfAudioUpdate -> {
        val updateListeners = participantUpdateListeners[controllerContainer.selfController.getSelf().id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onAudioUpdate(controllerContainer.selfController.getSelf(), controllerContainer.selfController.getSelf().audioEnabled)
          }
        }
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onAudioUpdate(
            controllerContainer.selfController.getSelf().audioEnabled
          )
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnSelfVideoUpdate -> {
        val updateListeners = participantUpdateListeners[controllerContainer.selfController.getSelf().id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onVideoUpdate(controllerContainer.selfController.getSelf(), controllerContainer.selfController.getSelf().videoEnabled)
          }
        }
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onVideoUpdate(
            controllerContainer.selfController.getSelf().videoEnabled
          )
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerJoin -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantJoin(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerLeft -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantLeave(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerAudioUpdate -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onAudioUpdate(eventType.participant, eventType.participant.audioEnabled)
            updateListener.onUpdate(eventType.participant)
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onAudioUpdate(
            eventType.participant.audioEnabled,
            eventType.participant
          )
        }
      }

      is OnPeerVideoUpdate -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onVideoUpdate(eventType.participant, eventType.participant.videoEnabled)
            updateListener.onUpdate(eventType.participant)
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onVideoUpdate(
            eventType.participant.videoEnabled,
            eventType.participant
          )
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerScreenShareUpdate -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onScreenSharesUpdated()
        }
      }

      is OnPeerUpdate -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onUpdate(eventType.participant)
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerPageUpdate -> {
        ArrayList(participantEventsListeners).forEach { dyteParticipantEventsListener ->
          dyteParticipantEventsListener.onGridUpdated(
            eventType.gridInfo
          )
          dyteParticipantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnActiveParticipantsChanged -> {
        ArrayList(participantEventsListeners).forEach { dyteParticipantEventsListener ->
          dyteParticipantEventsListener.onActiveParticipantsChanged(eventType.activeParticipants)
          dyteParticipantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnActiveSpeakerChanged -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onActiveSpeakerChanged(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      OnNoActiveSpeaker -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onNoActiveSpeaker()
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerPinned -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantPinned(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      OnPeerUnpinned -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantUnpinned()
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      OnMeetingRecordingStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRecordingStarted()
        }
      }

      OnMeetingRecordingStopped -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRecordingEnded()
        }
      }

      is OnMeetingRecordingError -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRecordingStopError(eventType.error)
        }
      }

      is OnMeetingRecordingStateUpdate -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRecordingStateUpdated(eventType.state)
        }
      }

      is OnMeetingMessagesReceived -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onChatUpdates(false, null, eventType.messages)
        }
      }

      is OnNewMeetingMessageReceived -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onChatUpdates(
            false,
            eventType.message,
            controllerContainer.chatController.dyteChat.messages
          )
        }
      }

      is OnMeetingPollsReceived -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onPollUpdates(false, eventType.polls, null)
        }
      }

      is OnNewMeetingPollReceived -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onPollUpdates(
            false,
            controllerContainer.pollsController.polls,
            eventType.poll
          )
        }
      }

      OnCameraClosed -> {
      }

      OnCameraDisconnected -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          controllerContainer.platformUtilsProvider.getMediaSoupUtils().onCameraStreamKilled()
          controllerContainer.platformUtilsProvider.getPeerConnectionUtils().onCameraClosed()
          cameraEventsListener.onCameraDisconnected()
        }
      }

      is OnCameraError -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          controllerContainer.platformUtilsProvider.getMediaSoupUtils().onCameraStreamKilled()
          controllerContainer.platformUtilsProvider.getPeerConnectionUtils().onCameraClosed()
          cameraEventsListener.onCameraError(eventType.message)
        }
      }

      is OnCameraFreezed -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          cameraEventsListener.onCameraFreezed(eventType.message)
        }
      }

      is OnCameraOpening -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          cameraEventsListener.onCameraOpening(eventType.message)
        }
      }

      OnFirstFrameAvailable -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          cameraEventsListener.onFirstFrameAvailable()
        }
      }

      is OnMicrophoneInitError -> {
        ArrayList(microphoneEventsListeners).forEach { microphoneEventsListener ->
          microphoneEventsListener.onAudioTrackInitError(eventType.message)
        }
      }

      is OnMicrophoneStartError -> {
        ArrayList(microphoneEventsListeners).forEach { microphoneEventsListener ->
          microphoneEventsListener.onAudioTrackStartError(eventType.message)
        }
      }

      is OnMicrophoneError -> {
        ArrayList(microphoneEventsListeners).forEach { microphoneEventsListener ->
          microphoneEventsListener.onAudioTrackError(eventType.message)
        }
      }

      OnAudioDevicesChanged -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onAudioDevicesUpdated()
        }
      }

      is OnProximityChanged -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onProximityChanged(eventType.isNear)
        }
      }

      OnMeetingRoomDisconnected -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onMeetingRoomDisconnected()
        }
      }

      is OnPluginEnabled -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginActivated(eventType.plugin)
        }
      }

      is OnPluginDisabled -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginDeactivated(eventType.plugin)
        }
      }

      is OnPluginMessage -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginMessage(eventType.message)
        }
      }

      is DyteEventType.OnPluginFileRequest -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginFileRequest(eventType.plugin)
        }
      }

      OnHostKicked -> {
        ArrayList(roomEventsListeners).forEach { roomEventListener ->
          roomEventListener.onHostKicked()
        }
      }

      is OnWaitListPeerJoined -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onWaitListParticipantJoined(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnWaitListPeerAccepted -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onWaitListParticipantAccepted(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnWaitListPeerClosed -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onWaitListParticipantClosed(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnWaitListPeerRejected -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onWaitListParticipantRejected(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnSelfWaitListStatusUpdate -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onWaitListStatusUpdate(controllerContainer.selfController.getSelf().waitListStatus)
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
      }

      OnWebinarPresentRequest -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onWebinarPresentRequestReceived()
        }
      }

      OnWebinarStoppedPresenting -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onStoppedPresenting()
        }
      }
    }
  }

  override fun addSelfEventListener(selfEventListener: DyteSelfEventsListener) {
    this.selfEventsListeners.add(selfEventListener)
  }

  override fun removeSelfEventListener(selfEventListener: DyteSelfEventsListener) {
    if (this.selfEventsListeners.contains(selfEventListener)) {
      this.selfEventsListeners.remove(selfEventListener)
    }
  }

  override fun addRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener) {
    this.roomEventsListeners.add(roomEventsListener)
  }

  override fun removeRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener) {
    if (this.roomEventsListeners.contains(roomEventsListener)) {
      this.roomEventsListeners.remove(roomEventsListener)
    }
  }

  override fun addParticipantEventListener(participantEventsListener: DyteParticipantEventsListener) {
    this.participantEventsListeners.add(participantEventsListener)
  }

  override fun removeParticipantEventListener(participantEventsListener: DyteParticipantEventsListener) {
    if (this.participantEventsListeners.contains(participantEventsListener)) {
      this.participantEventsListeners.remove(participantEventsListener)
    }
  }

  override fun addCameraEventListener(cameraEventsListener: DyteCameraEventsListener) {
    this.cameraEventsListeners.add(cameraEventsListener)
  }

  override fun removeCameraEventListener(cameraEventsListener: DyteCameraEventsListener) {
    if (this.cameraEventsListeners.contains(cameraEventsListener)) {
      this.cameraEventsListeners.remove(cameraEventsListener)
    }
  }

  override fun addMicrophoneEventListener(microphoneEventsListener: DyteMicrophoneEventsListener) {
    this.microphoneEventsListeners.add(microphoneEventsListener)
  }

  override fun removeMicrophoneEventListener(microphoneEventsListener: DyteMicrophoneEventsListener) {
    if (this.microphoneEventsListeners.contains(microphoneEventsListener)) {
      this.microphoneEventsListeners.remove(microphoneEventsListener)
    }
  }

  override fun addPermissionEventListener(permissionEventsListener: DytePermissionEventsListener) {
    this.permissionEventsListeners.add(permissionEventsListener)
  }

  override fun removePermissionEventListener(permissionEventsListener: DytePermissionEventsListener) {
    if (this.permissionEventsListeners.contains(permissionEventsListener)) {
      this.permissionEventsListeners.remove(permissionEventsListener)
    }
  }

  override fun addPluginEventListener(pluginEventsListener: DytePluginEventsListener) {
    this.pluginEventsListeners.add(pluginEventsListener)
  }

  override fun removePluginEventListener(pluginEventsListener: DytePluginEventsListener) {
    if (this.pluginEventsListeners.contains(pluginEventsListener)) {
      this.pluginEventsListeners.remove(pluginEventsListener)
    }
  }

  override fun addParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener
  ) {
    val existingListeners: ArrayList<DyteParticipantUpdateListener>? = participantUpdateListeners[dyteMeetingParticipant.id]
    val updatedListeners = arrayListOf<DyteParticipantUpdateListener>()
    if (existingListeners == null) {
      updatedListeners.add(dyteParticipantUpdateListener)
    } else {
      updatedListeners.addAll(existingListeners)
      if (existingListeners.contains(dyteParticipantUpdateListener).not()) {
        updatedListeners.add(dyteParticipantUpdateListener)
      }
    }
    participantUpdateListeners[dyteMeetingParticipant.id] = updatedListeners
  }

  override fun removeParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener
  ) {
    val existingListeners = participantUpdateListeners[dyteMeetingParticipant.id]
    existingListeners?.remove(dyteParticipantUpdateListener)
  }

  override fun dispose() {
    this.selfEventsListeners.clear()
    this.roomEventsListeners.clear()
    this.participantEventsListeners.clear()
    this.cameraEventsListeners.clear()
    this.microphoneEventsListeners.clear()
    this.permissionEventsListeners.clear()
    this.pluginEventsListeners.clear()
    this.participantUpdateListeners.clear()
  }
}

interface IEventController {
  fun addSelfEventListener(selfEventListener: DyteSelfEventsListener)
  fun removeSelfEventListener(selfEventListener: DyteSelfEventsListener)

  fun addRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener)
  fun removeRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener)

  fun addParticipantEventListener(participantEventsListener: DyteParticipantEventsListener)
  fun removeParticipantEventListener(participantEventsListener: DyteParticipantEventsListener)

  fun addCameraEventListener(cameraEventsListener: DyteCameraEventsListener)
  fun removeCameraEventListener(cameraEventsListener: DyteCameraEventsListener)

  fun addMicrophoneEventListener(microphoneEventsListener: DyteMicrophoneEventsListener)
  fun removeMicrophoneEventListener(microphoneEventsListener: DyteMicrophoneEventsListener)

  fun addPermissionEventListener(permissionEventsListener: DytePermissionEventsListener)
  fun removePermissionEventListener(permissionEventsListener: DytePermissionEventsListener)

  fun addPluginEventListener(pluginEventsListener: DytePluginEventsListener)
  fun removePluginEventListener(pluginEventsListener: DytePluginEventsListener)

  fun addParticipantUpdateListener(dyteMeetingParticipant: DyteMeetingParticipant, dyteParticipantUpdateListener: DyteParticipantUpdateListener)
  fun removeParticipantUpdateListener(dyteMeetingParticipant: DyteMeetingParticipant, dyteParticipantUpdateListener: DyteParticipantUpdateListener)

  fun dispose()

  fun triggerEvent(eventType: DyteEventType)
}

sealed class DyteEventType {
  object OnMeetingRoomInitStarted : DyteEventType()
  object OnMeetingRoomInitCompleted : DyteEventType()
  class OnMeetingRoomInitFailed(val exception: Exception) : DyteEventType()

  object OnMeetingRoomLeave : DyteEventType()
  object OnMeetingRoomLeaveStarted : DyteEventType()
  object OnMeetingRoomDisconnected : DyteEventType()

  object OnMeetingRoomJoinStarted : DyteEventType()
  class OnMeetingRoomJoined(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnMeetingRoomJoinFailed(val exception: Exception) : DyteEventType()
  object OnMeetingRoomJoinedWithoutCameraPermission : DyteEventType()
  object OnMeetingRoomJoinedWithoutMicPermission : DyteEventType()

  class OnMeetingStateChanged(val state: String, val isReconnected: Boolean = false) :
    DyteEventType()

  // participant related event callbacks
  class OnPeerJoin(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerLeft(val participant: DyteMeetingParticipant) : DyteEventType()
  object OnPeerScreenShareUpdate : DyteEventType()
  class OnPeerAudioUpdate(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerVideoUpdate(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerPinned(val participant: DyteMeetingParticipant) : DyteEventType()
  object OnPeerUnpinned : DyteEventType()
  class OnPeerUpdate(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerPageUpdate(val gridInfo: GridInfo) : DyteEventType()
  class OnActiveParticipantsChanged(val activeParticipants: List<DyteMeetingParticipant>): DyteEventType()

  class OnWaitListPeerJoined(val participant: DyteMeetingParticipant): DyteEventType()
  class OnWaitListPeerAccepted(val participant: DyteMeetingParticipant): DyteEventType()
  class OnWaitListPeerRejected(val participant: DyteMeetingParticipant): DyteEventType()
  class OnWaitListPeerClosed(val participant: DyteMeetingParticipant): DyteEventType()

  class OnSelfWaitListStatusUpdate(val status: WaitListStatus): DyteEventType()
  object OnSelfAudioUpdate : DyteEventType()
  object OnSelfVideoUpdate : DyteEventType()

  class OnActiveSpeakerChanged(val participant: DyteMeetingParticipant) : DyteEventType()
  object OnNoActiveSpeaker : DyteEventType()

  object OnAudioDevicesChanged : DyteEventType()
  class OnProximityChanged(val isNear: Boolean) : DyteEventType()

  object OnMeetingRecordingStarted : DyteEventType()
  object OnMeetingRecordingStopped : DyteEventType()
  class OnMeetingRecordingError(val error: Exception) : DyteEventType()
  class OnMeetingRecordingStateUpdate(val state: DyteRecordingState): DyteEventType()

  class OnMeetingPollsReceived(val polls: List<DytePollMessage>) : DyteEventType()
  class OnNewMeetingPollReceived(val poll: DytePollMessage) : DyteEventType()

  class OnMeetingMessagesReceived(val messages: List<DyteChatMessage>) : DyteEventType()
  class OnNewMeetingMessageReceived(val message: DyteChatMessage) : DyteEventType()

  class OnPermissionsUpdated(val permissions: List<Permission>) : DyteEventType()

  class OnCameraError(val message: String?) : DyteEventType()
  object OnCameraDisconnected : DyteEventType()
  class OnCameraFreezed(val message: String?) : DyteEventType()
  class OnCameraOpening(val message: String?) : DyteEventType()
  object OnFirstFrameAvailable : DyteEventType()
  object OnCameraClosed : DyteEventType()

  class OnMicrophoneInitError(val message: String) : DyteEventType()
  class OnMicrophoneStartError(val message: String) : DyteEventType()
  class OnMicrophoneError(val message: String) : DyteEventType()

  object OnHostKicked: DyteEventType()

  class OnPluginEnabled(val plugin: DytePlugin): DyteEventType()
  class OnPluginDisabled(val plugin: DytePlugin): DyteEventType()
  class OnPluginMessage(val message: JsonObject): DyteEventType()
  class OnPluginFileRequest(val plugin: DytePlugin): DyteEventType()

  object OnWebinarPresentRequest: DyteEventType()
  object OnWebinarStoppedPresenting: DyteEventType()
}