package io.dyte.core.controllers

import io.dyte.core.models.DyteActionResult
import io.dyte.core.models.DyteChat
import io.dyte.core.models.DyteChatMessage
import io.dyte.core.models.DyteImageMessage
import io.dyte.core.models.DyteTextMessage
import io.dyte.core.network.PresignedUrlApiService
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_CHAT_MESSAGES
import io.dyte.core.socket.events.OutboundMeetingEventType.SEND_CHAT_MESSAGE
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessage
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessagesModel
import io.ktor.client.request.put
import io.ktor.client.request.setBody
import io.ktor.client.request.url
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class ChatController(
  controllerContainer: IControllerContainer
) : IChatController, BaseController(controllerContainer) {

  override val dyteChat = DyteChat(controllerContainer)

  override fun init() {
  }

  override fun loadChatMessages() {
    val chatMessagesResponse =
      controllerContainer.socketController.sendMessageSync(GET_CHAT_MESSAGES, null)
    if (chatMessagesResponse == "NO ACK") {
      return
    }
    val parsedResponse =
      controllerContainer.socketMessageResponseParser.parseResponse(chatMessagesResponse)
    val chatMessages = parsedResponse.payload as WebSocketChatMessagesModel
    val messages = arrayListOf<DyteChatMessage>()
    chatMessages.messages?.forEach { webSocketChatMessage ->
      val msg = addMessage(webSocketChatMessage)
      msg?.let {
        messages.add(msg)
      }
    }
    dyteChat.addMessages(messages)
    if (dyteChat.messages.isNotEmpty()) {
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnMeetingMessagesReceived(
          messages
        )
      )
    }
  }

  override fun handleChatMessages(dyteChatMessages: WebSocketChatMessagesModel) {
    dyteChatMessages.messages?.forEach {
      val dyteChatMessage = addMessage(it)
      if (dyteChatMessage != null) {
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnNewMeetingMessageReceived(
            dyteChatMessage
          )
        )
      }
    }
  }

  override fun handleNewChatMessage(dyteChatMessage: WebSocketChatMessage) {
    val message = addMessage(dyteChatMessage)
    if (message != null) {
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnNewMeetingMessageReceived(
          message
        )
      )
    }
  }

  private fun addMessage(chatMessage: WebSocketChatMessage): DyteChatMessage? {
    var dyteChatMessage: DyteChatMessage? = null
    when (chatMessage.type) {
      // text message
      0 -> {
        dyteChatMessage = DyteTextMessage(
          chatMessage.userId,
          chatMessage.displayName,
          false,
          null,
          requireNotNull(chatMessage.message),
          chatMessage.time
        )
        dyteChat.addMessage(dyteChatMessage)
      }

      1 -> {
        dyteChatMessage = DyteImageMessage(
          chatMessage.userId,
          chatMessage.displayName,
          false,
          null,
          chatMessage.link ?: "",
          null,
          null,
          chatMessage.time
        )
        dyteChat.addMessage(dyteChatMessage)
      }

      else -> {
        controllerContainer.loggerController.traceError("message type not supported $chatMessage")
      }
    }

    return dyteChatMessage
  }

  override fun sendMessage(message: String): DyteActionResult {
    if (!controllerContainer.presetController.canSendChatText()) {
      return DyteActionResult.ActionNotPermitted
    }

    val content = HashMap<String, JsonElement>()
    content["message"] = JsonPrimitive(message)
    content["userId"] = JsonPrimitive(controllerContainer.metaController.getPeerId())
    content["displayName"] =
      JsonPrimitive(controllerContainer.metaController.getDisplayName())
    content["type"] = JsonPrimitive(0)
    content["time"] = JsonPrimitive(
      controllerContainer.platformUtilsProvider.getPlatformUtils().getCurrentTime()
    )
    controllerContainer.socketController.sendMessageSync(SEND_CHAT_MESSAGE, JsonObject(content))
    return DyteActionResult.Success
  }

  // TODO : add type for image and fike
  override fun sendFileMessage(fileName: String, filePath: String): DyteActionResult {
    if (!controllerContainer.presetController.canSendChatFile()) {
      return DyteActionResult.ActionNotPermitted
    }

    val presignedUrl = runBlocking(Dispatchers.Default) {
      PresignedUrlApiService().getPresignedUrl(
        fileName,
        controllerContainer.metaController.getRoomName()
      )
    }
    val putLocation = presignedUrl?.data?.putLocation
    val getLocation = presignedUrl?.data?.getLocation

    putLocation?.let {
      // Upload file to putLocation
      runBlocking {
        val fileContent =
          controllerContainer.platformUtilsProvider.getPlatformUtils().getFileContent(filePath)

        controllerContainer.apiClient.getClient().put {
          url(putLocation)
          setBody(fileContent)
        }
      }

      val content = HashMap<String, JsonElement>()
      content["userId"] = JsonPrimitive(controllerContainer.metaController.getPeerId())
      content["displayName"] =
        JsonPrimitive(controllerContainer.metaController.getDisplayName())
      content["type"] = JsonPrimitive(1)
      content["link"] = JsonPrimitive(getLocation)
      content["time"] = JsonPrimitive(
        controllerContainer.platformUtilsProvider.getPlatformUtils().getCurrentTime()
      )
      controllerContainer.socketController.sendMessageSync(SEND_CHAT_MESSAGE, JsonObject(content))
    }
    return DyteActionResult.Success
  }
}

interface IChatController {
  val dyteChat: DyteChat

  fun loadChatMessages()
  fun handleChatMessages(dyteChatMessages: WebSocketChatMessagesModel)
  fun handleNewChatMessage(dyteChatMessage: WebSocketChatMessage)

  fun sendMessage(message: String): DyteActionResult
  fun sendFileMessage(fileName: String, filePath: String): DyteActionResult
}