package io.dyte.core.models

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.socket.events.OutboundMeetingEventType.KICK_PEER
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_PEER_AUDIO
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

/**
 * Dyte meeting participant
 *
 * @property id The ID of the participant pertaining to local user
 * @property userId The userId of the participant.
 * @property name Contains Name of the local user.
 * @property picture The participant's picture (if any).
 * @property isHost
 * @property clientSpecificId  Identifier provided by the developer while adding the participant.
 * @property flags
 * @property audioEnabled A boolean value indicating if the audio currently enabled.
 * @property videoEnabled A boolean value indicating if the video currently enabled.
 * @constructor Create empty Dyte meeting participant
 */
abstract class DyteMeetingParticipant(
  open val id: String,
  open val userId: String,
  open val name: String,
  open val picture: String?,
  open val isHost: Boolean,
  open val clientSpecificId: String?,
  open val flags: ParticipantFlags,
  open var audioEnabled: Boolean,
  open var videoEnabled: Boolean,
  internal val controllerContainer: IControllerContainer
) {
  /**
   * The video track for the local user.
   */
  var videoTrack: Any? = null

  /**
   * The audio track for the local user.
   */
  var audioTrack: Any? = null
  var screenShareTrack: Any? = null

  @Throws(UnsupportedOperationException::class)
  open fun disableVideo() {
    controllerContainer.hostController.muteVideo(this)
  }

  open fun toMap() : Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["id"] = id
    map["userId"] = userId
    map["name"] = name
    map["picture"] = picture
    map["isHost"] = isHost
    map["clientSpecificId"] = clientSpecificId
    map["flags"] = flags.toMap()
    map["audioEnabled"] = audioEnabled
    map["videoEnabled"] = videoEnabled
    return map
  }

  @Throws(UnsupportedOperationException::class)
  open fun disableAudio() {
    controllerContainer.hostController.muteAudio(this)
  }

  @Throws(UnsupportedOperationException::class)
  open fun kick() {
    controllerContainer.hostController.kick(id)
  }
}

data class ParticipantFlags(
  val recorder: Boolean,
  val hiddenParticipant: Boolean
){
  fun toMap() : Map<String, Any?>{
    val map = HashMap<String, Any?>()
    map["recorder"] = recorder
    map["hiddenParticipant"] = hiddenParticipant
    return map
  }
}