package io.dyte.core.controllers

import io.dyte.core.network.BaseApiService
import io.dyte.core.network.models.IceServerData
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.CREATE_WEB_RTC_TRANSPORT
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROUTER_RTP_CAPABILITIES
import io.dyte.core.socket.events.payloadmodel.inbound.Device
import io.dyte.core.socket.events.payloadmodel.outbound.CreateWebRTCTransportPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.JoinRoomPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.RouterCapabilitiesModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.encodeToJsonElement

internal class RoomNodeController(
  controllerContainer: IControllerContainer
) : IRoomNodeController, BaseController(controllerContainer) {

  private lateinit var iceServers: List<IceServerData>
  private lateinit var jsonParser: Json

  private lateinit var roomState: WebSocketRoomStateModel
  private lateinit var routerCapabilities: RouterCapabilitiesModel
  private lateinit var webRtcCreateTransportModelConsumer: WebRtcCreateTransportModel
  private lateinit var webRtcCreateTransportModelProducer: WebRtcCreateTransportModel
  private lateinit var joinRoom: WebSocketJoinRoomModel

  override fun init() {
    jsonParser = BaseApiService.json

    try {
      iceServers = runBlocking(Dispatchers.Default) { getICEServers() }
    } catch (e: Exception) {
      controllerContainer.loggerController.traceError("error in getting ice servers ${e.message}")
      e.printStackTrace()
      return
    }
  }

  override fun joinRoom() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinStarted)
    val roomStateResponse =
      controllerContainer.socketController.sendMessageSync(GET_ROOM_STATE, null)

    roomState =
      requireNotNull(
        controllerContainer.socketMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel
      )
    controllerContainer.metaController.setMeetingTitle(roomState.roomState?.displayTitle ?: "")

    controllerContainer.participantController.handleRoomState(roomState)

    // Pick out active plugins, if any
    controllerContainer.pluginsController.handleRoomState(roomState)

    val routerCapabilitiesResponse =
      controllerContainer.socketController.sendMessageSync(GET_ROUTER_RTP_CAPABILITIES, null)

    routerCapabilities =
      (controllerContainer.socketMessageResponseParser.parseResponse(routerCapabilitiesResponse).payload as RouterCapabilitiesModel)

    controllerContainer.platformUtilsProvider.getMediaSoupUtils()
      .loadRouterRtpCapabilities(jsonParser.encodeToString(routerCapabilities))

    val consumerResponse =
      controllerContainer.socketController.sendMessageSync(
        CREATE_WEB_RTC_TRANSPORT, jsonParser.encodeToJsonElement(
          CreateWebRTCTransportPayloadRequestModel(
            forceTcp = false,
            producing = false,
            consuming = true
          )
        )
      )

    webRtcCreateTransportModelConsumer =
      (controllerContainer.socketMessageResponseParser.parseResponse(consumerResponse).payload as WebRtcCreateTransportModel)
    controllerContainer.platformUtilsProvider.getMediaSoupUtils()
      .createWebRtcTransportRecv(webRtcCreateTransportModelConsumer, iceServers)

    val producerResponse =
      controllerContainer.socketController.sendMessageSync(
        CREATE_WEB_RTC_TRANSPORT, jsonParser.encodeToJsonElement(
          CreateWebRTCTransportPayloadRequestModel(
            forceTcp = false,
            producing = true,
            consuming = false
          )
        )
      )

    webRtcCreateTransportModelProducer =
      controllerContainer.socketMessageResponseParser.parseResponse(producerResponse).payload as WebRtcCreateTransportModel
    controllerContainer.platformUtilsProvider.getMediaSoupUtils()
      .createWebRtcTransportProd(webRtcCreateTransportModelProducer, iceServers)

    val joinRoomPayload = JoinRoomPayloadRequestModel()
    joinRoomPayload.device = Device(true)
    joinRoomPayload.displayName = controllerContainer.selfController.getSelf().name
    joinRoomPayload.rtpCapabilities = routerCapabilities
    val payload = jsonParser.encodeToJsonElement(joinRoomPayload)
    val joinRoomResponse =
      controllerContainer.socketController.sendMessageSync(
        OutboundMeetingEventType.JOIN_ROOM,
        payload
      )

    joinRoom =
      controllerContainer.socketMessageResponseParser.parseResponse(joinRoomResponse).payload as WebSocketJoinRoomModel

    controllerContainer.metaController.setMeetingStartedTimestamp(requireNotNull(joinRoom.startedAt))
    controllerContainer.metaController.setIsHost(joinRoom.isHost)

    if(controllerContainer.presetController.canPublishAudio() || controllerContainer.presetController.canPublishVideo()) {
      controllerContainer.platformUtilsProvider.getMediaSoupUtils()
        .connectTransport(requireNotNull(webRtcCreateTransportModelConsumer.id), true) {

        }
    }

    val selfTracks = controllerContainer.platformUtilsProvider.getMediaSoupUtils().getSelfTrack()
    val selfParticipant = controllerContainer.selfController.getSelf()
    selfParticipant.audioTrack = selfTracks.first
    selfParticipant.videoTrack = selfTracks.second
    controllerContainer.participantController.onSelfJoined()

    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnMeetingRoomJoined(
        controllerContainer.selfController.getSelf(),
        requireNotNull(joinRoom.startedAt)
      )
    )

    try {
      controllerContainer.pollsController.loadPolls()
    } catch (e:Exception) {
      e.printStackTrace()
    }

    controllerContainer.chatController.loadChatMessages()
  }

  private suspend fun getICEServers(): List<IceServerData> {
    return requireNotNull(controllerContainer.apiClient.getICEServers().iceServers)
  }
}

interface IRoomNodeController {
  fun joinRoom()
}