package io.dyte.core.controllers

import io.dyte.core.controllers.DyteRecordingState.IDLE
import io.dyte.core.controllers.DyteRecordingState.RECORDING
import io.dyte.core.controllers.DyteRecordingState.STARTING
import io.dyte.core.controllers.DyteRecordingState.STOPPING
import io.dyte.core.network.models.RecordingData
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlin.Exception

internal class RecordingController(
  controllerContainer: IControllerContainer
) : BaseController(controllerContainer), IRecordingController {
  private lateinit var recordingData: RecordingData
  private var recordingState: DyteRecordingState = IDLE

  override fun init() {
    try {
      recordingData = runBlocking(Dispatchers.Default) {
        controllerContainer.apiClient.getActiveRecording().data.recording
      }
      recordingState = RECORDING
      controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
    } catch (e:Exception) {
      e.printStackTrace()
    }
  }

  override suspend fun start() {
    if (!controllerContainer.presetController.canRecord()) {
      throw UnsupportedOperationException("not allowed to start recording")
    }

    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      runBlocking {
        try {
          recordingState = STARTING
          recordingData = controllerContainer.apiClient.startRecording().data.recording
          recordingState = RECORDING
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStarted)
        } catch (e:Exception) {
          e.printStackTrace()
          controllerContainer.loggerController.traceError("error in starting recording ${e.message}")
          recordingState = IDLE
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingError(e))
        }
      }
    }
  }

  override suspend fun stop() {
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnIoThread {
      runBlocking {
        try {
          recordingState = STOPPING
          controllerContainer.apiClient.stopRecording(recordingData)
          recordingState = IDLE
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingStopped)
        } catch (e:Exception) {
          e.printStackTrace()
          controllerContainer.loggerController.traceError("error in stopping recording ${e.message}")
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRecordingError(e))
        }
      }
    }
  }

  override fun getState(): DyteRecordingState {
    return recordingState
  }

  override fun fetchRecordingData() {
    recordingData = runBlocking(Dispatchers.Default) {
      controllerContainer.apiClient.getActiveRecording().data.recording
    }
  }
}

interface IRecordingController {
  suspend fun start()
  suspend fun stop()
  fun getState(): DyteRecordingState

  fun fetchRecordingData()
}

enum class DyteRecordingState {
  IDLE,
  STARTING,
  RECORDING,
  STOPPING;
}