package io.dyte.core.controllers

import io.dyte.core.models.DytePlugin
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginDisabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEnabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEvent
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.ktor.http.URLBuilder
import io.ktor.http.set
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.*

internal class PluginsController(
  controllerContainer: IControllerContainer
) : BaseController(controllerContainer), IPluginsController {

  private val allPlugins = mutableMapOf<String, DytePlugin>()
  private val activePlugins = mutableMapOf<String, DytePlugin>()

  override val plugins: List<DytePlugin>
    get() = allPlugins.values.toList()

  override fun onEnablePlugin(payload: WebSocketPluginEnabled) {
    controllerContainer.platformUtilsProvider.getLogger()
      .logD("PluginFeat", "pluginEnabled => $payload")
    if (payload.id == null || payload.enabledBy == null) return
    enablePluginForLocalUser(payload.id!!, payload.enabledBy!!)
    activePlugins[payload.id]?.let { plugin ->
      if (plugin.isActive) {
        controllerContainer.eventController.triggerEvent(DyteEventType.OnPluginEnabled(plugin))
      }
    }
  }

  override fun onDisablePlugin(payload: WebSocketPluginDisabled) {
    println("Dyte Plugin | onDisablePlugin => $payload")
    val plugin = activePlugins[payload.id] ?: return
    plugin.disableLocal()
    activePlugins.remove(plugin.id)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnPluginDisabled(plugin))
  }

  override fun onPluginSocketEvent(type: String, payload: WebSocketPluginEvent) {
    println("Dyte Plugin | onPluginSocketEvent => $type, $payload")
    val pluginId = payload.payload["id"]?.jsonPrimitive?.content ?: return
    val pluginPayload = buildJsonObject {
      put("type", type)
      put("payload", payload.payload)
    }
    submitEventToPlugin(pluginId, pluginPayload)
  }

  override fun onPluginWebViewMessage(pluginId: String, message: JsonObject) {
    println("Dyte Plugin | onPluginWebViewMessage => $message")

    val type = message["type"]?.jsonPrimitive?.content
    when (type) {
      "pluginEvent",
      "storePluginData",
      "getPluginData",
      "enablePluginForAll",
      "enablePluginForUser",
      "disablePluginForUser",
      "chatMessage" -> {
        val response = controllerContainer.socketController.sendMessageSyncGeneric(message)
        val jsonResponse = Json { isLenient = true }.parseToJsonElement(response)
        submitEventToPlugin(pluginId, jsonResponse.jsonObject)
      }
      "getJoinedPeers" -> {
        val roomStateResponse =
          controllerContainer.socketController.sendMessageSync(GET_ROOM_STATE, null)
        val roomState = requireNotNull(
          controllerContainer.socketMessageResponseParser.parseResponse(
            roomStateResponse
          ).payload as WebSocketRoomStateModel
        )
        val peers = roomState.roomState?.peers ?: emptyList()
        val pluginEvent = buildJsonObject {
          put("type", "websocket/get-joined-peers")
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("peers", Json { encodeDefaults = true }.encodeToJsonElement(peers))
          }
        }

        submitEventToPlugin(pluginId, pluginEvent)
      }
      "getPeerInfo" -> {
        val peerId = message["payload"]?.jsonObject?.get("peerId")?.jsonPrimitive?.content
          ?: controllerContainer.selfController.getSelf().id
        val socketPayload = buildJsonObject {
          put("type", type)
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("peerId", peerId)
          }
        }
        val response = controllerContainer.socketController.sendMessageSyncGeneric(socketPayload)
        val jsonResponse = Json { isLenient = true }.parseToJsonElement(response)
        submitEventToPlugin(pluginId, jsonResponse.jsonObject)
      }
      "getRoomState" -> {
        val roomStateResponse =
          controllerContainer.socketController.sendMessageSync(GET_ROOM_STATE, null)
        val roomState = (controllerContainer.socketMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState ?: error("Failed to get the room state")

        val peers = roomState.peers ?: emptyList()
        val enabledBy = activePlugins[pluginId]?.enabledBy ?: error("Plugin must be active")
        val enablerPeerId =
          peers.find { it.userId == enabledBy }?.id ?: error("Plugin enabler not found")

        val payload = buildJsonObject {
          put("type", "websocket/get-room-state")
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("roomName", roomState.roomName)
            put("pluginInitiatorPeerId", enablerPeerId)
            put("pluginInitiatorUserId", enabledBy)
            put("peers", Json { encodeDefaults = true }.encodeToJsonElement(peers))
            put("displayTitle", roomState.displayTitle)
            put("roomUUID", roomState.roomUUID)
            put("currentPeer", controllerContainer.selfController.getSelf().id)
          }
        }
        submitEventToPlugin(pluginId, payload)
      }
      "getPluginInitiatorId" -> {
        val roomStateResponse =
          controllerContainer.socketController.sendMessageSync(GET_ROOM_STATE, null)
        val roomState = (controllerContainer.socketMessageResponseParser.parseResponse(
          roomStateResponse
        ).payload as WebSocketRoomStateModel).roomState ?: error("Failed to get the room state")

        val peers = roomState.peers ?: emptyList()
        val enabledBy = activePlugins[pluginId]?.enabledBy ?: error("Plugin must be active")
        val enablerPeerId =
          peers.find { it.userId == enabledBy }?.id ?: error("Plugin enabler not found")

        val payload = buildJsonObject {
          put("type", "websocket/get-plugin-initiator-peer-id")
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("pluginInitiatorPeerId", enablerPeerId)
          }
        }
        submitEventToPlugin(pluginId, payload)
      }
      "getPluginInitiatorUserId" -> {
        val enabledBy = activePlugins[pluginId]?.enabledBy ?: error("Plugin must be active")

        val payload = buildJsonObject {
          put("type", "websocket/get-plugin-initiator-user-id")
          putJsonObject("payload") {
            message["payload"]?.let { payloadElement ->
              payloadElement.jsonObject.entries.forEach {
                put(it.key, it.value)
              }
            }
            put("pluginInitiatorUserId", enabledBy)
          }
        }
        submitEventToPlugin(pluginId, payload)
      }
      "pluginEventToParent" -> {}
      else -> {
        println("Dyte Plugin | onPluginWebViewMessage => Message not supported")
      }
    }
  }

  override fun activatePlugin(plugin: DytePlugin) {
    if (controllerContainer.presetController.permission.permissions?.plugins?.canStart != true) {
      throw UnsupportedOperationException("Not allowed to enable plugin")
    }
    runBlocking(Dispatchers.Default) {
      val activatePluginPayload = buildJsonObject {
        put("id", plugin.id)
        put("staggered", plugin.staggered)
      }
      controllerContainer.socketController.sendMessageSync(
        OutboundMeetingEventType.ADD_ROOM_PLUGIN,
        activatePluginPayload
      )
    }
  }

  override fun deactivatePlugin(plugin: DytePlugin) {
    if (controllerContainer.presetController.permission.permissions?.plugins?.canClose != true) {
      throw UnsupportedOperationException("Not allowed to disable the plugin")
    }
    runBlocking(Dispatchers.Default) {
      val deactivatePluginPayload = buildJsonObject {
        put("id", plugin.id)
      }
      controllerContainer.socketController.sendMessageSync(
        OutboundMeetingEventType.REMOVE_ROOM_PLUGIN,
        deactivatePluginPayload
      )
    }
  }

  override fun handleRoomState(roomState: WebSocketRoomStateModel) {
    println("Dyte Plugin | handleRoomState => ${roomState.roomState?.plugins}")
    roomState.roomState?.plugins?.forEach { roomPlugin ->
      val plugin = allPlugins[roomPlugin.id]
      plugin?.let {
        enablePluginForLocalUser(plugin.id, roomPlugin.enabledBy)
      }
    }
  }

  private fun enablePluginForLocalUser(pluginId: String, enabledBy: String) {
    val plugin = allPlugins[pluginId] ?: return
    runBlocking(Dispatchers.Default) {
      try {
        val pluginAuthToken = controllerContainer.apiClient.authorizePlugin(plugin.id).data.token
        val pluginWebView = controllerContainer.platformUtilsProvider.createWebView()
        val pluginUrl = URLBuilder(plugin.baseURL)
        pluginUrl.set {
          parameters["auth"] = pluginAuthToken
          parameters["pluginId"] = plugin.id
          parameters["backend"] = controllerContainer.metaController.getBaseUrl()
          encodedParameters["parent"] = "*"
        }
        pluginWebView.setURL(pluginUrl.buildString())
        plugin.enableLocal(
          pluginAuthToken,
          pluginWebView,
          enabledBy
        )
        activePlugins[plugin.id] = plugin
      } catch (e: Exception) {
        println("Dyte Plugin | enablePluginForLocalUser => plugin-exception: $e")
      }
    }
  }

  override fun init() {
    runBlocking(Dispatchers.Default) {
      val response = controllerContainer.apiClient.getPlugins()
      response.data.plugins.forEach {
        val plugin = DytePlugin(
          id = it.id,
          name = it.name,
          picture = it.picture,
          description = it.description,
          private = it.private,
          staggered = it.private,
          baseURL = it.baseURL,
          controllerContainer = controllerContainer
        )
        allPlugins[plugin.id] = plugin
      }
    }
  }

  private fun submitEventToPlugin(pluginId: String, payload: JsonObject) {
    controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
      activePlugins[pluginId]?.submitEvent(payload) ?: error("Plugin must be active")
    }
  }
}

interface IPluginsController {
  val plugins: List<DytePlugin>

  fun onEnablePlugin(payload: WebSocketPluginEnabled)

  fun onDisablePlugin(payload: WebSocketPluginDisabled)

  fun onPluginSocketEvent(type: String, payload: WebSocketPluginEvent)

  fun onPluginWebViewMessage(pluginId: String, message: JsonObject)

  fun activatePlugin(plugin: DytePlugin)

  fun deactivatePlugin(plugin: DytePlugin)

  fun handleRoomState(roomState: WebSocketRoomStateModel)
}
