package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnPeerPageUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerLeft
import io.dyte.core.controllers.DyteEventType.OnPeerPinned
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerUnpinned
import io.dyte.core.controllers.DyteEventType.OnPeerVideoUpdate
import io.dyte.core.controllers.PageViewMode.GRID
import io.dyte.core.controllers.PageViewMode.PAGINATED
import io.dyte.core.models.DyteMeetingParticipant
import io.dyte.core.models.DyteParticipant
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.ParticipantFlags
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketGetPageModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerPinnedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerConnectModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAccepted
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAdded
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerClosed
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerRejected
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class ParticipantController(
  controllerContainer: IControllerContainer
) : IParticipantController, BaseController(controllerContainer) {

  private val waitlistedParticipants = arrayListOf<DyteMeetingParticipant>()
  private val joinedParticipants = arrayListOf<DyteMeetingParticipant>()
  private val activeParticipants = arrayListOf<DyteMeetingParticipant>()
  private var pinnedParticipant: DyteMeetingParticipant? = null
  private val screenshareParticipants = arrayListOf<DyteMeetingParticipant>()

  private val hiddenParticipants = arrayListOf<DyteMeetingParticipant>()

  override val meetingRoomParticipants = DyteRoomParticipants(
    waitlistedParticipants,
    joinedParticipants,
    activeParticipants,
    screenshareParticipants,
    controllerContainer
  )

  private var pageNumber: Int = 0
  private var maxVideoCount = 6
  private var viewMode: PageViewMode = GRID

  private var isLoadingPage = false

  override fun getPageNumber(): Int {
    return pageNumber
  }

  override fun getViewMode(): PageViewMode {
    return viewMode
  }

  override fun setViewMode(pageViewMode: PageViewMode) {
    viewMode = pageViewMode
    when (viewMode) {
      GRID -> {
        setPage(0)
      }
      PAGINATED -> {
        setPage(1)
      }
    }
  }

  override fun init() {
    maxVideoCount =
      controllerContainer.presetController.getMaxVideoCount()
  }

  /**
   * On peer joined
   *
   * when a remote peer joins, this does not contain audio track or video track
   *
   * @see onParticipantStreamConnected
   *
   * @param meetingPeerUser
   */
  override fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser) {
    val participant = getDyteParticipant(meetingPeerUser)
    if (participant.flags.hiddenParticipant || participant.flags.recorder) {
      hiddenParticipants.add(participant)
    } else {
      if (controllerContainer.presetController.permission.permissions?.showParticipantList == false) {
        throw UnsupportedOperationException("not allowed to view participant list")
      }
      joinedParticipants.add(participant)
      controllerContainer.eventController.triggerEvent(DyteEventType.OnPeerJoin(participant))
    }
  }


  override fun muteAll() {
  }

  override fun onPeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel) {
    var isHiddentParticipantLeft = false
    var peerToRemove = joinedParticipants.find { it.id == webSocketPeerLeftModel.peerId }
    if (peerToRemove == null) {
      peerToRemove = hiddenParticipants.find { it.id == webSocketPeerLeftModel.peerId }
      hiddenParticipants.remove(peerToRemove)
      isHiddentParticipantLeft = true
    }

    if (!isHiddentParticipantLeft) {
      peerToRemove?.let {
        val screenSharePeer = screenshareParticipants.find { it.id == peerToRemove.id }
        if (screenSharePeer != null) {
          screenshareParticipants.remove(screenSharePeer)
          controllerContainer.eventController.triggerEvent(OnPeerScreenShareUpdate)
        }

        joinedParticipants.remove(peerToRemove)
        controllerContainer.eventController.triggerEvent(OnPeerLeft(peerToRemove))
      }
    }
  }

  override fun onPeerAudioMuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
    participant?.let {
      participant.audioEnabled = false
      controllerContainer.eventController.triggerEvent(OnPeerAudioUpdate(participant))
    }
  }

  override fun onPeerAudioUnmuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
    participant?.let {
      participant.audioEnabled = true
      controllerContainer.eventController.triggerEvent(OnPeerAudioUpdate(participant))
    }
  }

  override fun onPeerPinned(webSocketPeerPinnedModel: WebSocketPeerPinnedModel) {
    val participant = joinedParticipants.find { it.id == webSocketPeerPinnedModel.peerId }
    participant?.let {
      pinnedParticipant = participant
      controllerContainer.eventController.triggerEvent(OnPeerPinned(participant))
    }
  }

  override fun onPeerUnpinned() {
    pinnedParticipant = null
    controllerContainer.eventController.triggerEvent(OnPeerUnpinned)
  }

  override fun onPeerVideoUnMuted(webSocketProducerConnectModel: WebSocketProducerConnectModel) {
  }

  override fun onPeerVideoMuted(webSocketProducerClosedModel: WebSocketProducerClosedModel) {
  }

  override fun onWaitlistPeerAdded(webSocketWaitlistPeerAdded: WebSocketWaitlistPeerAdded) {
  }

  override fun onWaitlistPeerAccepted(webSocketWaitlistPeerAccepted: WebSocketWaitlistPeerAccepted) {
  }

  override fun onWaitlistPeerRejected(webSocketWaitlistPeerRejected: WebSocketWaitlistPeerRejected) {
  }

  override fun onWaitlistPeerClosed(webSocketWaitlistPeerClosed: WebSocketWaitlistPeerClosed) {
  }

  override fun onParticipantVideoMuted(webSocketConsumerClosedModel: WebSocketConsumerClosedModel) {
    val consumerType =
      controllerContainer.platformUtilsProvider.getMediaSoupUtils().getConsumerType(
        requireNotNull(webSocketConsumerClosedModel.id)
      )
    if("audio" == consumerType) {
      return
    }
    webSocketConsumerClosedModel.id?.let {
      val appdata =
        controllerContainer.platformUtilsProvider.getMediaSoupUtils().getAppDataFromConsumerId(it)
      val peerId = appdata?.peerId
      peerId?.let {
        val participant = joinedParticipants.find { it.id == peerId }
        participant?.let {
          if (appdata.screenShare == false) {
            participant.videoEnabled = false
            controllerContainer.eventController.triggerEvent(OnPeerVideoUpdate(participant))
          }
        }
      }
    }
  }

  override fun onParticipantVideoUnmuted(webSocketConsumerClosedModel: WebSocketConsumerResumedModel) {
    webSocketConsumerClosedModel.id?.let {
      val appdata =
        controllerContainer.platformUtilsProvider.getMediaSoupUtils().getAppDataFromConsumerId(it)
      val peerId = appdata?.peerId
      peerId?.let {
        val participant = joinedParticipants.find { it.id == peerId }
        participant?.let {
          participant.videoEnabled = true
          controllerContainer.eventController.triggerEvent(OnPeerVideoUpdate(participant))
        }
      }
      controllerContainer.platformUtilsProvider.getMediaSoupUtils().resumeConsumer(it)
    }
  }

  override fun onParticipantStreamConnected(
    participant: DyteMeetingParticipant
  ) {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnPeerUpdate(participant))
  }

  private fun getDyteParticipant(meetingPeerUser: WebSocketMeetingPeerUser): DyteMeetingParticipant {
    val flags = ParticipantFlags(
      meetingPeerUser.flags?.recordere ?: false,
      meetingPeerUser.flags?.hiddenParticipant ?: false
    )
    return DyteParticipant(
      requireNotNull(meetingPeerUser.id),
      requireNotNull(meetingPeerUser.userId),
      requireNotNull(meetingPeerUser.displayName),
      meetingPeerUser.picture,
      meetingPeerUser.isHost ?: false,
      meetingPeerUser.clientSpecificId
        ?: controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid(),
      flags,
      meetingPeerUser.audioMuted?.not() ?: false,
      false,
      controllerContainer
    )
  }

  override fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel) {
    if (controllerContainer.presetController.permission.permissions?.showParticipantList == false) {
      throw UnsupportedOperationException("not allowed to view participant list")
    }
    webSocketRoomStateModel.roomState?.peers?.forEach {
      onPeerJoined(it)
    }
  }

  override fun onSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel) {
    if (viewMode == GRID) {
      activeParticipants.clear()
      activeParticipants.add(controllerContainer.selfController.getSelf())
      if(webSocketSelectedPeersModel.peerIds?.isNotEmpty() == true) {
        webSocketSelectedPeersModel.peerIds?.forEach { peerId ->
          val peer = joinedParticipants.find { it.id == peerId }
          peer?.let {
            activeParticipants.add(peer)
          }
        }
        controllerContainer.eventController.triggerEvent(OnPeerPageUpdate(false, false))
      }
    }
  }

  override fun onPageLoaded(webSocketGetPageModel: WebSocketGetPageModel, newPageNumber: Int) {
    if (webSocketGetPageModel.peerIds == null) {
      isLoadingPage = false
      return
    }
    activeParticipants.clear()
    webSocketGetPageModel.peerIds?.forEach { peerId ->
      val peer = joinedParticipants.find { it.id == peerId }
      peer?.let {
        activeParticipants.add(peer)
      }
    }
    isLoadingPage = false
    val isNextPagePossible = joinedParticipants.size > newPageNumber * getMaxVideoCountPerPage()
    val isPreviousPagePossible = newPageNumber > 1
    pageNumber = newPageNumber
    controllerContainer.eventController.triggerEvent(
      OnPeerPageUpdate(
        isNextPagePossible,
        isPreviousPagePossible
      )
    )
  }

  override fun shouldShowPaginator(): Boolean {
    return joinedParticipants.size > maxVideoCount
  }

  override fun getMaxVideoCountPerPage(): Int {
    return maxVideoCount
  }

  override fun onPeerScreenShareStarted(participant: DyteMeetingParticipant) {
    screenshareParticipants.add(participant)
    controllerContainer.eventController.triggerEvent(OnPeerScreenShareUpdate)
  }

  override fun onPeerScreenSharedEnded(participant: DyteMeetingParticipant) {
    screenshareParticipants.remove(participant)
    controllerContainer.eventController.triggerEvent(OnPeerScreenShareUpdate)
  }

  override fun setPage(newPageNumber: Int) {
    if (newPageNumber < 0) {
      throw IllegalArgumentException("page number cant be less than 0")
    }
    isLoadingPage = true
    val content = HashMap<String, JsonElement>()
    content["pageNum"] = JsonPrimitive(newPageNumber)
    val prevPageResponse =
      controllerContainer.socketController.sendMessageSync(
        OutboundMeetingEventType.GET_PAGE,
        JsonObject(content)
      )
    val pageModel =
      controllerContainer.socketMessageResponseParser.parseResponse(prevPageResponse).payload as WebSocketGetPageModel
    onPageLoaded(pageModel, newPageNumber)
  }

  override fun onSelfJoined() {
    joinedParticipants.add(controllerContainer.selfController.getSelf())
    // controllerContainer.eventController.triggerEvent(OnMeetingPageUpdate(false, false))
  }
}

@OptIn(ExperimentalSerializationApi::class, DelicateCoroutinesApi::class)
interface IParticipantController {
  val meetingRoomParticipants: DyteRoomParticipants

  fun getPageNumber(): Int
  fun setViewMode(pageViewMode: PageViewMode)
  fun getViewMode(): PageViewMode

  fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel)

  fun onPeerJoined(meetingPeerUser: WebSocketMeetingPeerUser)
  fun onPeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel)

  fun onPeerAudioMuted(webSocketPeerMuteModel: WebSocketPeerMuteModel)
  fun onPeerAudioUnmuted(webSocketPeerMuteModel: WebSocketPeerMuteModel)

  fun onPeerPinned(webSocketPeerPinnedModel: WebSocketPeerPinnedModel)
  fun onPeerUnpinned()

  fun onPeerScreenShareStarted(participant: DyteMeetingParticipant)
  fun onPeerScreenSharedEnded(participant: DyteMeetingParticipant)

  fun onParticipantStreamConnected(participant: DyteMeetingParticipant)

  fun onPeerVideoUnMuted(webSocketProducerConnectModel: WebSocketProducerConnectModel)
  fun onPeerVideoMuted(webSocketProducerClosedModel: WebSocketProducerClosedModel)

  fun onParticipantVideoMuted(webSocketConsumerClosedModel: WebSocketConsumerClosedModel)
  fun onParticipantVideoUnmuted(webSocketConsumerClosedModel: WebSocketConsumerResumedModel)

  fun onSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel)
  fun onPageLoaded(webSocketGetPageModel: WebSocketGetPageModel, newPageNumber: Int)

  fun onWaitlistPeerAdded(webSocketWaitlistPeerAdded: WebSocketWaitlistPeerAdded)
  fun onWaitlistPeerAccepted(webSocketWaitlistPeerAccepted: WebSocketWaitlistPeerAccepted)
  fun onWaitlistPeerRejected(webSocketWaitlistPeerRejected: WebSocketWaitlistPeerRejected)
  fun onWaitlistPeerClosed(webSocketWaitlistPeerClosed: WebSocketWaitlistPeerClosed)

  fun muteAll()
  fun shouldShowPaginator(): Boolean
  fun getMaxVideoCountPerPage(): Int

  fun setPage(newPageNumber: Int)

  fun onSelfJoined()
}

enum class PageViewMode {
  GRID,
  PAGINATED;
}