package io.dyte.core.media

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.TextView
import androidx.cardview.widget.CardView
import io.dyte.core.R
import io.dyte.core.Utils
import org.webrtc.EglBase
import org.webrtc.SurfaceViewRenderer
import org.webrtc.VideoTrack

class SelfView : FrameLayout {
  private lateinit var videoRenderer: SurfaceViewRenderer
  private lateinit var ivCamera: ImageView
  private lateinit var ivMic: ImageView
  private lateinit var tvName: TextView
  private lateinit var tvInitials: TextView
  private lateinit var cvVideoContainer: CardView

  private var isAudioEnabled = true
  private var isVideoEnabled = true

  private var videoTrack: VideoTrack? = null
  private var isEglContextAttached = false

  constructor(context: Context) : super(context) {
    init(context)
  }

  constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
    init(context)
  }

  constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
    context,
    attrs,
    defStyleAttr
  ) {
    init(context)
  }

  private fun init(context: Context) {
    val view = LayoutInflater.from(context).inflate(R.layout.view_self, this, false)
    addView(view)

    videoRenderer = view.findViewById(R.id.video_renderer)

    ivMic = view.findViewById(R.id.ivMic)
    ivCamera = view.findViewById(R.id.ivVideo)
    tvName = view.findViewById(R.id.tvName)
    tvInitials = view.findViewById(R.id.tvSelfInitials)
    cvVideoContainer = view.findViewById(R.id.cvVideoContainer)
  }

  fun setListener(
    onAudioToggle: (isEnabled: Boolean) -> Unit,
    onVideoToggle: (isEnabled: Boolean) -> Unit
  ) {
    ivMic.setOnClickListener {
      isAudioEnabled = !isAudioEnabled
      onAudioToggle.invoke(isAudioEnabled)
      if (isAudioEnabled) {
        ivMic.setImageResource(R.drawable.ic_baseline_mic_24)
      } else {
        ivMic.setImageResource(R.drawable.ic_baseline_mic_off_24)
      }
    }

    ivCamera.setOnClickListener {
      isVideoEnabled = !isVideoEnabled
      onVideoToggle.invoke(isVideoEnabled)
      if (isVideoEnabled) {
        ivCamera.setImageResource(R.drawable.ic_baseline_videocam_24)
        tvInitials.visibility = View.GONE
        videoRenderer.visibility = View.VISIBLE
      } else {
        ivCamera.setImageResource(R.drawable.ic_baseline_videocam_off_24)
        tvInitials.visibility = View.VISIBLE
        videoRenderer.visibility = View.GONE
      }
      videoTrack?.setEnabled(isVideoEnabled)
    }
  }

  fun render(videoTrack: VideoTrack?, eglContext: EglBase.Context) {
    this.videoTrack = videoTrack
    if(!isEglContextAttached) {
      videoRenderer.init(eglContext, null)
      isEglContextAttached = true
    }
    videoTrack?.let {
      tvInitials.visibility = View.GONE
      this.videoTrack?.setEnabled(true)
      this.videoTrack?.addSink(videoRenderer)
      videoRenderer.visibility = VISIBLE
    } ?: run {
      videoRenderer.visibility = GONE
      tvInitials.visibility = View.VISIBLE
    }

    cvVideoContainer.visibility = VISIBLE
  }

  fun refreshName(name: String) {
    tvInitials.text = Utils.getInitialsFromName(name)
    if (name.isNotEmpty()) {
      tvName.visibility = View.VISIBLE
      tvName.text = name
    } else {
      tvName.visibility = View.GONE
    }
  }

  fun release() {
    isEglContextAttached = false
    videoRenderer.release()
    videoTrack?.dispose()
  }
}