package io.dyte.core.models

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.network.models.PluginConfigResponse
import io.dyte.core.platform.DytePluginFile
import io.dyte.core.platform.IDyteWebView
import io.dyte.core.platform.IDyteWebViewListener
import io.dyte.core.platform.WebView
import kotlinx.serialization.json.JsonObject

/**
 * Dyte Plugin
 *
 * @property controllerContainer
 * @constructor Create un-activated Dyte Plugin
 */
class DytePlugin(
  val id: String,
  val name: String,
  val description: String,
  val picture: String,
  val isPrivate: Boolean,
  val staggered: Boolean,
  val baseURL: String,
  private val controllerContainer: IControllerContainer
) {

  var enabledBy: String? = null

  companion object {
    private const val KEY_DEFAULT_WEBVIEW = "default"
  }

  var config: PluginConfigResponse? = null

  private var authToken: String? = null
  private  var urlToLoad: String? = null
  private val webViews: MutableMap<String, IDyteWebView> = mutableMapOf()

  private var _isActive: Boolean = false
  val isActive: Boolean
    get() = _isActive

  private val webViewListener: IDyteWebViewListener = object : IDyteWebViewListener {
    override fun onWebViewMessage(message: JsonObject) {
      controllerContainer.pluginsController.onPluginWebViewMessage(id, message)
    }

    override fun onFileRequest() {
      controllerContainer.pluginsController.onPluginFileRequest(this@DytePlugin)
    }
  }

  /**
   * Submits the event to the Plugin SDK through WebView.
   *
   * @param payload event JSON
   */
  internal fun submitEvent(payload: JsonObject) {
    webViews[KEY_DEFAULT_WEBVIEW]?.submitEvent(payload)
  }

  /**
   * Activate this plugin for all participants.
   */
  fun activate(): DyteActionResult {
    if (isActive.not()) {
      return controllerContainer.pluginsController.activatePlugin(this)
    }
    return DyteActionResult.Success
  }

  /**
   * Deactivate this plugin for all participants.
   */
  fun deactivate(): DyteActionResult {
    if (isActive) {
      return controllerContainer.pluginsController.deactivatePlugin(this)
    }
    return DyteActionResult.Success
  }

  fun getPluginView(): WebView {
    return webViews.getOrElse(KEY_DEFAULT_WEBVIEW) {
      val webView =  controllerContainer.platformUtilsProvider.createWebView()
      webView.initializeWebView()
      webView.setListener(webViewListener)
      urlToLoad?.let {
        webView.loadURL(it)
      }
      webViews[KEY_DEFAULT_WEBVIEW] = webView
      webViews.getValue(KEY_DEFAULT_WEBVIEW)
    }.getWebView()
  }

  private fun removePluginView() {
    webViews.remove(KEY_DEFAULT_WEBVIEW)
  }

  /**
   * Uploads selected file to Plugin.
   */
  fun uploadFile(file: DytePluginFile) {
    webViews[KEY_DEFAULT_WEBVIEW]?.uploadFile(file)
  }

  /**
   * Enable this plugin for the local user.
   */
  internal fun enableLocal(
    authToken: String,
    url: String,
    enabledBy: String
  ) {
    if (isActive) return

    this.authToken = authToken
    this.enabledBy = enabledBy
    this.urlToLoad = url
    _isActive = true
  }

  /**
   * Disable this plugin for the local user.
   */
  internal fun disableLocal() {
    if (!isActive) return

    webViews[KEY_DEFAULT_WEBVIEW]?.removeListener()
    removePluginView()
    _isActive = false
  }
}