package io.dyte.core.controllers

import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.listeners.DyteCameraEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.models.DyteChat
import io.dyte.core.models.DyteMeetingInfo
import io.dyte.core.models.DyteMeetingInfoV2
import io.dyte.core.models.DyteMeta
import io.dyte.core.models.DytePlugins
import io.dyte.core.models.DytePoll
import io.dyte.core.models.DyteRecording
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.DyteWebinar
import io.dyte.core.network.ApiClient
import io.dyte.core.network.IApiClient
import io.dyte.core.observability.LoggerController
import io.dyte.core.observability.ILoggerController
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.SocketMessageResponseParser
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext

internal class Controller(private val _platformUtilsProvider: IDytePlatformUtilsProvider) :
  IController, IControllerContainer {

  override val platformUtilsProvider: IDytePlatformUtilsProvider
    get() = _platformUtilsProvider

  private lateinit var _chatController: ChatController
  override val chatController: IChatController
    get() = _chatController

  private lateinit var _participantController: ParticipantController
  override val participantController: IParticipantController
    get() = this._participantController

  private lateinit var _pollController: PollController
  override val pollsController: IPollController
    get() = _pollController

  private lateinit var _metaController: MetaController
  override val metaController: IMetaController
    get() = _metaController

  private lateinit var _selfController: SelfController
  override val selfController: ISelfController
    get() = this._selfController

  private lateinit var _roomNodeController: RoomNodeController
  override val roomNodeController: IRoomNodeController
    get() = this._roomNodeController

  private lateinit var _presetController: PresetController
  override val presetController: IPresetController
    get() = this._presetController

  private lateinit var _hostController: HostController
  override val hostController: IHostController
    get() = this._hostController

  private lateinit var _apiClient: ApiClient
  override val apiClient: IApiClient
    get() = this._apiClient

  private lateinit var _socketController: SocketController
  override val socketController: ISocketController
    get() = this._socketController

  private var _eventController = EventController(this)
  override val eventController: IEventController
    get() = this._eventController

  private var _connectionController = ConnectionController(this)
  override val connectionController: IConnectionController
    get() = this._connectionController

  private lateinit var _socketMessageResponseParser: SocketMessageResponseParser
  override val socketMessageResponseParser: ISocketMessageResponseParser
    get() = this._socketMessageResponseParser

  private lateinit var _recordingController: RecordingController
  override val recordingController: IRecordingController
    get() = this._recordingController

  private lateinit var _permissionController: PermissionController
  override val permissionController: IPermissionController
    get() = this._permissionController

  private lateinit var _pluginsController: PluginsController
  override val pluginsController: IPluginsController
    get() = this._pluginsController

  private lateinit var _loggerController: LoggerController
  override val loggerController: ILoggerController
    get() = this._loggerController

  private lateinit var _mediaSoupController: MediaSoupController
  override val mediaSoupController: IMediaSoupController
    get() = this._mediaSoupController

  private lateinit var _webinarController: WebinarController
  override val webinarController: IWebinarController
    get() = this._webinarController

  private lateinit var recording: DyteRecording
  private lateinit var poll: DytePoll
  private lateinit var meta: DyteMeta
  private lateinit var plugins: DytePlugins
  private lateinit var webinar: DyteWebinar

  init {
    this._eventController.init()
  }

  override fun init(dyteMeetingInfo: DyteMeetingInfo) {
    _init(MeetingInfo(dyteMeetingInfo.authToken, MeetingConfig(dyteMeetingInfo.enableAudio, dyteMeetingInfo.enableVideo)).apply {
      roomName = dyteMeetingInfo.roomName
    })
  }

  override fun init(dyteMeetingInfoV2: DyteMeetingInfoV2) {
    _init(MeetingInfo(dyteMeetingInfoV2.authToken, MeetingConfig(dyteMeetingInfoV2.enableAudio, dyteMeetingInfoV2.enableVideo)))
  }

  fun _init(meetingInfo: MeetingInfo) {
    try {
      _platformUtilsProvider.getPlatformUtils().listenForCrashes()
      this.eventController.triggerEvent(DyteEventType.OnMeetingRoomInitStarted)
      _platformUtilsProvider.init(this@Controller)
      _metaController = MetaController(meetingInfo, this@Controller)

      try {
        _metaController.init()
      } catch (e: Exception) {
        loggerController.traceWarning("Invalid auth token")
        eventController.triggerEvent(
          DyteEventType.OnMeetingRoomInitFailed(
            IllegalArgumentException(
              "Invalid auth token"
            )
          )
        )
        return
      }

      _permissionController = PermissionController(this)
      _permissionController.init()

      _loggerController = LoggerController(this@Controller)
      _loggerController.init()

      val permissionsToAsk = arrayListOf<PermissionType>()
      if (meetingInfo.config.enableVideo && _permissionController.isPermissionGrated(
          CAMERA
        ).not()
      ) {
        permissionsToAsk.add(CAMERA)
      }
      if (meetingInfo.config.enableAudio && _permissionController.isPermissionGrated(
          MICROPHONE
        ).not()
      ) {
        permissionsToAsk.add(MICROPHONE)
      }

      _permissionController.askPermissions(permissionsToAsk,
        {
          loggerController.traceLog("permissions granted")
          doInit(meetingInfo)
        },
        {
          loggerController.traceLog("permissions not granted")
          doInit(meetingInfo)
        })
    } catch (e: Exception) {
      eventController.triggerEvent(DyteEventType.OnMeetingRoomInitFailed(e))
    }
  }

  private fun doInit(meetingInfo: MeetingInfo) {
    _platformUtilsProvider.getPlatformUtils().runOnIoThread {
      runBlocking {
        try {
          _platformUtilsProvider.getLogger().enableLogger(true)

          _connectionController = ConnectionController(this@Controller)
          _apiClient = ApiClient(this@Controller)

          val participantInfo = runBlocking(Dispatchers.Default) {
            apiClient.getParticipantInfo()
          }

          loggerController.setOrgId(participantInfo.organizationId)
          _metaController.setParticipantInfo(participantInfo)
          _mediaSoupController = MediaSoupController(this@Controller)
          _presetController = PresetController(participantInfo, this@Controller)
          _socketController = SocketController(this@Controller)
          _participantController = ParticipantController(this@Controller)
          _selfController = SelfController(participantInfo, this@Controller)
          _chatController = ChatController(this@Controller)
          _pollController = PollController(this@Controller)
          _hostController = HostController(this@Controller)
          _recordingController = RecordingController(this@Controller)
          _pluginsController = PluginsController(this@Controller)
          _webinarController = WebinarController(this@Controller)
          _pluginsController.init()

          recording = DyteRecording(this@Controller)
          poll = DytePoll(this@Controller)
          meta = DyteMeta(this@Controller)
          plugins = DytePlugins(this@Controller)
          webinar = DyteWebinar(this@Controller)

          _mediaSoupController.init()
          _permissionController.init()
          _connectionController.init()
          _socketController.init()
          _presetController.init()
          _participantController.init()
          _selfController.init()
          _chatController.init()
          _pollController.init()
          _hostController.init()
          _webinarController.init()
          _recordingController.init()

          withContext(Dispatchers.Main) {
            platformUtilsProvider.getMediaSoupUtils().init(this@Controller)
            platformUtilsProvider.getMediaUtils().init()
          }

          _socketMessageResponseParser = SocketMessageResponseParser()
          _roomNodeController = RoomNodeController(this@Controller)
          _roomNodeController.init()

          socketController.connect()
        } catch (e: Exception) {
          eventController.triggerEvent(DyteEventType.OnMeetingRoomInitFailed(e))
          loggerController.traceError("do init crashed ${e.message}")
        }
      }
    }
  }

  override fun joinRoom() {
    platformUtilsProvider.getPlatformUtils().runOnIoThread {
      permissionController.processPermissionForSelf()
      roomNodeController.joinRoom()
    }
  }

  override fun leaveRoom() {
    eventController.triggerEvent(DyteEventType.OnMeetingRoomLeaveStarted)
    platformUtilsProvider.getMediaSoupUtils().leaveCall()
    socketController.disconnect()
    eventController.triggerEvent(DyteEventType.OnMeetingRoomLeave)
  }

  override fun getSelf(): DyteSelfParticipant {
    return this.selfController.getSelf()
  }

  override fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) {
    this.eventController.addRoomEventListener(meetingRoomEventsListener)
  }

  override fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) {
    this.eventController.removeRoomEventListener(meetingRoomEventsListener)
  }

  override fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    this.eventController.addSelfEventListener(selfEventsListener)
  }

  override fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    this.eventController.removeSelfEventListener(selfEventsListener)
  }

  override fun addParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener) {
    this.eventController.addParticipantEventListener(participantEventsListener)
  }

  override fun removeParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener) {
    this.eventController.removeParticipantEventListener(participantEventsListener)
  }

  override fun addCameraEventsListener(cameraEventsListener: DyteCameraEventsListener) {
    this.eventController.addCameraEventListener(cameraEventsListener)
  }

  override fun removeCameraEventsListener(cameraEventsListener: DyteCameraEventsListener) {
    this.eventController.removeCameraEventListener(cameraEventsListener)
  }

  override fun getMeetingParticipant(): DyteRoomParticipants {
    return this.participantController.meetingRoomParticipants
  }

  override fun getChat(): DyteChat {
    return chatController.dyteChat
  }

  override fun getRecording(): DyteRecording {
    return recording
  }

  override fun getPolls(): DytePoll {
    return poll
  }

  override fun getMeta(): DyteMeta {
    return meta
  }

  override fun getPlugins(): DytePlugins {
    return plugins
  }

  override fun getWebinar(): DyteWebinar {
    return webinar
  }
}

internal interface IController {
  fun init(dyteMeetingInfoV2: DyteMeetingInfoV2)

  fun init(dyteMeetingInfo: DyteMeetingInfo)
  fun joinRoom()
  fun leaveRoom()

  fun getSelf(): DyteSelfParticipant
  fun getRecording(): DyteRecording

  fun getMeetingParticipant(): DyteRoomParticipants
  fun getChat(): DyteChat
  fun getPolls(): DytePoll
  fun getMeta(): DyteMeta
  fun getPlugins(): DytePlugins
  fun getWebinar(): DyteWebinar

  /**
   * Add meeting room events listener
   *
   * @param meetingRoomEventsListener
   */
  fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)
  fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)

  /**
   * Add self events listener
   *
   * @param selfEventsListener
   */
  fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener)
  fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener)

  /**
   * Add participant events listener
   *
   * @param participantEventsListener
   */
  fun addParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)
  fun removeParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)

  fun addCameraEventsListener(cameraEventsListener: DyteCameraEventsListener)
  fun removeCameraEventsListener(cameraEventsListener: DyteCameraEventsListener)
}

interface IControllerContainer {
  val chatController: IChatController
  val participantController: IParticipantController
  val pollsController: IPollController
  val metaController: IMetaController
  val selfController: ISelfController
  val roomNodeController: IRoomNodeController
  val presetController: IPresetController
  val hostController: IHostController
  val apiClient: IApiClient
  val socketController: ISocketController
  val socketMessageResponseParser: ISocketMessageResponseParser
  val platformUtilsProvider: IDytePlatformUtilsProvider
  val recordingController: IRecordingController
  val eventController: IEventController
  val connectionController: IConnectionController
  val permissionController: IPermissionController
  val pluginsController: IPluginsController
  val loggerController: ILoggerController
  val mediaSoupController: IMediaSoupController
  val webinarController: IWebinarController
}

data class MeetingInfo(
  val authToken: String,
  val config: MeetingConfig
) {
  internal var isV2Meeting: Boolean = false
  internal var meetingId: String? = null
  internal var roomName: String? = null
  internal var viewType: String = "GROUP_CALL"
  internal var meetingTitle: String? = null
  internal var meetingStartedAt: String? = null

  fun getBaseUrl(): String {
    return BASE_URL_PROD
  }

  companion object {
    val BASE_URL_PROD = "https://api.cluster.dyte.in"
    val BASE_URL_TEST = "https://api.staging.dyte.in"
  }

  fun getRoomName(): String {
    return meetingId ?: roomName ?: "no-name"
  }
}

data class MeetingConfig(
  val enableAudio: Boolean,
  val enableVideo: Boolean
)
