package io.dyte.core

import io.dyte.core.controllers.Controller
import io.dyte.core.listeners.DyteCameraEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DytePluginEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.models.DyteChat
import io.dyte.core.models.DyteMeetingInfo
import io.dyte.core.models.DyteMeetingInfoV2
import io.dyte.core.models.DyteMeta
import io.dyte.core.models.DytePlugins
import io.dyte.core.models.DytePoll
import io.dyte.core.models.DyteRecording
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.DyteWebinar
import io.dyte.core.models.MeetingRoomState
import io.dyte.core.models.MeetingRoomState.LEAVING
import io.dyte.core.models.MeetingRoomState.LEFT
import io.dyte.core.models.MeetingRoomState.NONE
import io.dyte.core.platform.IDytePlatformUtilsProvider

/**
 * Dyte mobile client
 *
 * todo : rename -> DyteClient
 *
 * @constructor Create empty Dyte mobile client
 */
open class DyteMobileClient(private val dytePlatformUtilsProvider: IDytePlatformUtilsProvider) :
  IDyteClient {
  private var controller: Controller = Controller(dytePlatformUtilsProvider)

  private var meetingRoomState: MeetingRoomState = NONE

  override val localUser: DyteSelfParticipant
    get() = controller.getSelf()

  override val participants: DyteRoomParticipants
    get() = controller.getMeetingParticipant()

  override val chat: DyteChat
    get() = controller.getChat()

  override val recording: DyteRecording
    get() = controller.getRecording()

  override val polls: DytePoll
    get() = controller.getPolls()

  override val meta: DyteMeta
    get() = controller.getMeta()

  override val plugins: DytePlugins
    get() = controller.getPlugins()

  override val webinar: DyteWebinar
    get() = controller.getWebinar()

  override fun init(dyteMeetingInfo: DyteMeetingInfo) {
    controller.init(dyteMeetingInfo)
  }

  override fun init(dyteMeetingInfo: DyteMeetingInfoV2) {
    controller.init(dyteMeetingInfo)
  }

  override fun joinRoom() {
    controller.joinRoom()
  }

  override fun leaveRoom() {
    meetingRoomState = LEAVING
    controller.leaveRoom()
    meetingRoomState = LEFT
  }

  override fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) {
    controller.addMeetingRoomEventsListener(meetingRoomEventsListener)
  }

  override fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) {
    controller.removeMeetingRoomEventsListener(meetingRoomEventsListener)
  }

  override fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    controller.addSelfEventsListener(selfEventsListener)
  }

  override fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    controller.removeSelfEventsListener(selfEventsListener)
  }

  override fun addParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener) {
    controller.addParticipantEventsListener(participantEventsListener)
  }

  override fun removeParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener) {
    controller.removeParticipantEventsListener(participantEventsListener)
  }

  override fun addCameraEventsListener(cameraEventsListener: DyteCameraEventsListener) {
    controller.addCameraEventsListener(cameraEventsListener)
  }

  override fun removeCameraEventsListener(cameraEventsListener: DyteCameraEventsListener) {
    controller.removeCameraEventsListener(cameraEventsListener)
  }

  override fun addPluginEventsListener(pluginEventsListener: DytePluginEventsListener) {
    controller.eventController.addPluginEventListener(pluginEventsListener)
  }

  override fun removePluginEventsListener(pluginEventsListener: DytePluginEventsListener) {
    controller.eventController.removePluginEventListener(pluginEventsListener)
  }
}

/**
 * Dyte client
 *
 * @constructor Create empty Dyte client
 */
internal interface IDyteClient {
  /**
   * This method makes the DyteClient ready to join in a given meeting.
   * Should be called from a background thread.
   *
   * @param dyteMeetingInfo
   */
  fun init(dyteMeetingInfo: DyteMeetingInfo)

  /**
   *
   */
  fun init(dyteMeetingInfo: DyteMeetingInfoV2)

  /**
   * Used to join in the room
   */
  fun joinRoom()

  /**
   * The leaveRoom() method can be used to leave a meeting.
   */
  fun leaveRoom()

  /**
   * The self object can be used to manipulate audio and video settings,
   * and other configurations for the local participant.
   * This exposes methods to enable and disable media tracks, share the user's screen, etc.
   */
  val localUser: DyteSelfParticipant

  /**
   * The participants object consists of 4 lists of participants.
   * waitlisted, joined, active, pinned.
   * @return [DyteRoomParticipants]
   */
  val participants: DyteRoomParticipants

  /**
   * The chat object stores the chat messages that were sent in the meeting.
   * This includes text messages, images, and files.
   */
  val chat: DyteChat

  /**
   * This class helps you start and stop recording of a given meeting.
   */
  val recording: DyteRecording

  /**
   * The polls object stores the polls that were initiated in the meeting.
   * It exposes methods to create and vote on polls.
   */
  val polls: DytePoll

  /**
   * Meta object contains information about meetingTitle, meetingStartedAt, meetingType.
   */
  val meta: DyteMeta

  val plugins: DytePlugins

  val webinar: DyteWebinar

  fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)
  fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)

  fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener)
  fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener)

  fun addParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)
  fun removeParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)

  fun addCameraEventsListener(cameraEventsListener: DyteCameraEventsListener)
  fun removeCameraEventsListener(cameraEventsListener: DyteCameraEventsListener)

  fun addPluginEventsListener(pluginEventsListener: DytePluginEventsListener)
  fun removePluginEventsListener(pluginEventsListener: DytePluginEventsListener)
}

