package io.dyte.mobilecorebridge

import io.dyte.core.DyteMobileClient
import io.dyte.core.controllers.DyteRecordingState
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.DytePlugin
import io.dyte.core.feat.DytePollMessage
import io.dyte.core.feat.DytePollOption
import io.dyte.core.feat.DytePollVote
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.listeners.DyteDataUpdateListener
import io.dyte.core.listeners.DyteLiveStreamEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DyteParticipantUpdateListener
import io.dyte.core.listeners.DytePluginEventsListener
import io.dyte.core.listeners.DytePollEventsListener
import io.dyte.core.listeners.DyteRecordingEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.listeners.DyteStageEventListener
import io.dyte.core.listeners.DyteWaitlistEventsListener
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.models.DyteMeetingInfo
import io.dyte.core.models.DyteMeetingInfoV2
import io.dyte.core.models.DyteVideoDevice

class DyteClient(private val dyteClient: DyteMobileClient) {

  var isRoomJoined: Boolean = false
  var core: DyteMobileClient = dyteClient

  fun initV1(meetingInfo: DyteMeetingInfo) {
    dyteClient.init(meetingInfo)
  }

  fun initV2(meetingInfo: DyteMeetingInfoV2) {
    dyteClient.init(meetingInfo)
  }

  fun joinRoom() {
    dyteClient.joinRoom()
    isRoomJoined = true
  }

  fun enableCache() {
    dyteClient.participants.enableCache()
  }

  fun disableCache() {
    dyteClient.participants.disableCache()
  }

  fun leaveRoom() {
    dyteClient.leaveRoom()
    isRoomJoined = false
  }

  fun getMetaData(): Map<String, Any?> {
    return dyteClient.meta.toMap()
  }

  fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) =
    dyteClient.addMeetingRoomEventsListener(meetingRoomEventsListener)

  fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener) =
    dyteClient.removeMeetingRoomEventsListener(meetingRoomEventsListener)

  fun participants() = dyteClient.participants

  fun addParticipantEventListener(participantEventsListener: DyteParticipantEventsListener) {
    dyteClient.addParticipantEventsListener(participantEventsListener)
  }

  fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    dyteClient.addSelfEventsListener(selfEventsListener)
  }

  fun removeParticipantEventListener(participantEventListener: DyteParticipantEventsListener) {
    dyteClient.removeParticipantEventsListener(participantEventListener)
  }

  fun addPluginEventsListener(pluginEventsListener: DytePluginEventsListener) =
    dyteClient.addPluginEventsListener(pluginEventsListener)

  fun removePluginEventsListener(pluginEventsListener: DytePluginEventsListener) {
    dyteClient.removePluginEventsListener(pluginEventsListener)
  }

  fun removeLocalUserListener(selfEventsListener: DyteSelfEventsListener) {
    dyteClient.removeSelfEventsListener(selfEventsListener)
  }

  fun addParticipantUpdateListener(
    participant: DyteMeetingParticipant,
    participantUpdateListener: DyteParticipantUpdateListener,
  ) {
    participant.addParticipantUpdateListener(participantUpdateListener)
  }

  fun removeParticipantUpdateListener(
    participant: DyteMeetingParticipant,
    participantUpdateListener: DyteParticipantUpdateListener,
  ) {
    participant.removeParticipantUpdateListener(participantUpdateListener)
  }

  fun removeParticipantUpdateListeners(participant: DyteMeetingParticipant) {
    participant.removeParticipantUpdateListeners()
  }

  fun addChatListener(chatEventsListener: DyteChatEventsListener) {
    dyteClient.addChatEventsListener(chatEventsListener)
  }

  fun removeChatListener(chatEventsListener: DyteChatEventsListener) {
    dyteClient.removeChatEventsListener(chatEventsListener)
  }

  fun addPollListener(pollEventsListener: DytePollEventsListener) {
    dyteClient.addPollEventsListener(pollEventsListener)
  }

  fun removePollListener(pollEventsListener: DytePollEventsListener) {
    dyteClient.removePollEventsListener(pollEventsListener)
  }

  fun addDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    dyteClient.addDataUpdateListener(dataUpdateListener)
  }

  fun removeDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    dyteClient.removeDataUpdateListener(dataUpdateListener)
  }

  fun addRecordingEventListener(recordingEventsListener: DyteRecordingEventsListener) {
    dyteClient.addRecordingEventsListener(recordingEventsListener)
  }

  fun removeRecordingEventListener(recordingEventsListener: DyteRecordingEventsListener) {
    dyteClient.removeRecordingEventsListener(recordingEventsListener)
  }

  fun addWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    dyteClient.addWaitlistEventsListener(waitlistEventsListener)
  }

  fun removeWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    dyteClient.removeWaitlistEventsListener(waitlistEventsListener)
  }

  fun addStageEventsListener(stageEventsListener: DyteStageEventListener) {
    dyteClient.addStageEventsListener(stageEventsListener)
  }

  fun removeStageEventsListener(stageEventsListener: DyteStageEventListener) {
    dyteClient.removeStageEventsListener(stageEventsListener)
  }

  fun addLivestreamEventsListener(livestreamEventsListener: DyteLiveStreamEventsListener) {
    dyteClient.addLiveStreamEventsListener(livestreamEventsListener)
  }

  fun removeLivestreamEventsListener(livestreamEventsListener: DyteLiveStreamEventsListener) {
    dyteClient.removeLiveStreamEventsListener(livestreamEventsListener)
  }

  fun enableAudio() {
    dyteClient.localUser.enableAudio()
  }

  fun disableAudio() {
    dyteClient.localUser.disableAudio()
  }

  fun enableVideo() {
    dyteClient.localUser.enableVideo()
  }

  fun disableVideo() {
    dyteClient.localUser.disableVideo()
  }

  fun sendTextMessage(text: String) {
    dyteClient.chat.sendTextMessage(text)
  }

  fun sendFileMessage(path: String, text: String) {
    dyteClient.chat.sendFileMessage(path)
  }

  fun sendImageMessage(path: String, text: String) {
    dyteClient.chat.sendImageMessage(path)
  }

  fun setDisplayName(name: String) {
    dyteClient.localUser.setDisplayName(name)
  }

  fun getAudioDevices(): List<DyteAudioDevice> {
    return dyteClient.localUser.getAudioDevices()
  }

  fun getVideoDevices(): List<DyteVideoDevice> {
    return dyteClient.localUser.getVideoDevices()
  }

  fun setAudioDevice(dyteAndroidDevice: DyteAudioDevice) {
    dyteClient.localUser.setAudioDevice(dyteAndroidDevice)
  }

  fun setVideoDevice(dyteVideoDevice: DyteVideoDevice) {
    dyteClient.localUser.setVideoDevice(dyteVideoDevice)
  }

  fun getSelectedVideoDevice(): DyteVideoDevice? {
    return dyteClient.localUser.getSelectedVideoDevice()
  }

  fun getSelectedAudioDevice(): DyteAudioDevice? {
    return dyteClient.localUser.getSelectedAudioDevice()
  }

  fun switchCamera() {
    return dyteClient.localUser.switchCamera()
  }

  fun createPoll(pollChars: Map<String, Any>) {
    dyteClient.polls.create(
      pollChars["question"] as String,
      pollChars["options"] as List<String>,
      pollChars["anonymous"] as Boolean,
      pollChars["hideVotes"] as Boolean,
    )
  }

  fun voteOnPoll(pollVote: Map<String, Any>) {
    val pollMsg = pollVote["pollMessage"] as Map<String, Any>
    val options = pollMsg["options"] as List<Map<String, Any>>
    val pollOptions: List<DytePollOption> = options.map { DytePollOption.fromMap(it) }
    val pollMessage: DytePollMessage =
      DytePollMessage(
        pollMsg["id"] as String,
        pollMsg["question"] as String,
        pollMsg["anonymous"] as Boolean,
        pollMsg["hideVotes"] as Boolean,
        pollMsg["createdBy"] as String,
        pollOptions,
      )

    val votedOption = pollVote["pollOption"] as Map<String, Any>
    val optVotes = votedOption["votes"] as? List<Map<String, Any>>

    val optionVotes = optVotes?.map { DytePollVote.fromMap(it) } ?: arrayOf<DytePollVote>()
    val pollOption =
      DytePollOption(
        votedOption["text"] as String,
        optionVotes as List<DytePollVote>,
        votedOption["count"] as Int,
      )
    dyteClient.polls.vote(pollMessage, pollOption)
  }

  fun setPage(pageNumber: Int) {
    dyteClient.participants.setPage(pageNumber)
  }

  fun startRecording() {
    dyteClient.recording.start()
  }

  fun stopRecording() {
    dyteClient.recording.stop()
  }

  fun getRecordingState(): DyteRecordingState {
    return dyteClient.recording.recordingState
  }

  private fun dytePluginMap(dytePlugin: DytePlugin): Map<String, Any> {
    return mapOf<String, Any>(
      "id" to dytePlugin.id,
      "name" to dytePlugin.name,
      "description" to dytePlugin.description,
      "picture" to dytePlugin.picture,
      "private" to dytePlugin.isPrivate,
      "staggered" to dytePlugin.staggered,
      "baseURL" to dytePlugin.baseURL,
    )
  }

  fun getAllPlugins(): List<Map<String, Any>> {
    val allPlugins: List<DytePlugin> = dyteClient.plugins.all
    return allPlugins.map { dytePluginMap(it) }
  }

  fun getActivePlugins(): List<Map<String, Any>> {
    val activePlugins: List<DytePlugin> = dyteClient.plugins.active
    return activePlugins.map { dytePluginMap(it) }
  }

  fun activatePlugin(pluginId: String) {
    val allPlugins: List<DytePlugin> = dyteClient.plugins.all
    val plugin: DytePlugin? = allPlugins.find { it.id == pluginId }
    plugin?.activate()
  }

  fun deactivatePlugin(pluginId: String) {
    val activePlugins: List<DytePlugin> = dyteClient.plugins.active
    val activePlugin: DytePlugin? = activePlugins.find { it.id == pluginId }
    activePlugin?.deactivate()
  }

  fun pinParticipant(participantId: String) {
    val pinningParticipant = dyteClient.participants.joined.first { it.id == participantId }
    pinningParticipant.pin()
  }

  fun unpinParticipant() {
    dyteClient.participants.pinned?.unpin()
  }

  fun disableParticipantAudio(participantId: String) {
    val participant = dyteClient.participants.joined.first { it.id == participantId }
    participant.disableAudio()
  }

  fun disableParticipantVideo(participantId: String) {
    val participant = dyteClient.participants.joined.first { it.id == participantId }
    participant.disableVideo()
  }

  fun kickParticipant(participantId: String) {
    val participant = dyteClient.participants.joined.first { it.id == participantId }
    participant.kick()
  }

  fun kickAll() {
    dyteClient.participants.kickAll()
  }

  fun acceptWaitListedRequest(participantId: String) {
    val participant = dyteClient.participants.waitlisted.first { it.id == participantId }
    participant.acceptWaitListedRequest()
  }

  fun rejectWaitListedRequest(participantId: String) {
    val participant = dyteClient.participants.waitlisted.first { it.id == participantId }
    participant.rejectWaitListedRequest()
  }

  fun muteAllAudios() {
    dyteClient.participants.disableAllAudio()
  }

  fun disableAllVideos() {
    dyteClient.participants.disableAllVideo()
  }

  fun startLvs() {
    dyteClient.liveStream.start()
  }

  fun stopLvs() {
    dyteClient.liveStream.stop()
  }

  fun getStreamState(): String? {
    return dyteClient.liveStream.state.toMap()["state"]
  }

  fun getStreamUrl(): String? {
    return dyteClient.liveStream.liveStreamUrl
  }

  fun getStreamRoomName(): String {
    return dyteClient.liveStream.roomName
  }

  fun requestToJoinStage() {
    return dyteClient.stage.requestAccess()
  }

  fun withdrawJoinStageRequest() {
    return dyteClient.stage.cancelRequestAccess()
  }

  fun grantAccessToStage(peerId: String) {
    return dyteClient.stage.grantAccess(peerId)
  }

  fun denyAccessToStage(peerId: String) {
    return dyteClient.stage.denyAccess(peerId)
  }

  fun acceptAllJoinStageRequests() {
    return dyteClient.stage.grantAccessAll()
  }

  fun denyAllJoinStageRequests() {
    return dyteClient.stage.denyAccessAll()
  }

  fun joinStage() {
    return dyteClient.stage.join()
  }

  fun leaveStage() {
    return dyteClient.stage.leave()
  }

  fun kickPeerFromStage(participantId: String) {
    return dyteClient.stage.kick(participantId)
  }

  fun broadcastMessage(type: String, payload: Map<String, *>) {
    return dyteClient.participants.broadcastMessage(type, payload)
  }

  fun enableScreenShare() {
    return dyteClient.localUser.enableScreenshare()
  }

  fun disableScreenshare() {
    return dyteClient.localUser.disableScreenshare()
  }
}
