package io.dyte.media.hive

import io.dyte.webrtc.*
import io.dyte.media.hive.handlers.HiveCodecOptions
import io.dyte.media.hive.handlers.HiveHandlerInterface
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.flow.MutableSharedFlow

class HiveProducerOptions(
  val track: MediaStreamTrack?,
  val stream: MediaStream,
  val encodings: List<CommonRtpEncodingParameters>?,
  val codecOptions: HiveCodecOptions?,
  val stopTracks: Boolean?,
  val disableTrackOnPause: Boolean?,
  val zeroRtpOnPause: Boolean?,
  val appData: Map<String, Any>?
)

class HiveInternalProducerOptions(
  val id: String,
  val localId: String,
  val track: MediaStreamTrack,
  val stopTracks: Boolean,
  val disableTrackOnPause: Boolean,
  val zeroRtpOnPause: Boolean,
  val handler: HiveHandlerInterface,
  val appData: Map<String, Any>
)

class HiveProducer(
  options: HiveInternalProducerOptions
) {
  /** Producer Id */
  private val _id = options.id

  /** Local Id */
  private val _localId = options.localId

  /** Closed flag */
  private var _closed = false

  /** Associated RtpSender */
  private val _handler = options.handler

  /** Local track */
  private var _track = options.track

  /** Producer kind */
  private val _kind = this._track.kind

  /** Paused flag */
  private var _paused = if (options.disableTrackOnPause) !options.track.enabled else false

  /** Video max spatial layer */
  private var _maxSpatialLayer: Long? = null

  /** Whether the Producer should call stop() in the given tracks */
  private var _stopTracks = options.stopTracks

  /** Whether the Producer should set track.enabled = false when paused */
  private var _disableTrackOnPause = options.disableTrackOnPause

  /** Whether we should replace the RTCRtpSender.track with null when paused */
  private var _zeroRtpOnPause = options.zeroRtpOnPause

  /** Custom data */
  private val _appData = options.appData

  /** Observer for external libs listen for few events */
  val observer = MutableSharedFlow<HiveEmitData>()

  init {
    @Suppress("DeferredResultUnused")
    CoroutineScope(Dispatchers.Default).async {
      _handleTrack()
    }
  }

  /** Producer Id */
  fun getId() = this._id

  /** Local Id */
  fun getLocalId() = this._localId

  /** Whether the Producer is closed */
  fun getClosed() = this._closed

  /** Media kind */
  fun getKind() = this._kind

  /** The associated track */
  fun getTrack() = this._track

  /** Whether the Producer is paused */
  fun getPaused() = this._paused

  /** Max spatial layer */
  fun getMaxSpatialLayer() = this._maxSpatialLayer

  /** App custom data */
  fun getAppData() = this._appData

//  fun setAppData() = throw Error("Cannot override appData object")

  /** Closes the Producer */
  suspend fun close(reason: String?) {
    if (this._closed) return

    println("DyteMediaClient: Producer: close() with reason: $reason")

    this._closed = true

    this._destroyTrack()

    if (reason == REASON_TRANSPORT_CLOSED) {
      this.observer.emit(HiveEmitData("close"))
      return
    }

    val handlerResult = this._handler.stopSending(this._localId)

    this.observer.emit(HiveEmitData(
      eventName = "close",
      data = mapOf(
        "offerSdp" to handlerResult.offerSdp,
        "callback" to handlerResult.callback,
        "reason" to reason
      )
    ))
  }

  /** Get associated RtpSender stats */
  fun getStats(): List<RtcStatsReport> {
    if (this._closed) throw IllegalStateException("Producer closed")

    return this._handler.getSenderStats(this._localId)
  }

  /** Pauses sending media */
  suspend fun pause() {
    println("DyteMediaClient: Producer: pause() for ${this._id}")

    if (this._closed) println("DyteMediaClient: Producer: pause(): Producer closed")

    this._paused = true

    if (this._disableTrackOnPause) this._track.enabled = false

    if (this._zeroRtpOnPause) this._handler.replaceTrack(this._localId, null)

    this.observer.emit(HiveEmitData("pause"))
  }

  /** Resumes sending media */
  suspend fun resume() {
    println("DyteMediaClient: Producer: resume() for ${this._id}")

    if (this._closed) {
      println("DyteMediaClient: Producer: resume(): Producer closed")

      return
    }

    this._paused = false

    if (this._disableTrackOnPause) this._track.enabled = true

    if (this._zeroRtpOnPause) this._handler.replaceTrack(this._localId, this._track)

    this.observer.emit(HiveEmitData("resume"))
  }

  /** Replaces the current track with a new one or null */
  suspend fun replaceTrack(track: MediaStreamTrack?) {
    println("DyteMediaClient: Producer: replaceTrack() with $track")

    if (this._closed) {
      // This must be done here. Otherwise, there is no chance to stop the given track
      if (track != null && this._stopTracks) {
        try {
          track.stop()
        } catch (e: Error) {
          println(e)
        }
      }

      throw IllegalStateException("Producer closed")
    } else if (track != null && track.readyState is MediaStreamTrackState.Ended) {
      throw IllegalStateException("Track ended")
    }

    // Do nothing if this is the same track as the current handled one
    if (track == this._track) {
      println("DyteMediaClient: Producer: replaceTrack(): Same track, ignored")

      return
    }

    if (!this._zeroRtpOnPause or !this._paused) this._handler.replaceTrack(this._localId, track)

    // Destroy the previous track
    this._destroyTrack()

    // Set the new track
    this._track = track!!

    // If this producer was paused/resumed and the state of the new track does not match, fix it
    if (this._disableTrackOnPause) {
      if (!this._paused) this._track.enabled = true
      else if (this._paused) this._track.enabled = false
    }

    // Handle the effective track
    this._handleTrack()
  }

  /** Sets the video max spatial layer to be sent */
  suspend fun setMaxSpatialLayer(spatialLayer: Long) {
    if (this._closed) throw IllegalStateException("closed")
    else if (this._kind != MediaStreamTrackKind.Video) throw UnsupportedOperationException("Not a Video Producer")

    if (spatialLayer == this._maxSpatialLayer) return

    this._handler.setMaxSpatialLayer(this._localId, spatialLayer)

    this._maxSpatialLayer = spatialLayer
  }

  /** Sets the DSCP value */
  suspend fun setRtpEncodingParameters(params: CommonRtpEncodingParameters) {
    if (this._closed) throw IllegalStateException("Producer closed")

    this._handler.setRtpEncodingParameters(this._localId, params)
  }

  private suspend fun _onTrackEnded() {
    println("DyteMediaClient: Producer: Track ended event")

    this.observer.emit(HiveEmitData("trackended"))
  }

  private suspend fun _handleTrack() {
    this._track.onEnded.collect {
      _onTrackEnded()
    }
  }

  private fun _destroyTrack() {
    try {
      // Just stop the track unless the app set stopTracks to false
      if (this._stopTracks) this._track.stop()
    } catch (e: Error) {
      println(e)
    }
  }
}