package io.dyte.media.hive

import io.dyte.webrtc.*
import io.dyte.media.hive.handlers.HiveHandlerInterface
import io.dyte.media.utils.IMediaClientLogger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.launch

open class HiveConsumerOptions (
  open val id: String?,
  open val producerId: String,
  open val producingPeerId: String,
  open val kind: MediaStreamTrackKind,
  open var paused: Boolean?,
  open val appData: Map<String, Any>?,
)

class HiveInternalConsumerOptions(
  override val id: String?,
  override val producerId: String,
  override val producingPeerId: String,
  override val kind: MediaStreamTrackKind,
  override var paused: Boolean?,
  override val appData: Map<String, Any>?,

  val localId: String,
  val handler: HiveHandlerInterface,
  val track: MediaStreamTrack?,
  val reuseTrack: Boolean? = null
): HiveConsumerOptions(id, producerId, producingPeerId, kind, paused, appData)

class HiveConsumer(
  private val options: HiveInternalConsumerOptions,
  private val coroutineScope: CoroutineScope,
  private val logger: IMediaClientLogger
) {
  /** Consumer Id */
  private val _id = options.id!!

  /** Local Id */
  private val _localId = options.localId

  /** Associated producer Id */
  private val _producerId = options.producerId

  /** Closed flag */
  private var _closed = false

  /** Associated RtpReceiver */
  private val _handler = options.handler

  /** Remote track */
  private val _track = options.track!!

  /** Destroy track on close flag */
  private val _reuseTrack = options.reuseTrack ?: false

  /** Paused flag */
  private var _paused = options.paused ?: false

  /** Peer Id of consumer */
  private var _peerId = options.producingPeerId

  /** App custom data */
  private val _appData = options.appData

  private val _kind = this._track.kind

  /** Observer is for external libs to listen for some events */
  val observer = MutableSharedFlow<HiveEmitData>()

  init {
    coroutineScope.launch {
      _handleTrack()
    }
  }

  /** Consumer Id */
  fun getId() = this._id

  /** Peer Id of consumer */
  fun getPeerId() = this._peerId

  /** Local Id */
  fun getLocalId() = this._localId

  /** Associated producer Id */
  fun getProducerId() = this._producerId

  /** Whether the Consumer is closed */
  fun getClosed() = this._closed

  /** Media kind */
  fun getKind() = this._kind

  /** The associated track */
  fun getTrack() = this._track

  /** Whether the Consumer is paused */
  fun getPaused() = this._paused

  fun getRemotelyPaused() = this.options.paused

  /** Custom data */
  fun getAppData() = this._appData

//  fun setAppData() = throw Error("Cannot override appData object")

  /** Closes the Consumer */
  suspend fun close(reason: String) {
    if (this._closed) return

    logger.traceLog("DyteMediaClient: Consumer: close() with reason: $reason")

    this._closed = true

    if (!this._reuseTrack) this._destroyTrack()

    this.observer.emit(HiveEmitData(
      eventName = "close",
      data = reason
    ))
  }

  /** Get associated RtpReceiver stats */
  suspend fun getStats(): List<RtcStatsReport> {
    if (this._closed) throw IllegalStateException("Consumer closed")

    return this._handler.getReceiverStats(this._localId)
  }

  /** Pauses receiving media */
  suspend fun pause() {
    logger.traceLog("DyteMediaClient: Consumer: pause() for ${this._id}")

    if (this._closed) {
      logger.traceLog("DyteMediaClient: Consumer: pause(): Consumer closed")

      return
    }

    this._paused = true
    this._track.enabled = false

    this.observer.emit(HiveEmitData("pause"))
  }

  /** Resumes receiving media */
  suspend fun resume() {
    logger.traceLog("DyteMediaClient: Consumer: resume() for ${this._id}")

    if (this._closed) {
      logger.traceLog("DyteMediaClient: Consumer: resume(): Consumer closed")

      return
    }

    this._paused = false
    this._track.enabled = true

    this.observer.emit(HiveEmitData("resume"))
  }

  private suspend fun _onTrackEnded() {
    logger.traceLog("DyteMediaClient: Consumer: Track ended event")

    this.observer.emit(HiveEmitData("trackended"))
  }

  private suspend fun _handleTrack() {
    this._track.onEnded.collect {
      _onTrackEnded()
    }
  }

  private fun _destroyTrack() {
    try {
      this._track.stop()
    } catch (e: Error) {
      logger.traceError(e.toString())
    }
  }
}