package io.dyte.media.utils

import io.dyte.webrtc.CommonRtpEncodingParameters
import io.dyte.webrtc.RtpEncodingParameters
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.descriptors.serialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder
import kotlinx.serialization.json.JsonNames
import kotlinx.serialization.json.JsonPrimitive

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class RtpCapabilities(
    @JsonNames("codecs")
    val codecs: MutableList<LocalRtpCodecParameters>? = mutableListOf(),
    @JsonNames("headerExtentions")
    val headerExtensions: MutableList<RtpHeaderExtension> = mutableListOf(),
    @JsonNames("fecMechanisms")
    val fecMechanisms: List<String> = emptyList()
) {
    companion object {
        fun copy(
            old: RtpCapabilities,
            codecs: List<LocalRtpCodecParameters>?,
            headerExtensions: List<RtpHeaderExtension>?,
            fecMechanisms: List<String>?
        ): RtpCapabilities {
            return RtpCapabilities(
                codecs = if (codecs != null) codecs.toMutableList() else old.codecs,
                headerExtensions = if (headerExtensions != null) headerExtensions.toMutableList()
                    else old.headerExtensions,
                fecMechanisms = if (fecMechanisms != null) fecMechanisms else old.fecMechanisms
            )
        }
    }
}

@kotlinx.serialization.Serializable(with = RtpHeaderDirectionSerializer::class)
enum class RtpHeaderDirection {
    SendRecv,
    SendOnly,
    RecvOnly,
    Inactive;

    companion object {
        private val types: Map<String, RtpHeaderDirection> = mapOf(
            "sendrecv" to SendRecv,
            "sendonly" to SendOnly,
            "recvonly" to RecvOnly,
            "inactive" to Inactive
        )

        val values = mapOf(
            SendRecv to "sendrecv",
            SendOnly to "sendonly",
            RecvOnly to "recvonly",
            Inactive to "inactive"
        )

        fun fromString(i: String) : RtpHeaderDirection = requireNotNull(RtpHeaderDirection.types[i])

        fun value(r: RtpHeaderDirection?): String = requireNotNull(RtpHeaderDirection.values[r])
    }
}

object RtpHeaderDirectionSerializer : KSerializer<RtpHeaderDirection> {
    override val descriptor: SerialDescriptor = serialDescriptor<String>()
    override fun serialize(output: Encoder, obj: RtpHeaderDirection) {
        output.encodeString(RtpHeaderDirection.value(obj))
    }
    override fun deserialize(input: Decoder): RtpHeaderDirection {
        return RtpHeaderDirection.fromString(input.decodeString())
    }
}

enum class RTCRtpMediaType {
    RTCRtpMediaTypeAudio,
    RTCRtpMediaTypeVideo,
    RTCRtpMediaTypeData;

    companion object{
        private val types: Map<String, RTCRtpMediaType> = mapOf(
            "audio" to RTCRtpMediaTypeAudio,
            "video" to RTCRtpMediaTypeVideo,
            "data" to RTCRtpMediaTypeData
        )

        val values = mapOf(
            RTCRtpMediaTypeAudio to "audio",
            RTCRtpMediaTypeVideo to "video",
            RTCRtpMediaTypeData to "data"
        )

        fun fromString(i: String) : RTCRtpMediaType = requireNotNull(RTCRtpMediaType.types[i.lowercase()])

        fun value(r: RTCRtpMediaType?): String = requireNotNull(RTCRtpMediaType.values[r])
    }
}

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class RtcpFeedback(
    @JsonNames("type")
    val type: String? = null,
    @JsonNames("parameter")
    var parameter: String? = ""
)

class ExtendedRtpCodec(
    val kind: RTCRtpMediaType,
    val mimeType: String,
    val clockRate: Int,
    val channels: Int?,
    val rtcpFeedback: List<RtcpFeedback> =  emptyList(),
    var localPayloadType: Int? = null,
    var localRtxPayloadType: Int? = null,
    var remotePayloadType: Int? = null,
    var remoteRtxPayloadType: Int? = null,
    var localParameters: MutableMap<String, JsonPrimitive?>,
    var remoteParameters: MutableMap<String, JsonPrimitive?>
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class LocalRtpCodecParameters(
    @JsonNames("kind")
    var kind: RTCRtpMediaType? = null,
    @JsonNames("mimeType")
    val mimeType: String = "",
    @JsonNames("preferredPayloadType")
    val preferredPayloadType: Int? = null,
    @JsonNames("clockRate")
    val clockRate: Int? = null,
    @JsonNames("channels")
    var numChannels: Int? = 1,
    @JsonNames("parameters")
    var parameters: MutableMap<String, JsonPrimitive?> = mutableMapOf(),
    //var parameters: RtpCodecParameters? = null,
    @JsonNames("rtcpFeedback")
    var rtcpFeedback: MutableList<RtcpFeedback> = mutableListOf(),
    @JsonNames("payloadType")
    var payloadType: Int? = null
)

class ExtendedRtpHeaderExtension(
    val kind: RTCRtpMediaType,
    val uri: String,
    val sendId: Int,
    val recvId: Int,
    val encrypt: Boolean,
    var direction: RtpHeaderDirection
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class RtpHeaderExtension(
    @JsonNames("kind")
    val kind: RTCRtpMediaType? = null,
    @JsonNames("uri")
    val uri: String? = null,
    @JsonNames("preferredId")
    val preferredId: Int? = null,
    @JsonNames("preferredEncrypt")
    var preferredEncrypt: Boolean? = null,
    @JsonNames("direction")
    var direction: RtpHeaderDirection? = null,
    @JsonNames("id")
    val id: Int? = null,
    @JsonNames("encrypt")
    val encrypted: Boolean? = null
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class RtxSsrc(
    @JsonNames("ssrc")
    val ssrc: Int?
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class RtpHeaderExtensionParameters(
    @JsonNames("uri")
    val uri: String? = null,
    @JsonNames("id")
    val id: Int? = null,
    @JsonNames("encrypt")
    var encrypted: Boolean? = null,
    @JsonNames("parameters")
    val parameters: String? = null,
)

class LocalRtpParameters(
    var mid: String? = null,
    var codecs: MutableList<LocalRtpCodecParameters> = mutableListOf(),
    var headerExtension: MutableList<RtpHeaderExtensionParameters> = mutableListOf(),
    var encodings: List<CommonRtpEncodingParameters>? = listOf(),
    var rtcp: LocalRtcpParameters? = null
)

class LocalRtcpParameters(
    val mux: Boolean? = null,
    var cname: String = "",
    var reducedSize: Boolean = true
)