package io.dyte.media

import io.dyte.webrtc.IceServer
import io.dyte.webrtc.IceTransportPolicy
import io.dyte.media.handlers.HandlerInterface
import io.dyte.media.utils.*
import kotlinx.coroutines.flow.MutableSharedFlow

class Device {
    var loaded: Boolean = false
    private var _extendedRtpCapabilities: ExtendedRtpCapabilities? = null
    private var _recvRtpCapabilities: RtpCapabilities? =  null
    private var _canProduceByKind: CanProduceByKind? = null
    private var _sctpCapabilities: SctpCapabilities? = null
    private val observer = MutableSharedFlow<EmitData>()

    var rtpCapabilities: () -> RtpCapabilities? = {
        if (loaded == false) throw Exception("not loaded")
        else _recvRtpCapabilities
    }

    var sctpCapabilities: () -> SctpCapabilities? = {
        if (loaded == false) throw Exception("not loaded")
        else _sctpCapabilities
    }

    suspend fun load(routerRtpCapabilities: RtpCapabilities) {
        println("MEDIASOUP: Device: load() [routerRtpCapabilities:${routerRtpCapabilities.toString()}]")

        var handler: HandlerInterface? = null

        try {
            if (loaded) throw Exception("already loaded")

            Ortc.validateRtpCapabilities(routerRtpCapabilities)
            handler = HandlerInterface.handlerFactory()

            val nativeRtpCapabilities: RtpCapabilities = handler.getNativeRtpCapabilities()

            println("MEDIASOUP: Device: load() | got native RTP capabilities:$nativeRtpCapabilities")

            Ortc.validateRtpCapabilities(nativeRtpCapabilities)
            _extendedRtpCapabilities = Ortc.getExtendedRtpCapabilities(nativeRtpCapabilities, routerRtpCapabilities)

            println("MEDIASOUP: Device: load() | got extended RTP capabilities:$_extendedRtpCapabilities")

            _canProduceByKind = CanProduceByKind(
                audio = Ortc.canSend(RTCRtpMediaType.RTCRtpMediaTypeAudio, _extendedRtpCapabilities!!),
                video = Ortc.canSend(RTCRtpMediaType.RTCRtpMediaTypeVideo, _extendedRtpCapabilities!!)
            )

            _recvRtpCapabilities = Ortc.getRecvRtpCapabilities(_extendedRtpCapabilities!!)
            Ortc.validateRtpCapabilities(_recvRtpCapabilities)

            println("MEDIASOUP: Device: load() | got receiving RTP capabilities:$_recvRtpCapabilities")

            _sctpCapabilities = handler.getNativeSctpCapabilities()

            println("MEDIASOUP: Device: load() | got native SCTP capabilities:$_sctpCapabilities")

            Ortc.validateSctpCapabilities(_sctpCapabilities)

            println("MEDIASOUP: Device: load() successed")

            loaded = true
            handler.close()
        } catch (e: Error) {
            if (handler != null) handler.close()

            throw e
        }
    }

    fun canProduce(kind: RTCRtpMediaType): Boolean {
        if (!loaded) {
            throw Exception("not loaded")
        } else if (kind != RTCRtpMediaType.RTCRtpMediaTypeAudio && kind != RTCRtpMediaType.RTCRtpMediaTypeVideo) {
            throw Exception("invalid kind ${RTCRtpMediaType.value(kind)}")
        }

        return _canProduceByKind!!.canIt(kind)
    }

    suspend private fun _createTransport(
        direction: Direction,
        id: String,
        iceParameters: IceParameters,
        iceCandidates: List<IceCandidate>,
        dtlsParameters: DtlsParameters,
        sctpParameters: SctpParameters? = null,
        iceServers: List<IceServer> = emptyList(),
        iceTransportPolicy: IceTransportPolicy? = null,
        additionalSettings: Map<String, Any> = emptyMap(),
        proprietaryConstraints: Map<String, Any> = emptyMap(),
        appData: Map<String, Any> = emptyMap(),
        producerCallback: ((Any?) -> Unit)? = null,
        consumerCallback: ((Any?, Any?) -> Unit)? = null
    ): Transport {
        if (!loaded) throw Exception("not loaded")

        val transport = Transport(
            direction = direction,
            id = id,
            iceParameters = iceParameters,
            iceCandidates = iceCandidates,
            dtlsParameters = dtlsParameters,
            sctpParameters = sctpParameters,
            iceServers = iceServers,
            iceTransportPolicy = iceTransportPolicy,
            additionalSettings = additionalSettings.toMutableMap(),
            proprietaryConstraints = proprietaryConstraints,
            appData = appData,
            extendedRtpCapabilities = _extendedRtpCapabilities,
            canProduceByKind = _canProduceByKind!!,
            producerCallback = producerCallback,
            consumerCallback = consumerCallback
        )

        observer.emit(EmitData(
            name = "newtransport",
            data = mapOf(
                "transport" to transport
            )
        ))

        return transport
    }

    suspend fun createSendTransport(
        id: String,
        iceParameters: IceParameters,
        iceCandidates: List<IceCandidate>,
        dtlsParameters: DtlsParameters,
        sctpParameters: SctpParameters? = null,
        iceServers: List<IceServer> = emptyList(),
        iceTransportPolicy: IceTransportPolicy? = null,
        additionalSettings: Map<String, Any> = emptyMap(),
        proprietaryConstraints: Map<String, Any> = emptyMap(),
        appData: Map<String, Any> = emptyMap(),
        producerCallback: ((Any?) -> Unit)? = null
    ): Transport {
        println("MEDIASOUP: Device: createSendTransport()")

        return _createTransport(
            direction = Direction.send,
            id = id,
            iceParameters = iceParameters,
            iceCandidates = iceCandidates,
            dtlsParameters = dtlsParameters,
            sctpParameters = sctpParameters,
            iceServers = iceServers,
            iceTransportPolicy = iceTransportPolicy,
            additionalSettings = additionalSettings,
            proprietaryConstraints = proprietaryConstraints,
            appData = appData,
            producerCallback = producerCallback
        )
    }

    suspend fun createRecvTransport(
        id: String,
        iceParameters: IceParameters,
        iceCandidates: List<IceCandidate>,
        dtlsParameters: DtlsParameters,
        sctpParameters: SctpParameters? = null,
        iceServers: List<IceServer> = emptyList(),
        iceTransportPolicy: IceTransportPolicy? = null,
        additionalSettings: Map<String, Any> = emptyMap(),
        proprietaryConstraints: Map<String, Any> = emptyMap(),
        appData: Map<String, Any> = emptyMap(),
        consumerCallback: ((Any?, Any?) -> Unit)? = null
    ): Transport {
        println("MEDIASOUP: Device: createRecvTransport()")

        return _createTransport(
            direction = Direction.recv,
            id = id,
            iceParameters = iceParameters,
            iceCandidates = iceCandidates,
            dtlsParameters = dtlsParameters,
            sctpParameters = sctpParameters,
            iceServers = iceServers,
            iceTransportPolicy = iceTransportPolicy,
            additionalSettings = additionalSettings,
            proprietaryConstraints = proprietaryConstraints,
            appData = appData,
            consumerCallback = consumerCallback
        )
    }
}