package io.dyte.media

import io.dyte.media.handlers.sdp.Connection
import io.dyte.media.handlers.sdp.Fingerprint
import io.dyte.media.handlers.sdp.MediaObject
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonNames

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Origin(
  @JsonNames("username") val username: String,
  @JsonNames("sessionId") val sessionId: Long,
  @JsonNames("sessionVersion") var sessionVersion: Int = 0,
  @JsonNames("netType") val netType: String,
  @JsonNames("ipVer") var ipVer: Int,
  @JsonNames("address") var address: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Invalid(@JsonNames("value") val value: String)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Timing(@JsonNames("start") val start: Int, @JsonNames("stop") val stop: Int)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Group(@JsonNames("type") val type: String, @JsonNames("mids") var mids: String)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class MsidSemantic(
  @JsonNames("semantic") val semantic: String,
  @JsonNames("token") val token: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class SdpObject(
  @JsonNames("version") val version: Int? = null,
  @JsonNames("origin") val origin: Origin? = null,
  @JsonNames("name") val name: String? = null,
  @JsonNames("invalid") val invalid: List<Invalid> = emptyList(),
  @JsonNames("description") val description: String? = null,
  @JsonNames("timing") val timing: Timing? = null,
  @JsonNames("connection") val connection: Connection? = null,
  @JsonNames("iceUfrag") val iceUfrag: String? = null,
  @JsonNames("icePwd") val icePwd: String? = null,
  @JsonNames("fingerprint") var fingerprint: Fingerprint? = null,
  @JsonNames("media") var media: MutableList<MediaObject> = mutableListOf(),
  @JsonNames("groups") var groups: List<Group> = listOf(),
  @JsonNames("msidSemantic") var msidSemantic: MsidSemantic? = null,
  @JsonNames("icelite") var icelite: String? = null,
  @JsonNames("extmapAllowMixed") var extmapAllowMixed: String? = null,
) {
  fun write(): String {
    val lines = mutableListOf<String>()
    if (version != null) lines.add("v=" + SdpGrammar.writeVersion(version!!))
    if (origin != null) lines.add("o=" + SdpGrammar.writeOrigin(origin!!))
    if (name != null) lines.add("s=" + SdpGrammar.writeName(name!!))
    if (description != null) lines.add("i=" + SdpGrammar.writeDescription(description!!))
    if (timing != null) lines.add("t=" + SdpGrammar.writeTiming(timing!!))
    if (connection != null) lines.add("c=" + SdpGrammar.writeConnection(connection!!))
    if (icelite != null) lines.add("a=" + SdpGrammar.writeIcelite(icelite!!))
    if (iceUfrag != null) lines.add("a=" + SdpGrammar.writeIceufrag(iceUfrag!!))
    if (icePwd != null) lines.add("a=" + SdpGrammar.writeIcepwd(icePwd!!))
    if (fingerprint != null) lines.add("a=" + SdpGrammar.writeFingerprint(fingerprint!!))
    groups.forEach { item -> lines.add("a=" + SdpGrammar.writeGroups(item)) }
    if (extmapAllowMixed != null) lines.add("a=extmap-allow-mixed")
    if (msidSemantic != null) lines.add("a=" + SdpGrammar.writeMsidsemantic(msidSemantic!!))
    for (m in media) {
      lines.add("m=" + SdpGrammar.writeMline(m.type!!, m.port!!, m.protocol!!, m.payloads!!))
      if (m.description != null) lines.add("i=" + SdpGrammar.writeDescription(m.description!!))
      if (m.connection != null) lines.add("c=" + SdpGrammar.writeConnection(m.connection!!))
      m.bandwidth?.forEach { item -> lines.add("b=" + SdpGrammar.writeBandwidth(item)) }
      m.rtp?.forEach { item -> lines.add("a=" + SdpGrammar.writeRtp(item)) }
      m.fmtp?.forEach { item -> lines.add("a=" + SdpGrammar.writeFmtp(item)) }
      if (m.rtcp != null) lines.add("a=" + SdpGrammar.writeRtcp(m.rtcp!!))
      m.rtcpFbTrrInt?.forEach { item -> lines.add("a=" + SdpGrammar.writeRtcpfbtrrint(item)) }
      m.rtcpFb?.forEach { item -> lines.add("a=" + SdpGrammar.writeRtcpfb(item)) }
      m.ext?.forEach { item -> lines.add("a=" + SdpGrammar.writeExt(item)) }
      if (m.extmapAllowMixed == true) lines.add("a=extmap-allow-mixed")
      m.crypto?.forEach { item -> lines.add("a=" + SdpGrammar.writeCrypto(item)) }
      if (m.setup != null) lines.add("a=" + SdpGrammar.writeSetup(m.setup!!))
      if (m.mid != null) lines.add("a=" + SdpGrammar.writeMid(m.mid!!))
      if (m.msid != null) lines.add("a=" + SdpGrammar.writeMsid(m.msid!!))
      if (m.ptime != null) lines.add("a=" + SdpGrammar.writePtime(m.ptime!!))
      if (m.maxptime != null) lines.add("a=" + SdpGrammar.writeMaxptime(m.maxptime!!))
      if (m.direction != null) lines.add("a=" + SdpGrammar.writeDirection(m.direction!!))
      if (m.iceUfrag != null) lines.add("a=" + SdpGrammar.writeIceufrag(m.iceUfrag!!))
      if (m.icePwd != null) lines.add("a=" + SdpGrammar.writeIcepwd(m.icePwd!!))
      if (m.fingerprint != null) lines.add("a=" + SdpGrammar.writeFingerprint(m.fingerprint!!))
      m.candidates?.forEach { item -> lines.add("a=" + SdpGrammar.writeCandidates(item)) }
      if (m.endOfCandidates != null)
        lines.add("a=" + SdpGrammar.writeEndofcandidates(m.endOfCandidates!!))
      if (m.iceOptions != null) lines.add("a=" + SdpGrammar.writeIceoptions(m.iceOptions!!))
      m.ssrcs?.forEach { item -> lines.add("a=" + SdpGrammar.writeSsrcs(item)) }
      m.ssrcGroups?.forEach { item -> lines.add("a=" + SdpGrammar.writeSsrcgroups(item)) }
      if (m.rtcpMux != null) lines.add("a=" + SdpGrammar.writeRtcpmux(m.rtcpMux!!))
      if (m.rtcpRsize != null) lines.add("a=" + SdpGrammar.writeRtcprsize(m.rtcpRsize!!))
      if (m.sctpmap != null) lines.add("a=" + SdpGrammar.writeSctpmap(m.sctpmap!!))
      if (m.xGoogleFlag != null) lines.add("a=" + SdpGrammar.writeXgoogleflag(m.xGoogleFlag!!))
      m.rids?.forEach { item -> lines.add("a=" + SdpGrammar.writeRids(item)) }
      m.imageattrs?.forEach { item -> lines.add("a=" + SdpGrammar.writeImageattrs(item)) }
      if (m.simulcast != null) lines.add("a=" + SdpGrammar.writeSimulcast(m.simulcast!!))
      if (m.framerate != null) lines.add("a=" + SdpGrammar.writeFramerate(m.framerate!!))
      if (m.sourceFilter != null) lines.add("a=" + SdpGrammar.writeSourcefilter(m.sourceFilter!!))
      if (m.bundleOnly != null) lines.add("a=" + SdpGrammar.writeBundleonly(m.bundleOnly!!))
      if (m.label != null) lines.add("a=" + SdpGrammar.writeLabel(m.label!!))
      if (m.sctpPort != null) lines.add("a=" + SdpGrammar.writeSctpport(m.sctpPort!!))
      if (m.maxMessageSize != null)
        lines.add("a=" + SdpGrammar.writeMaxmessagesize(m.maxMessageSize!!))
    }

    return lines.joinToString("\r\n", postfix = "\r\n")
  }
}
