package io.dyte.media.utils

import io.dyte.media.utils.sdp.toIntIfInt
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.int

class ProfileLevelId(val profile: Int, val level: Int)

class ProfilePattern(val profile_idc: Int, val profile_iop: BitPattern, val profile: Int)

class BitPattern(var str: String) {
  private var _mask = H264Utils.byteMaskString("x", str)
  private var _maskedValue = H264Utils.byteMaskString("1", str)

  fun isMatch(value: Int) = _maskedValue == (value and _mask)
}

val profilePatterns: List<ProfilePattern> =
  listOf(
    ProfilePattern(
      profile_idc = 0x42,
      profile_iop = BitPattern("x1xx0000"),
      H264Utils.ProfileConstrainedBaseline
    ),
    ProfilePattern(
      profile_idc = 0x4D,
      profile_iop = BitPattern("1xxx0000"),
      H264Utils.ProfileConstrainedBaseline
    ),
    ProfilePattern(
      profile_idc = 0x58,
      profile_iop = BitPattern("11xx0000"),
      H264Utils.ProfileConstrainedBaseline
    ),
    ProfilePattern(
      profile_idc = 0x42,
      profile_iop = BitPattern("x0xx0000"),
      H264Utils.ProfileBaseline
    ),
    ProfilePattern(
      profile_idc = 0x58,
      profile_iop = BitPattern("10xx0000"),
      H264Utils.ProfileBaseline
    ),
    ProfilePattern(profile_idc = 0x4D, profile_iop = BitPattern("0x0x0000"), H264Utils.ProfileMain),
    ProfilePattern(profile_idc = 0x64, profile_iop = BitPattern("00000000"), H264Utils.ProfileHigh),
    ProfilePattern(
      profile_idc = 0x64,
      profile_iop = BitPattern("00001100"),
      H264Utils.ProfileConstrainedHigh
    )
  )

class H264Utils {
  companion object {
    val ConstraintSet3Flag = 0x10

    val Level1_b: Int = 0
    val Level1: Int = 10
    val Level1_1: Int = 11
    val Level1_2: Int = 12
    val Level1_3: Int = 13
    val Level2: Int = 20
    val Level2_1: Int = 21
    val Level2_2: Int = 22
    val Level3: Int = 30
    val Level3_1: Int = 31
    val Level3_2: Int = 32
    val Level4: Int = 40
    val Level4_1: Int = 41
    val Level4_2: Int = 42
    val Level5: Int = 50
    val Level5_1: Int = 51
    val Level5_2: Int = 52

    val ProfileConstrainedBaseline: Int = 1
    val ProfileBaseline: Int = 2
    val ProfileMain: Int = 3
    val ProfileConstrainedHigh: Int = 4
    val ProfileHigh: Int = 5

    fun byteMaskString(c: String, str: String): Int {
      val str0: Int = if (str[0].toString() == c) 1 else 0
      val str1: Int = if (str[1].toString() == c) 1 else 0
      val str2: Int = if (str[2].toString() == c) 1 else 0
      val str3: Int = if (str[3].toString() == c) 1 else 0
      val str4: Int = if (str[4].toString() == c) 1 else 0
      val str5: Int = if (str[5].toString() == c) 1 else 0
      val str6: Int = if (str[6].toString() == c) 1 else 0
      val str7: Int = if (str[7].toString() == c) 1 else 0

      return (str0 shl 7) or
        (str1 shl 6) or
        (str2 shl 5) or
        (str3 shl 4) or
        (str4 shl 3) or
        (str5 shl 2) or
        (str6 shl 1) or
        (str7 shl 0)
    }

    fun defaultProfileLevelId() =
      ProfileLevelId(level = Level3_1, profile = ProfileConstrainedBaseline)

    fun parseProfileLevelId(str: String?): ProfileLevelId? {
      if (str == null || str.length != 6) return null

      val profile_level_id_numeric: Int = str.toInt(16)

      if (profile_level_id_numeric == 0) return null

      val level_idc: Int = profile_level_id_numeric and 0xFF
      val profile_iop: Int = profile_level_id_numeric shr 8 and 0xFF
      val profile_idc: Int = profile_level_id_numeric shr 16 and 0xFF

      var level: Int

      when (level_idc) {
        Level1_1 -> level = if (profile_iop and ConstraintSet3Flag != 0) Level1_b else Level1_1
        Level1,
        Level1_2,
        Level1_3,
        Level2,
        Level2_1,
        Level2_2,
        Level3,
        Level3_1,
        Level3_2,
        Level4,
        Level4_1,
        Level4_2,
        Level5,
        Level5_1,
        Level5_2 -> level = level_idc
        else -> {
          println(
            "MEDIASOUP: H264Utils: parseProfileLevelId() | unrecognized level_idc:$level_idc'"
          )
          return null
        }
      }

      for (pattern: ProfilePattern in profilePatterns) {
        if (profile_idc == pattern.profile_idc && pattern.profile_iop.isMatch(profile_iop))
          return ProfileLevelId(profile = pattern.profile, level = level)
      }

      println(
        "MEDIASOUP: H264Utils: parseProfileLevelId() | unrecognized profile_idc/profile_iop combination"
      )

      return null
    }

    fun isLevelAsymmetryAllowedJSON(params: Map<String, JsonPrimitive?> = emptyMap()): Boolean {
      var level_asymmetry_allowed = params["level-asymmetry-allowed"]?.content
      var level_asymmetry_allowed_int = toIntIfInt(level_asymmetry_allowed)

      return (level_asymmetry_allowed_int == 1 || level_asymmetry_allowed == "1")
    }

    fun isLevelAsymmetryAllowed(params: Map<String, String?> = emptyMap()): Boolean {
      var level_asymmetry_allowed = params["level-asymmetry-allowed"]

      return level_asymmetry_allowed == "1"
    }

    fun parseSdpProfileLevelIdJSON(
      params: Map<String, JsonPrimitive?> = emptyMap()
    ): ProfileLevelId? {
      val profile_level_id = params["profile-level-id"]?.int

      return if (profile_level_id == null) defaultProfileLevelId()
      else parseProfileLevelId(profile_level_id.toString())
    }

    fun parseSdpProfileLevelId(params: Map<String, String?> = emptyMap()): ProfileLevelId? {
      val profile_level_id = params["profile-level-id"]

      return if (profile_level_id == null) defaultProfileLevelId()
      else parseProfileLevelId(profile_level_id.toString())
    }

    fun isSameProfileJSON(
      params1: Map<String, JsonPrimitive?>,
      params2: Map<String, JsonPrimitive?>
    ): Boolean {
      val profile_level_id_1: ProfileLevelId? = parseSdpProfileLevelIdJSON(params1)
      val profile_level_id_2: ProfileLevelId? = parseSdpProfileLevelIdJSON(params2)

      return profile_level_id_1 != null &&
        profile_level_id_2 != null &&
        profile_level_id_1.profile == profile_level_id_2.profile
    }

    fun isSameProfile(params1: Map<String, String?>, params2: Map<String, String?>): Boolean {
      val profile_level_id_1: ProfileLevelId? = parseSdpProfileLevelId(params1)
      val profile_level_id_2: ProfileLevelId? = parseSdpProfileLevelId(params2)

      return profile_level_id_1 != null &&
        profile_level_id_2 != null &&
        profile_level_id_1.profile == profile_level_id_2.profile
    }

    fun isSameProfileJSON2(
      params1: Map<String, String?>,
      params2: Map<String, JsonPrimitive?>
    ): Boolean {
      val profile_level_id_1: ProfileLevelId? = parseSdpProfileLevelId(params1)
      val profile_level_id_2: ProfileLevelId? = parseSdpProfileLevelIdJSON(params2)

      return profile_level_id_1 != null &&
        profile_level_id_2 != null &&
        profile_level_id_1.profile == profile_level_id_2.profile
    }

    fun isLessLevel(a: Int, b: Int): Boolean {
      if (a == Level1_b) return b != Level1 && b != Level1_b

      if (b == Level1_b) return a != Level1

      return a < b
    }

    fun minLevel(a: Int, b: Int): Int {
      return if (isLessLevel(a, b)) a else b
    }

    fun profileLevelIdtoString(profile_level_id: ProfileLevelId): String? {
      if (profile_level_id.level == Level1_b) {
        when (profile_level_id.profile) {
          ProfileConstrainedBaseline -> return "42f00b"
          ProfileBaseline -> return "42100b"
          ProfileMain -> return "4d100b"
          else -> {
            println(
              "MEDIASOUP: H264Utils: profileLevelidToString() | Level 1_b not is allowed for profile:${profile_level_id.profile}"
            )

            return null
          }
        }
      }

      var profile_idc_iop_string: String

      when (profile_level_id.profile) {
        ProfileConstrainedBaseline -> profile_idc_iop_string = "42e0"
        ProfileBaseline -> profile_idc_iop_string = "4200"
        ProfileMain -> profile_idc_iop_string = "4d00"
        ProfileConstrainedHigh -> profile_idc_iop_string = "640c"
        ProfileHigh -> profile_idc_iop_string = "6400"
        else -> {
          println(
            "MEDIASOUP: H264Utils: profileLevelIdToString() | unrecognized profile:${profile_level_id.profile}"
          )

          return null
        }
      }

      var levelStr = profile_level_id.level.toString(16)

      if (levelStr.length == 1) levelStr = "0$levelStr"

      return "$profile_idc_iop_string$levelStr"
    }

    fun generateProfileLevelIdForAnswerJSON(
      local_supported_params: Map<String, JsonPrimitive?> = emptyMap(),
      remote_offered_params: Map<String, JsonPrimitive?> = emptyMap()
    ): String? {
      if (
        local_supported_params["profile-level-id"] == null &&
          remote_offered_params["profile-level-id"] == null
      ) {
        println(
          "MEDIASOUP: H264Utils: generateProfileLevelIdForAnswer() | no profile-level-id in local and remote params"
        )

        return null
      }

      val local_profile_level_id = parseSdpProfileLevelIdJSON(local_supported_params)
      val remote_profile_level_id = parseSdpProfileLevelIdJSON(remote_offered_params)

      if (local_profile_level_id == null) throw Exception("invalid local_profile_level_id")
      if (remote_profile_level_id == null) throw Exception("invalid remote_profile_level_id")

      if (local_profile_level_id.profile != remote_profile_level_id.profile)
        throw Exception("H264 Profile mismatch")

      val level_asymmetry_allowed =
        (isLevelAsymmetryAllowedJSON(local_supported_params) &&
          isLevelAsymmetryAllowedJSON(remote_offered_params))

      val local_level = local_profile_level_id.level
      val remote_level = remote_profile_level_id.level
      val min_level = minLevel(local_level, remote_level)

      val answer_level = if (level_asymmetry_allowed) local_level else min_level

      println(
        "MEDIASOUP: H264Utils: generateProfileLevelIdForAnswer() | result: [profile:${local_profile_level_id.profile}, level:$answer_level"
      )

      return profileLevelIdtoString(
        ProfileLevelId(profile = local_profile_level_id.profile, level = answer_level)
      )
    }

    fun generateProfileLevelIdForAnswer(
      local_supported_params: Map<String, String?> = emptyMap(),
      remote_offered_params: Map<String, String?> = emptyMap()
    ): String? {
      if (
        local_supported_params["profile-level-id"] == null &&
          remote_offered_params["profile-level-id"] == null
      ) {
        println(
          "MEDIASOUP: H264Utils: generateProfileLevelIdForAnswer() | no profile-level-id in local and remote params"
        )

        return null
      }

      val local_profile_level_id = parseSdpProfileLevelId(local_supported_params)
      val remote_profile_level_id = parseSdpProfileLevelId(remote_offered_params)

      if (local_profile_level_id == null) throw Exception("invalid local_profile_level_id")
      if (remote_profile_level_id == null) throw Exception("invalid remote_profile_level_id")

      if (local_profile_level_id.profile != remote_profile_level_id.profile)
        throw Exception("H264 Profile mismatch")

      val level_asymmetry_allowed =
        (isLevelAsymmetryAllowed(local_supported_params) &&
          isLevelAsymmetryAllowed(remote_offered_params))

      val local_level = local_profile_level_id.level
      val remote_level = remote_profile_level_id.level
      val min_level = minLevel(local_level, remote_level)

      val answer_level = if (level_asymmetry_allowed) local_level else min_level

      println(
        "MEDIASOUP: H264Utils: generateProfileLevelIdForAnswer() | result: [profile:${local_profile_level_id.profile}, level:$answer_level"
      )

      return profileLevelIdtoString(
        ProfileLevelId(profile = local_profile_level_id.profile, level = answer_level)
      )
    }

    fun generateProfileLevelIdForAnswerJSON2(
      local_supported_params: Map<String, String?> = emptyMap(),
      remote_offered_params: Map<String, JsonPrimitive?> = emptyMap()
    ): String? {
      if (
        local_supported_params["profile-level-id"] == null &&
          remote_offered_params["profile-level-id"] == null
      ) {
        println(
          "MEDIASOUP: H264Utils: generateProfileLevelIdForAnswer() | no profile-level-id in local and remote params"
        )

        return null
      }

      val local_profile_level_id = parseSdpProfileLevelId(local_supported_params)
      val remote_profile_level_id = parseSdpProfileLevelIdJSON(remote_offered_params)

      if (local_profile_level_id == null) throw Exception("invalid local_profile_level_id")
      if (remote_profile_level_id == null) throw Exception("invalid remote_profile_level_id")

      if (local_profile_level_id.profile != remote_profile_level_id.profile)
        throw Exception("H264 Profile mismatch")

      val level_asymmetry_allowed =
        (isLevelAsymmetryAllowed(local_supported_params) &&
          isLevelAsymmetryAllowedJSON(remote_offered_params))

      val local_level = local_profile_level_id.level
      val remote_level = remote_profile_level_id.level
      val min_level = minLevel(local_level, remote_level)

      val answer_level = if (level_asymmetry_allowed) local_level else min_level

      println(
        "MEDIASOUP: H264Utils: generateProfileLevelIdForAnswer() | result: [profile:${local_profile_level_id.profile}, level:$answer_level"
      )

      return profileLevelIdtoString(
        ProfileLevelId(profile = local_profile_level_id.profile, level = answer_level)
      )
    }
  }
}
