package io.dyte.media.hive.handlers

import io.dyte.media.handlers.sdp.Fmtp
import io.dyte.media.hive.HiveConnectionState
import io.dyte.media.hive.HiveEmitData
import io.dyte.media.utils.IMediaClientLogger
import io.dyte.media.utils.sdp.SDPUtils
import io.dyte.webrtc.*
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach

class HiveUnifiedPlan(val coroutineScope: CoroutineScope, private val logger: IMediaClientLogger?) :
  HiveHandlerInterface() {
  /** Handler Direction */
  private lateinit var _direction: RtpTransceiverDirection

  /** RTCPeerConnection instance */
  private lateinit var _pc: PeerConnection

  /** Got transport local and remote parameters */
  private var _transportReady = false

  val observer = MutableSharedFlow<HiveEmitData>()

  companion object {
    fun createFactory(logger: IMediaClientLogger? = null): HiveHandlerFactory = {
      HiveUnifiedPlan(coroutineScope = CoroutineScope(Dispatchers.Main), logger = logger)
    }
  }

  override fun getName() = "UnifiedPlan"

  override fun getPc() = this._pc

  override fun close() {
    logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: close()")

    if (this::_pc.isInitialized) {
      try {
        this._pc.close()
      } catch (e: Error) {
        logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: pc.close() $e")
      }
    }
  }

  override suspend fun init(options: HiveHandlerRunOptions) {
    logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: init()")

    this._direction = options.direction

    @Suppress("UNCHECKED_CAST")
    this._pc =
      PeerConnection(
        RtcConfiguration(
          iceServers =
            options.additionalSettings?.get("iceServers") as? List<IceServer>
              ?: options.iceServers
              ?: emptyList(),
          iceTransportPolicy =
            options.additionalSettings?.get("iceTransportPolicy") as? IceTransportPolicy
              ?: options.iceTransportPolicy
              ?: IceTransportPolicy.All,
          bundlePolicy =
            options.additionalSettings?.get("bundlePolicy") as? BundlePolicy
              ?: BundlePolicy.MaxBundle,
          rtcpMuxPolicy =
            options.additionalSettings?.get("rtcpMuxPolicy") as? RtcpMuxPolicy
              ?: RtcpMuxPolicy.Require,
          //      sdpSemantics = "unified-plan"
          //      proprietaryConstraints
        )
      )

    _pc.onTrack
      .onEach {
        logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: Received track event")
        options.onTrackHandler(it)
      }
      .launchIn(coroutineScope)

    _addEventListeners()
  }

  override suspend fun connect(): HiveGenericHandlerResult {
    /**
     * Since we have enabled bundlePolicy to MaxBundle, we need to have max-bundle attribute in SDP
     * and to generate that we are using a hacky fix by creating a temp data channel
     */
    val dc = this._pc.createDataChannel("dyte")!!

    val offer = this._pc.createOffer(OfferAnswerOptions())

    this._pc.setLocalDescription(offer)

    val callback: suspend (SessionDescription) -> Unit = { answer ->
      this._pc.setRemoteDescription(answer)
      dc.close()
    }

    return HiveGenericHandlerResult(offerSdp = offer, callback = callback)
  }

  override suspend fun updateIceServers(iceServers: List<IceServer>) {
    logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: updateIceServers()")

    //    val configuration = this._pc.getConfiguration()
    //    configuration.iceServers = iceServers
    //    this._pc.setConfiguration(configuration)

    // TODO: Modify to retain other configuration parameters or implement getConfiguration
    this._pc.setConfiguration(RtcConfiguration(iceServers = iceServers))
  }

  override suspend fun restartIce(): HiveGenericHandlerResult {
    logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: restartIce()")

    val offer = this._pc.createOffer(OfferAnswerOptions(iceRestart = true))

    val callback: suspend (SessionDescription) -> Unit = { answer ->
      this._pc.setRemoteDescription(answer)
    }

    return HiveGenericHandlerResult(offerSdp = offer, callback = callback)
  }

  override suspend fun getTransportStats(): RtcStatsReport? = this._pc.getStats()

  override suspend fun send(options: HiveHandlerSendOptions): HiveHandlerSendResult {
    logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: Send for ${options.track.kind}")

    this._assertSendDirection()

    val trans =
      this._pc.addTransceiver(
        track = options.track,
        kind = options.track.kind,
        init =
          CommonRtpTransceiverInit(
            direction = RtpTransceiverDirection.SendOnly,
            streams = listOf(options.stream),
            sendEncodings = options.encodings ?: emptyList(),
          ),
      )!!

    // Note(anunaym14): Similar to Firefox, WebRTC-KMP doesn't support
    // RtpTransceiver.setCodecPreferences()
    // So, let's enforce SFU to only use VP8 in server side.

    val offer = this._pc.createOffer(OfferAnswerOptions())

    // if (!this._transportReady) throw Error("WebRTC Transport not connected")

    this._pc.setLocalDescription(offer)

    var updatedOffer: SessionDescription? = null

    /** mangle offer with new fmtp parameters and nack */
    if (options.codecOptions != null && options.codecOptions.name == "opus") {
      var updatedOfferSDP =
        offer.sdp.replace("minptime=10;useinbandfec=1", "minptime=10;useinbandfec=1;usedtx=1")

      updatedOfferSDP += "a=rtcp-fb:111 nack\r\n"

      updatedOffer = SessionDescription(type = offer.type, sdp = updatedOfferSDP)
    }

    var ssrc: Long? = null
    val parsedOffer = SDPUtils.parse(updatedOffer?.sdp ?: offer.sdp)

    parsedOffer.media.map {
      if (it.type == options.track.kind.toString().lowercase()) {
        ssrc = it.ssrcs?.first()?.id
        val updatedMedia = it

        val filteredRtp = it.rtp?.firstOrNull { rtp -> rtp.codec == options.codecOptions?.name }

        if (filteredRtp != null) {
          val filteredRtcpFb = it.rtcpFb?.filter { rtcpFb -> rtcpFb.payload == filteredRtp.payload }

          val filteredFmtp =
            it.fmtp?.filter { fmtp -> fmtp.payload == filteredRtp.payload }?.toMutableList()
              ?: mutableListOf()

          if (options.codecOptions?.parameters != null) {
            options.codecOptions.parameters.forEach { param ->
              if (filteredFmtp.none { it.config == param }) {
                filteredFmtp.add(Fmtp(payload = filteredRtp.payload, config = param))
              }
            }
          }

          updatedMedia.rtp = listOf(filteredRtp).toMutableList()
          updatedMedia.rtcpFb = filteredRtcpFb?.toMutableList()
          updatedMedia.fmtp = filteredFmtp?.toMutableList()
          updatedMedia.payloads = filteredRtp.payload.toString()
        }

        updatedMedia
      } else {
        it
      }
    }

    val finalOffer = SessionDescription(type = offer.type, sdp = SDPUtils.write(parsedOffer))

    val setAnswer: suspend (SessionDescription) -> String = { answer ->
      this._pc.setRemoteDescription(answer)

      // Store in the map
      this.mapMidTransceiver[trans.mid] = trans

      trans.mid
    }

    return HiveHandlerSendResult(offerSdp = finalOffer, callback = setAnswer, ssrc = ssrc)
  }

  override suspend fun stopSending(localId: String): HiveGenericHandlerResult {
    this._assertSendDirection()

    logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: stopSending for localId: $localId")

    val transceiver =
      this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    transceiver.sender.replaceTrack(null)
    this._pc.removeTrack(transceiver.sender)

    transceiver.direction = RtpTransceiverDirection.Inactive

    val offer = this._pc.createOffer(OfferAnswerOptions())

    this._pc.setLocalDescription(offer)

    val callback: suspend (SessionDescription) -> Unit = { answer ->
      this._pc.setRemoteDescription(answer)

      this.mapMidTransceiver.remove(localId)
    }

    return HiveGenericHandlerResult(offerSdp = offer, callback = callback)
  }

  override suspend fun replaceTrack(localId: String, track: MediaStreamTrack?) {
    this._assertSendDirection()

    if (track != null) {
      logger?.traceLog(
        "DyteMediaClient: HiveUnifiedPlan: replaceTrack() with localId=$localId, trackId=${track.id}"
      )
    } else {
      logger?.traceLog(
        "DyteMediaClient: HiveUnifiedPlan: replaceTrack() localId=$localId with no track"
      )
      return
    }

    val transceiver =
      this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    transceiver.sender.replaceTrack(track)
  }

  override suspend fun setMaxSpatialLayer(localId: String, spatialLayer: Long) {
    this._assertSendDirection()

    logger?.traceLog(
      "DyteMediaClient: HiveUnifiedPlan: setMaxSpatialLayer() with localId=$localId spatialLayer=$spatialLayer"
    )

    val transceiver =
      this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    val parameters = transceiver.sender.parameters

    parameters.encodings.forEachIndexed { idx, encoding ->
      // TODO: Verify if changing active to var changes it internally
      encoding.active = idx <= spatialLayer
    }

    transceiver.sender.parameters = parameters
  }

  override suspend fun setRtpEncodingParameters(localId: String, params: Any) {
    this._assertSendDirection()

    logger?.traceLog(
      "DyteMediaClient: HiveUnifiedPlan: setRtpEncodingParameters() with localId=$localId params=$params"
    )

    val transceiver =
      this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    val parameters = transceiver.sender.parameters

    // TODO: This is required for simulcast which Hive doesn't support right now, fix for future
    // Also, figure out how to use common encodings here
    // parameters.encodings.add(params as RtpEncodingParameters)

    transceiver.sender.parameters = parameters
  }

  override fun getSenderStats(localId: String): List<RtcStatsReport> {
    this._assertSendDirection()

    val transceiver =
      this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    // TODO: Implement sender getStats()
    //    return transceiver.sender.getStats()
    return emptyList()
  }

  override suspend fun stopReceiving(localId: String): HiveGenericHandlerResult {
    this._assertRecvDirection()

    logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: stopReceiving() with localId=$localId")

    throw Error("Method not implemented")
  }

  override suspend fun pauseReceiving(localId: String): HiveGenericHandlerResult {
    this._assertRecvDirection()

    logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: pauseReceiving() with localId=$localId")

    throw Error("Method not implemented")
  }

  override suspend fun resumeReceiving(localId: String): HiveGenericHandlerResult {
    this._assertRecvDirection()

    logger?.traceLog("DyteMediaClient: UnifiedPlan: resumeReceiving() with localId=$localId")

    throw Error("Method not implemented")
  }

  override suspend fun getReceiverStats(localId: String): List<RtcStatsReport> {
    this._assertRecvDirection()

    val transceiver =
      this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    // TODO: Implement receiver getStats()
    //    return transceiver.receiver.getStats()
    return emptyList()
  }

  private fun _assertSendDirection() {
    if (this._direction != RtpTransceiverDirection.SendOnly) {
      throw Error("Method can just be called for handlers with send direction")
    }
  }

  private fun _assertRecvDirection() {
    if (this._direction != RtpTransceiverDirection.RecvOnly) {
      throw Error("Method can just be called for handlers with recv direction")
    }
  }

  private suspend fun _generateOffer(): SessionDescription {
    val offer = this._pc.createOffer(OfferAnswerOptions())

    this._pc.setLocalDescription(offer)

    return offer
  }

  private suspend fun _setAnswer(answer: SessionDescription) {
    this._pc.setRemoteDescription(answer)
  }

  private suspend fun _addEventListeners() {
    var job: Job? = null

    this._pc.onIceConnectionStateChange
      .onEach {
        logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: IceConnectionStateChange: ${it.name}")
        when (it) {
          IceConnectionState.Checking ->
            this.observer.emit(
              HiveEmitData(
                eventName = "@connectionstatechange",
                data = HiveConnectionState.Connecting,
              )
            )
          IceConnectionState.Connected ->
            this.observer.emit(
              HiveEmitData(
                eventName = "@connectionstatechange",
                data = HiveConnectionState.Connected,
              )
            )
          IceConnectionState.Completed ->
            this.observer.emit(
              HiveEmitData(
                eventName = "@connectionstatechange",
                data = HiveConnectionState.Connected,
              )
            )
          IceConnectionState.Failed ->
            this.observer.emit(
              HiveEmitData(eventName = "@connectionstatechange", data = HiveConnectionState.Failed)
            )
          IceConnectionState.Disconnected ->
            this.observer.emit(
              HiveEmitData(
                eventName = "@connectionstatechange",
                data = HiveConnectionState.Disconnected,
              )
            )
          IceConnectionState.Closed ->
            this.observer.emit(
              HiveEmitData(eventName = "@connectionstatechange", data = HiveConnectionState.Closed)
            )
          else -> logger?.traceLog("Unknown IceConnectionState")
        }
      }
      .launchIn(coroutineScope)

    this._pc.onIceCandidate
      .onEach {
        logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: IceCandidate: ${it.candidate}")
        this.observer.emit(HiveEmitData(eventName = "@icecandidate", data = it))
      }
      .launchIn(coroutineScope)

    this._pc.onNegotiationNeeded
      .onEach {
        logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: NegotiationNeeded")
        this.observer.emit(HiveEmitData("@negotiationneeded"))
      }
      .launchIn(coroutineScope)

    this._pc.onIceGatheringState
      .onEach {
        logger?.traceLog("DyteMediaClient: HiveUnifiedPlan: IceGatheringState: ${it.name}")
        when (it) {
          IceGatheringState.Gathering -> {
            this.observer.emit(
              HiveEmitData(
                eventName = "@icegatheringstatechange",
                data = IceGatheringState.Gathering,
              )
            )
          }
          IceGatheringState.Complete -> {
            this.observer.emit(
              HiveEmitData(
                eventName = "@icegatheringstatechange",
                data = IceGatheringState.Complete,
              )
            )
          }
          else -> {}
        }
      }
      .launchIn(coroutineScope)

    // this._pc.onIceCandidateError.collect {}

    this._pc.onDataChannel
      .onEach { channel ->
        channel.onOpen
          .onEach {
            logger?.traceLog(
              "DyteMediaClient: HiveUnifiedPlan: Data channel open: ${channel.label}"
            )
          }
          .launchIn(coroutineScope)

        channel.onClose
          .onEach {
            logger?.traceLog(
              "DyteMediaClient: HiveUnifiedPlan: Data channel closed: ${channel.label}"
            )
          }
          .launchIn(coroutineScope)

        channel.onError
          .onEach { error ->
            logger?.traceLog(
              "DyteMediaClient: HiveUnifiedPlan: Data channel error: $error for ${channel.label}"
            )
          }
          .launchIn(coroutineScope)

        channel.onMessage
          .onEach { msg ->
            logger?.traceLog(
              "DyteMediaClient: HiveUnifiedPlan: Data channel message: $msg for ${channel.label}"
            )

            this.observer.emit(
              HiveEmitData(
                eventName = "datachannel",
                data = mapOf("channel" to channel, "message" to msg.decodeToString()),
              )
            )
          }
          .launchIn(coroutineScope)
      }
      .launchIn(coroutineScope)
  }
}
