package io.dyte.media

import io.dyte.media.utils.IMediaClientLogger
import io.dyte.media.utils.LocalRtpParameters
import io.dyte.media.utils.RTCRtpMediaType
import io.dyte.webrtc.*
import kotlinx.coroutines.flow.*

class ConsumerOptions(
  val id: String,
  val producerID: String,
  val kind: RTCRtpMediaType,
  val rtpParameters: RtpParameters,
)

typealias ConsumerOnTrackEnded = () -> Unit

data class Consumer(
  val id: String,
  val localID: String,
  val producerID: String,
  var closed: Boolean = false,
  var rtpReceiver: RtpReceiver?,
  var track: MediaStreamTrack,
  val rtpParameters: LocalRtpParameters,
  val appData: Map<String, Any>,
  val stream: MediaStream,
  var peerID: String?,
  var paused: Boolean = !track.enabled,
  private val logger: IMediaClientLogger,
) {
  val observer = MutableSharedFlow<EmitData>()
  val kind: MediaStreamTrackKind = track.kind

  suspend fun close() {
    if (closed) return

    logger.traceLog("MEDIASOUP: Consumer: close()")
    closed = true
    destroyTrack()
    observer.emit(EmitData("@close"))
  }

  suspend fun transportClosed() {
    if (closed) return

    logger.traceLog("MEDIASOUP: Consumer: transportClosed()")
    closed = true
    destroyTrack()
    observer.emit(EmitData("@close"))
  }

  suspend fun getStats(): Any {
    if (closed) throw Exception("")

    return observer.emit(EmitData("@getstats"))
  }

  suspend fun pause(): Unit {
    logger.traceLog("MEDIASOUP: Consumer: pause()")

    if (closed) {
      logger.traceError("Consumer: pause() | Consumer closed.")
      return
    }

    paused = true
    track.enabled = false
    observer.emit(EmitData("@pause"))
  }

  suspend fun resume(): Unit {
    logger.traceLog("MEDIASOUP: Consumer: resume()")

    if (closed) {
      logger.traceError("Consumer: resume() | Consumer closed.")
      return
    }

    paused = false
    track.enabled = true
    observer.emit(EmitData("@resume"))
  }

  private suspend fun onTrackEnded(): Unit {
    logger.traceLog("MEDIASOUP: Consumer: track ended event")
    observer.emit(EmitData("@trackended"))
  }

  private suspend fun handleTrack(): Unit {
    track.onEnded.collectLatest { onTrackEnded() }
  }

  private fun destroyTrack() =
    try {
      // track.onEnded = null
      track.stop()
    } catch (e: Exception) {}

  override fun equals(other: Any?): Boolean {
    if (this === other) return true

    return (other is Consumer &&
      other.id == id &&
      other.localID == localID &&
      other.producerID == producerID &&
      other.closed == closed &&
      other.rtpReceiver == rtpReceiver &&
      other.track == track &&
      other.rtpParameters == rtpParameters &&
      other.paused == paused &&
      other.appData == appData &&
      other.stream == stream &&
      other.peerID == peerID)
  }

  override fun hashCode(): Int {
    return id.hashCode() xor
      localID.hashCode() xor
      producerID.hashCode() xor
      closed.hashCode() xor
      rtpReceiver.hashCode() xor
      track.hashCode() xor
      rtpParameters.hashCode() xor
      paused.hashCode() xor
      appData.hashCode() xor
      stream.hashCode() xor
      peerID.hashCode()
  }
}
