package io.dyte.media.common

import io.dyte.media.handlers.sdp.RtcpFb
import io.dyte.media.utils.IMediaClientLogger
import io.dyte.media.utils.UUIDUtils
import io.dyte.media.utils.sdp.SDPUtils
import io.dyte.webrtc.*
import kotlinx.coroutines.*
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.receiveAsFlow
import kotlinx.coroutines.flow.takeWhile
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.addJsonObject
import kotlinx.serialization.json.buildJsonArray
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.put

const val REASON_TRANSPORT_CLOSED = "transport closed"
const val REASON_DISCONNECTION_CLEANUP = "disconnection cleanup"

class MediaTransport(
  val sfu: MediaNodeType,
  val options: MediaInternalTransportOptions,
  private val coroutineScope: CoroutineScope,
  private val logger: IMediaClientLogger,
) {
  /** Id */
  private val _id = options.id

  private lateinit var _serverId: String

  /** Closed flag */
  private var _closed = false

  /** Direction */
  private val _direction = options.direction

  /** SCTP max message size if enabled, null otherwise. */
  //  private val _maxSctpMessageSize: Long [Unused]

  /** RTC handler instance */
  //  private val _handler = options.handlerFactory // check
  private var _handler = MediaUnifiedPlan(coroutineScope, logger)

  /** Transport connection state */
  private var _connectionState = SfuMediaConnectionState.New

  /** Producers map */
  private var _producers = mutableMapOf<String, MediaProducer>()

  /** Consumers map */
  private var _consumers = mutableMapOf<String, MediaConsumer>()

  private var _connected = false

  private val _transportConnection = CompletableDeferred<Boolean>()

  @OptIn(ExperimentalCoroutinesApi::class)
  val limitedDispatcher = Dispatchers.Default.limitedParallelism(1)

  val limitedScope = CoroutineScope(limitedDispatcher)

  val observer = MutableSharedFlow<SfuMediaEmitData>()

  val externalObserver = MutableSharedFlow<SfuMediaEmitData>()

  val consumerChannel = Channel<SfuMediaEmitData>(Channel.BUFFERED)

  private var consumerTrackEvents = mutableMapOf<String, consumerTrackEventHandler>()

  private val consumerTrackPool = mutableMapOf<String, ConsumerTrackEvent>()

  private var unknownTracksMap = mutableMapOf<String, TrackEvent>()

  /** App custom data */
  private val _appData: Map<String, Any> = options.appData ?: emptyMap()

  private var dataChannelCache = mutableMapOf<String, Array<DCMessageChunked?>>()

  private var _dataChannels = mutableMapOf<String, DataChannel>()

  private var consumerCounter = 0

  /** Transport Id */
  fun getId() = this._id

  fun getServerId() = this._serverId

  fun getConnected() = this._connected

  fun getIsConnected() = this._transportConnection

  /** Whether the Transport is closed. */
  fun getClosed() = this._closed

  /** Transport direction * */
  fun getDirection() = this._direction

  /** RTC Handler instance */
  fun getHandler() = this._handler

  /** Connection state */
  fun getConnectionState() = this._connectionState

  /** Custom data */
  fun getAppData() = this._appData

  //  fun setAppData() = throw Error("Cannot override appData object")

  fun setServerId(id: String) {
    this._serverId = id
  }

  fun getDataChannels() = _dataChannels

  fun getDataChannel(label: String) = this._dataChannels[label]

  suspend fun init() {
    _handler.init(
      MediaHandlerRunOptions(
        direction = options.direction,
        iceServers = options.iceServers,
        iceTransportPolicy = options.iceTransportPolicy,
        additionalSettings = options.additionalSettings,
        proprietaryConstraints = options.proprietaryConstraints,
        onTrackHandler = ::_onTrack,
      )
    )

    coroutineScope.launch {
      observer.collect {
        when (it.event) {
          is SfuMediaEvents.Connected -> _transportConnection.complete(true)
          is SfuMediaEvents.Disconnected -> _transportConnection.complete(false)
          is SfuMediaEvents.Close -> _transportConnection.complete(false)
          else -> {}
        }
      }
    }

    coroutineScope.launch {
      _handler.observer.collect {
        when (it.event) {
          is UnifiedPlanEvents.ConnectionStateChange -> {
            val connectionState = it.event.state

            if (connectionState != _connectionState) {
              logger.traceLog(
                "DyteMediaClient: ${sfu.name}: ${getDirection()} Transport: Connection state changed to ${connectionState.name}"
              )

              _connectionState = connectionState

              when (connectionState) {
                SfuMediaConnectionState.Connected -> {
                  _connected = true
                  observer.emit(SfuMediaEmitData(SfuMediaEvents.Connected))
                }
                SfuMediaConnectionState.Disconnected -> {
                  _connected = false
                  observer.emit(SfuMediaEmitData(SfuMediaEvents.Disconnected))
                }
                SfuMediaConnectionState.Failed,
                SfuMediaConnectionState.Closed -> {
                  _connected = false
                  observer.emit(SfuMediaEmitData(SfuMediaEvents.Disconnected))
                }
                else -> {}
              }

              if (!_closed)
                observer.emit(
                  SfuMediaEmitData(SfuMediaEvents.ConnectionStateChanged(connectionState))
                )
            }
          }
          is UnifiedPlanEvents.IceCandidate -> {
            if (!_closed) {
              observer.emit(SfuMediaEmitData(SfuMediaEvents.IceCandidate(it.event.candidate)))
            }
          }
          is UnifiedPlanEvents.DataChannelMessage -> {
            @Suppress("UNCHECKED_CAST") val data = it.event.map as Map<String, Any>

            val channel = data["channel"] as DataChannel

            if (!_dataChannels.contains(channel.label)) _dataChannels[channel.label] = channel

            val dcmsgstr = data["message"] as String

            logger.traceLog(
              "DyteMediaClient: MediaTransport - ${sfu.name}: DataChannel message received - $dcmsgstr on ${channel.id} ${channel.label}"
            )

            // Handle errornous data channel messages to prevent crashes like:
            // kotlinx.serialization.json.internal.JsonDecodingException: Expected start of the
            // object '{', but had 'EOF' instead at path: $
            // JSON input:
            // ,�Z4????????????�????????????????????????????????????????????????????????????????`�?�z????��
            var dcmsg =
              try {
                Json.decodeFromString(DCMessageChunked.serializer(), dcmsgstr)
              } catch (e: Exception) {
                logger.traceLog(
                  "DyteMediaClient: MediaTransport - ${sfu.name}: Error parsing datachannel message chunk - $e"
                )
                return@collect
              }

            // The message is received in chunks, so we need to cache it until we have all the
            // chunks.
            // First check if we have a cache entry for this message id. If we don't,
            if (!dataChannelCache.contains(dcmsg.id)) {
              dataChannelCache[dcmsg.id] = Array(dcmsg.count) { null }
            }

            val messageCache = dataChannelCache[dcmsg.id]!!
            // Add the chunk to the cache
            messageCache[dcmsg.chunkIndex] = dcmsg

            // Check if we have all the chunks
            if (
              dataChannelCache[dcmsg.id]?.size == dcmsg.count &&
                dataChannelCache[dcmsg.id]?.none { c -> c == null } == true
            ) {
              // We have all the chunks, so we can reassemble the message
              val chunks = dataChannelCache[dcmsg.id]
              val message =
                chunks?.fold("") { acc, dcMessageChunked -> acc + dcMessageChunked!!.chunk }

              // Delete the cache entry
              dataChannelCache.remove(dcmsg.id)

              // The message itself is a JSON object, so we need to parse it
              try {
                val parsedMessage = Json.decodeFromString(DCMessage.serializer(), message!!)
                observer.emit(
                  SfuMediaEmitData(
                    SfuMediaEvents.DataChannelMessage(
                      mapOf(
                        "channel" to channel, // get label from channel.label
                        "parsedMessage" to parsedMessage,
                      )
                    )
                  )
                )
              } catch (e: Error) {
                logger.traceWarning(
                  "DyteMediaClient: MediaTransport - ${sfu.name}: Error assembling datachannel message chunks - $e"
                )
              }
            }
          }
          else -> {
            /* no-op */
          }
        }
      }
    }
  }

  /** Close the transport */
  suspend fun close() {
    if (this._closed) return

    logger.traceLog("DyteMediaClient: MediaTransport - ${sfu.name}: close()")

    this._connected = false
    this._closed = true

    // Close the handler
    this._handler.close()

    // Close all producers
    val producers = ArrayList(_producers.values)
    producers.forEach { producer -> producer.close(REASON_TRANSPORT_CLOSED) }
    this._producers.clear()

    // Close all consumers
    val consumers = ArrayList(_consumers.values)
    consumers.forEach { consumer -> consumer.close(REASON_TRANSPORT_CLOSED) }
    this._consumers.clear()

    this.consumerTrackPool.clear()
    this.consumerTrackEvents.clear()

    observer.emit(SfuMediaEmitData(SfuMediaEvents.Close()))
  }

  /** Get associated Transport (RTCPeerConnection) stats */
  suspend fun getStats(): RtcStatsReport? {
    if (this._closed) throw IllegalStateException("closed")

    return this._handler.getTransportStats()
  }

  suspend fun connect() {
    logger.traceLog("DyteMediaClient: Connecting ${sfu.name} transport: $this.id")

    try {
      val connectResult = this._handler.connect()

      // TODO: Check how to receive answer
      this.observer.emit(SfuMediaEmitData(SfuMediaEvents.Connect(connectResult.offerSdp)))

      // val externalResult = SfuMediaEmitData("returnConnect")

      val answer =
        externalObserver.first { it.event is SfuMediaEvents.ConnectAnswer }.event
          as SfuMediaEvents.ConnectAnswer

      // call callback on answer
      connectResult.callback.invoke(answer.sdp)

      if (!this.getIsConnected().await()) throw Error("Ice Connection Failed")
    } catch (e: Error) {
      logger.traceLog("DyteMediaClient: MediaTransport - ${sfu.name}: Failed to connect - $e")
    }
  }

  /** Restart ICE connection */
  suspend fun restartIce(): MediaGenericHandlerResult {
    logger.traceLog("DyteMediaClient: MediaTransport - ${sfu.name}: restartIce()")

    if (this._closed) throw IllegalStateException("closed")

    return this._handler.restartIce()
  }

  /** Update ICE servers */
  suspend fun updateIceServers(iceServers: List<IceServer>) {
    logger.traceLog("DyteMediaClient: MediaTransport - ${sfu.name}: updateIceServers()")

    if (this._closed) throw IllegalStateException("closed")

    this._handler.updateIceServers(iceServers)
  }

  private suspend fun _handleProducer(producer: MediaProducer) {
    producer.observer.collect {
      if (it.event is SfuMediaEvents.Close) this._producers.remove(producer.id)
    }
  }

  private val producerMutex = Mutex()
  private val consumerMutex = Mutex()

  suspend fun produce(options: MediaProducerOptions): MediaProducer {
    return producerMutex.withLock { produceInternal(options) }
  }

  /** Create a producer */
  suspend fun produceInternal(options: MediaProducerOptions): MediaProducer {
    if (options.track == null) throw Error("TypeError: Missing Track")
    else if (this._direction != RtpTransceiverDirection.SendOnly)
      throw UnsupportedOperationException("Not a sending transport")
    else if (options.track.readyState is MediaStreamTrackState.Ended)
      throw IllegalStateException("Track ended")

    if (!this.getIsConnected().await()) throw Error("Transport not connected")

    lateinit var producerId: String
    lateinit var localId: String

    // First we generate offer SDP
    val sendResult =
      _handler.send(
        MediaHandlerSendOptions(
          track = options.track,
          encodings = options.encodings ?: emptyList(),
          codecOptions = options.codecOptions,
          screenShare = options.appData?.get("screenShare") as? Boolean ?: false,
          stream = options.stream,
        )
      )

    val appData = mutableMapOf<String, Any>()
    appData["mid"] = sendResult.mid
    options.appData?.let { appData.putAll(it) }

    // Then we send this offer to the server
    observer.emit(
      SfuMediaEmitData(
        SfuMediaEvents.Produce(
          mapOf(
            "offer" to sendResult.offerSdp,
            "kind" to options.track.kind,
            "paused" to
              if (options.disableTrackOnPause != null && options.disableTrackOnPause)
                !options.track.enabled
              else false,
            "appData" to appData,
            "codecOptions" to options.codecOptions,
          )
        )
      )
    )

    val data =
      externalObserver.first { it.event is SfuMediaEvents.ProduceAnswer }.event
        as SfuMediaEvents.ProduceAnswer

    val answer = data.map["answer"] as SessionDescription
    producerId = data.map["producerId"] as String

    // Then we set the answer on remote and get the localId
    localId = sendResult.callback(answer) as String

    val producer =
      MediaProducer(
        MediaInternalProducerOptions(
          id = producerId,
          localId = localId,
          track = options.track,
          stopTracks = options.stopTracks ?: true,
          disableTrackOnPause = options.disableTrackOnPause ?: true,
          zeroRtpOnPause = options.zeroRtpOnPause ?: false,
          appData = options.appData ?: emptyMap(),
          handler = this.getHandler(),
        ),
        coroutineScope = coroutineScope,
        logger = logger,
      )

    logger.traceLog(
      "DyteMediaClient: MediaTransport - ${sfu.name}: ${producer.kind} producer created ${producer.id}"
    )

    this._producers[producerId] = producer

    coroutineScope.launch { _handleProducer(producer) }

    this.observer.emit(SfuMediaEmitData(SfuMediaEvents.NewProducer(producer)))

    return producer
  }

  suspend fun consume(producers: List<PeerProducerMeta>): List<CompletableDeferred<MediaConsumer>> {
    consumerMutex.withLock {
      logger.traceLog(
        "DyteMediaClient: MediaTransport - ${sfu.name}: consume() with producers = $producers"
      )

      if (this._closed) throw IllegalStateException("closed")
      else if (this._direction != RtpTransceiverDirection.RecvOnly)
        throw UnsupportedOperationException("Not a receiving transport")

      if (!this.getIsConnected().await()) throw IllegalStateException("Transport not connected")

      val deferredResults = mutableListOf<CompletableDeferred<MediaConsumer>>()
      when (sfu) {
        MediaNodeType.HIVE,
        MediaNodeType.ROOM_NODE -> {
          val producersMap = mutableMapOf<String, PeerProducerMeta>()
          producers.forEach { producersMap[it.producerId] = it }

          val consumersMap = mutableMapOf<String, HiveConsumerStateObject>()

          createConsumersOverDataChannel(producers)
          val res =
            consumerChannel.receiveAsFlow().first().event as SfuMediaEvents.HiveConsumePeerAnswer

          logger.traceLog(
            "DyteMediaClient: MediaTransport - ${sfu.name}: Consumers created over DC $res"
          )

          res.value.forEach { (consumerId, entry) ->
            producersMap[entry.producerId]?.let { p ->
              consumersMap[entry.producerId] =
                HiveConsumerStateObject(
                  consumerId = consumerId,
                  trackId = entry.trackId,
                  streamId = entry.streamId,
                  screenShare = p.screenShare,
                  paused = p.paused,
                  kind =
                    if (p.kind == "video") MediaStreamTrackKind.Video
                    else MediaStreamTrackKind.Audio,
                  producingTransportId = p.producingTransportId,
                  mimeType = p.mimeType,
                  producingPeerId = p.producingPeerId,
                )
            }
          }

          consumersMap.forEach { (producerId, entry) ->
            val producer = producersMap[producerId]

            if (producer == null) {
              logger.traceError(
                "unknown entry in create consumer response, producerId: $producerId"
              )
              return mutableListOf()
            }

            deferredResults.add(
              this._consumerCreationTask(
                MediaConsumerCreationTaskOptions(
                  consumerId = entry.consumerId,
                  trackId = entry.trackId,
                  streamId = entry.streamId,
                  kind = entry.kind,
                  producerId = producerId,
                  producingPeerId = producer.producingPeerId,
                  paused = entry.paused,
                  screenShare = entry.screenShare,
                  producingTransportId = entry.producingTransportId,
                  appData = mutableMapOf("screenShare" to entry.screenShare),
                  mimeType = entry.mimeType,
                )
              )
            )
          }
        }
        MediaNodeType.CF -> {
          this.observer.emit(SfuMediaEmitData(SfuMediaEvents.ConsumePeer(producers)))

          val data =
            externalObserver.first { it.event is SfuMediaEvents.CfConsumePeerAnswer }.event
              as SfuMediaEvents.CfConsumePeerAnswer

          data.consumersMap.forEach { (producerId, entry) ->
            deferredResults.add(
              this._consumerCreationTask(
                MediaConsumerCreationTaskOptions(
                  consumerId = entry.consumerId,
                  trackId = entry.trackId,
                  streamId = entry.streamId,
                  kind = entry.kind,
                  producerId = producerId,
                  producingPeerId = entry.producingPeerId,
                  paused = entry.paused,
                  screenShare = entry.screenShare,
                  producingTransportId = entry.producingTransportId,
                  appData = mutableMapOf("screenShare" to entry.screenShare),
                  mimeType = "video/vp8",
                )
              )
            )
          }

          // the onTrack callbacks will fire only once we negotiate
          this.observer.emit(SfuMediaEmitData(SfuMediaEvents.Negotiate(sdp = data.sdp)))
        }
      }

      return deferredResults
    }
  }

  private fun _handleConsumer(consumer: MediaConsumer, scope: CoroutineScope) {
    consumer.observer
      .takeWhile { it.event is SfuMediaEvents.Close }
      .onEach {
        this._consumers.remove(consumer.getId())
        this._handler.mapMidTransceiver.remove(consumer.getLocalId()) // transceiver.mid
      }
      .launchIn(scope)
  }

  private suspend fun _consumerCreationTask(
    options: MediaConsumerCreationTaskOptions
  ): CompletableDeferred<MediaConsumer> {
    val key = "${options.streamId}:${options.kind}"
    val exception = HiveConsumerCreationTaskException(options)

    val deferredConsumer = CompletableDeferred<MediaConsumer>()

    val timeoutTimer =
      coroutineScope.launch {
        delay(5000)
        if (isActive) {
          consumerTrackEvents.remove(key)
          exception.isTimedOut = true
          deferredConsumer.completeExceptionally(exception)
        }
      }

    val consumeHandler: suspend (ConsumerTrackEvent) -> Unit = { event ->
      try {
        if (event.track?.readyState is MediaStreamTrackState.Ended) {
          timeoutTimer.cancel()
          deferredConsumer.completeExceptionally(exception)
        } else {
          val consumerLocalId = event.mid

          this._handler.mapMidTransceiver[consumerLocalId] = event.transceiver!!
          this._handler.mapMidReceiver[consumerLocalId] = event.receiver
          event.track!!.enabled = true

          val consumer =
            MediaConsumer(
              MediaInternalConsumerOptions(
                id = options.consumerId,
                localId = consumerLocalId,
                track = event.track,
                kind = event.track.kind,
                paused = options.paused,
                producerId = options.producerId,
                producingPeerId = options.producingPeerId,
                producingTransportId = options.producingTransportId,
                handler = this._handler,
                appData = options.appData,
                // screenShare = options.screenShare
                reuseTrack = true,
                ssrc = getConsumerSsrc(event.track),
                mimeType = options.mimeType,
              ),
              coroutineScope = coroutineScope,
              logger = logger,
            )

          this._consumers[options.consumerId] = consumer
          consumerCounter++

          _handleConsumer(consumer, coroutineScope)

          logger.traceLog(
            "DyteMediaClient: MediaTransport - ${sfu.name}: Consumer created for producerId = ${options.producerId} trackId = ${options.trackId} " +
              "producingPeerId = ${options.producingPeerId} + kind = ${options.kind}"
          )

          this.observer.emit(SfuMediaEmitData(SfuMediaEvents.NewConsumer(consumer)))

          timeoutTimer.cancel()
          deferredConsumer.complete(consumer)
        }
      } catch (e: Error) {
        logger.traceLog("ConsumerDebug: Error while creating consumer: $e")
        timeoutTimer.cancel()
        deferredConsumer.completeExceptionally(exception)
      }
    }

    val reuseTrack = this.consumerTrackPool[options.consumerId]

    if (reuseTrack != null) {
      if (reuseTrack.track != null && reuseTrack.transceiver != null) {
        consumeHandler(reuseTrack)
      }
      return deferredConsumer
    }

    val existingTrackEvent = this.unknownTracksMap[key]

    if (existingTrackEvent != null) {
      this.unknownTracksMap.remove(key)
      consumeHandler(
        ConsumerTrackEvent(
          mid = existingTrackEvent.mid,
          receiver = existingTrackEvent.receiver,
          track = existingTrackEvent.track,
          transceiver = existingTrackEvent.transceiver,
        )
      )
    } else {
      this.consumerTrackEvents[key] = consumeHandler
    }

    return deferredConsumer
  }

  private fun getConsumerSsrc(track: MediaStreamTrack?): Long? {
    var result: Long? = null

    val parsedSdp = SDPUtils.parse(this._handler.getPc().remoteDescription?.sdp!!)

    parsedSdp.media.forEach { media ->
      if (media.type == track?.kind.toString().lowercase()) {
        media.ssrcs?.forEach { ssrc ->
          if (ssrc.value.split(" ").contains(track?.id)) result = ssrc.id
        }
      }
    }

    return result
  }

  private suspend fun _onTrack(event: TrackEvent) {
    val key =
      when (sfu) {
        MediaNodeType.HIVE,
        MediaNodeType.ROOM_NODE -> "${event.streams[0].id}:${event.track?.kind}"
        MediaNodeType.CF -> "${event.mid}:${event.track?.kind}"
      }

    val trackId = event.track?.id

    event.track
      ?.onEnded
      ?.onEach {
        this.consumerTrackPool.remove(trackId)
        this.unknownTracksMap.remove(key)
      }
      ?.launchIn(coroutineScope)

    val consumerTrackEvent =
      ConsumerTrackEvent(
        mid = event.mid,
        receiver = event.receiver,
        track = event.track,
        transceiver = event.transceiver,
      )

    consumerTrackPool[trackId!!] = consumerTrackEvent

    val eventHandler = this.consumerTrackEvents[key]

    if (eventHandler != null) {
      eventHandler(consumerTrackEvent)
      this.consumerTrackEvents.remove(key)
    } else {
      logger.traceWarning(
        "DyteMediaClient: MediaTransport - ${sfu.name}: Track event handler not found for key = $key"
      )

      this.unknownTracksMap[key] = event
    }
  }

  private suspend fun setRemoteDescription(sdp: SessionDescription) =
    this._handler.getPc().setRemoteDescription(sdp)

  private suspend fun setLocalDescription(sdp: SessionDescription) {
    this._handler.getPc().setLocalDescription(sdp)
  }

  suspend fun setRemoteOffer(offer: SessionDescription): SessionDescription {
    this.setRemoteDescription(offer)

    val ans = this._handler.getPc().createAnswer(OfferAnswerOptions())

    val parsedSDP = SDPUtils.parse(ans.sdp)

    parsedSDP.media =
      parsedSDP.media
        .map {
          if (it.type == "audio") {
            val updatedMediaObject = it

            if (updatedMediaObject.rtcpFb != null) {
              updatedMediaObject.rtcpFb = mutableListOf()
            }

            val hasNack =
              updatedMediaObject.rtcpFb?.any { rtcpFb -> rtcpFb.type == "nack" } ?: false

            val payload =
              when (sfu) {
                MediaNodeType.HIVE,
                MediaNodeType.ROOM_NODE -> updatedMediaObject.payloads!!
                MediaNodeType.CF -> updatedMediaObject.payloads!!.split(" ").first()
              }.toInt(10)

            val opusRtcpFb = RtcpFb(type = "nack", payload = payload)

            if (!hasNack)
              if (updatedMediaObject.rtcpFb == null) {
                updatedMediaObject.rtcpFb = mutableListOf(opusRtcpFb)
              } else {
                updatedMediaObject.rtcpFb!!.add(opusRtcpFb)
              }

            updatedMediaObject
          } else {
            it
          }
        }
        .toMutableList()

    val updatedAnswer = SessionDescription(type = ans.type, sdp = SDPUtils.write(parsedSDP))

    this.setLocalDescription(updatedAnswer)

    return updatedAnswer
  }

  private fun createConsumersOverDataChannel(producers: List<PeerProducerMeta>) {
    val channel = _dataChannels["events"]

    if (channel == null) {
      logger.traceLog("createConsumersOverDataChannel: events datachannel not ready")
      return
    }

    if (producers.isEmpty()) {
      logger.traceLog(
        "createConsumersOverDataChannel: one of producer id or publisher id is required"
      )
      return
    }

    val producersPayload = buildJsonObject {
      producers.map { p ->
        put(
          "producers",
          buildJsonArray {
            addJsonObject {
              put("producerId", p.producerId)
              put("producingTransportId", p.producingTransportId)
              put("preferredCodec", buildJsonObject { put(p.kind, p.mimeType) })
            }
          },
        )
      }
    }

    logger.traceLog("createConsumersOverDataChannel: producersPayload: $producersPayload")

    val req =
      DCMessage(type = "create_consumer", payload = producersPayload)
        .withBolt(Bolt(id = DCMessage.generateId(), type = BoltSendType.REQUEST))

    logger.traceLog("createConsumersOverDataChannel: sending create_consumer request: $req")

    channel.send(Json.encodeToString(DCMessage.serializer(), req).encodeToByteArray())
  }

  fun sendResponseOverDataChannel(label: String, id: String, message: DCMessage) {
    val channel = _dataChannels[label] ?: throw Error("DataChannel not found")

    val extendedDcMessage = message.withBolt(Bolt(id = id, type = BoltSendType.RESPONSE))

    val jsonPayload = Json.encodeToString(DCMessage.serializer(), extendedDcMessage)
    val maxChunkSize = 16348

    if (jsonPayload.length > maxChunkSize) {
      val chunkSize = maxChunkSize - 200 // buffer for other data
      val rawChunks = jsonPayload.chunked(chunkSize)
      val totalChunks = rawChunks.size
      val messageId = UUIDUtils.getRandom()

      rawChunks.forEachIndexed { i, chunk ->
        val chunkedDCMessage =
          DCMessageChunked(id = messageId, count = totalChunks, chunkIndex = i, chunk = chunk)
        channel.send(
          Json.encodeToString(DCMessageChunked.serializer(), chunkedDCMessage).encodeToByteArray()
        )
      }
    } else {
      channel.send(jsonPayload.encodeToByteArray())
    }
  }

  suspend fun retryFailedConsumerCreationTasks(
    tasks: List<HiveConsumerCreationTaskException>
  ): List<CompletableDeferred<MediaConsumer>> {
    val deferredResults = mutableListOf<CompletableDeferred<MediaConsumer>>()

    tasks.forEach { deferredResults.add(this._consumerCreationTask(it.options)) }

    return deferredResults
  }
}
