package io.dyte.media.common

import io.dyte.media.Direction
import io.dyte.media.SdpObject
import io.dyte.media.handlers.sdp.CommonUtils
import io.dyte.media.utils.IMediaClientLogger
import io.dyte.media.utils.RtpCapabilities
import io.dyte.media.utils.UUIDUtils
import io.dyte.media.utils.sdp.SDPUtils
import io.dyte.webrtc.BundlePolicy
import io.dyte.webrtc.CommonRtpTransceiverInit
import io.dyte.webrtc.IceTransportPolicy
import io.dyte.webrtc.MediaStreamTrackKind
import io.dyte.webrtc.OfferAnswerOptions
import io.dyte.webrtc.PeerConnection
import io.dyte.webrtc.RtcConfiguration
import io.dyte.webrtc.RtcpMuxPolicy
import io.dyte.webrtc.RtpTransceiverDirection
import io.dyte.webrtc.SessionDescription
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow

data class MediaDeviceOptions(
  /** The name of one of the builtin handlers, only unified plan in our case */
  val handlerName: String = "UnifiedPlan",
  /** Custom handler factory */
  val handlerFactory: MediaHandlerFactory = MediaUnifiedPlan.createFactory(),
)

class MediaDevice(
  private val sfu: MediaNodeType,
  private val options: MediaDeviceOptions,
  private val coroutineScope: CoroutineScope,
  private val logger: IMediaClientLogger,
) {
  /** RTC Handler factory */
  private val handlerFactory = options.handlerFactory

  val observer = MutableSharedFlow<SfuMediaEmitData>()

  /** RTC Handler name */
  private var _handlerName: String
  val handlerName: String
    get() = this._handlerName

  init {
    // Create a temporary handler to get its name
    val handler = this.handlerFactory()
    _handlerName = handler.getName()
  }

  private suspend fun getRtpCapabilities(direction: Direction): RtpCapabilities {
    val pc =
      PeerConnection(
        RtcConfiguration(
          bundlePolicy = BundlePolicy.MaxBundle,
          iceTransportPolicy = IceTransportPolicy.All,
          rtcpMuxPolicy = RtcpMuxPolicy.Require,
        )
      )

    try {
      val tempDirection =
        if (direction == Direction.recv) {
          RtpTransceiverDirection.RecvOnly
        } else {
          RtpTransceiverDirection.SendOnly
        }

      pc.addTransceiver(
        track = null,
        kind = MediaStreamTrackKind.Audio,
        init =
          CommonRtpTransceiverInit(
            direction = tempDirection,
            streams = emptyList(),
            sendEncodings = emptyList(),
          ),
      )

      pc.addTransceiver(
        track = null,
        kind = MediaStreamTrackKind.Video,
        init =
          CommonRtpTransceiverInit(
            direction = tempDirection,
            streams = emptyList(),
            sendEncodings = emptyList(),
          ),
      )

      val offer: SessionDescription = pc.createOffer(OfferAnswerOptions())
      val parsedOffer = SDPUtils.parse(requireNotNull(offer.sdp))
      val sdpObject: SdpObject = parsedOffer
      val nativeRtpCapabilities: RtpCapabilities = CommonUtils.extractRtpCapabilities(sdpObject)

      return nativeRtpCapabilities
    } catch (e: Exception) {
      try {
        pc.close()
      } catch (_: Exception) {}
      throw e
    }
  }

  suspend fun getSendRtpCapabilities(): RtpCapabilities = getRtpCapabilities(Direction.send)

  suspend fun getRecvRtpCapabilities(): RtpCapabilities = getRtpCapabilities(Direction.recv)

  suspend fun createSendTransport(options: MediaTransportOptions): MediaTransport =
    createTransport(options, RtpTransceiverDirection.SendOnly)

  suspend fun createRecvTransport(options: MediaTransportOptions): MediaTransport =
    createTransport(options, RtpTransceiverDirection.RecvOnly)

  private suspend fun createTransport(
    options: MediaTransportOptions,
    direction: RtpTransceiverDirection,
  ): MediaTransport {
    logger.traceLog(
      "DyteMediaClient: ${sfu.name}: Creating ${direction.name} transport with options: $options"
    )

    val transportOptions =
      MediaInternalTransportOptions(
        id = UUIDUtils.getRandom(),
        direction = direction,
        iceServers = options.iceServers!!,
        iceTransportPolicy = options.iceTransportPolicy,
        additionalSettings = options.additionalSettings,
        proprietaryConstraints = options.proprietaryConstraints,
        appData = options.appData,
        handlerFactory = handlerFactory,
      )

    val transport = MediaTransport(sfu, transportOptions, coroutineScope, logger)

    observer.emit(SfuMediaEmitData(SfuMediaEvents.NewTransport(this)))

    return transport
  }
}
