package io.dyte.media.handlers.sdp

import io.dyte.media.DtlsFingerprint
import io.dyte.media.DtlsParameters
import io.dyte.media.DtlsRole
import io.dyte.media.SdpObject
import io.dyte.media.utils.*
import io.dyte.media.utils.sdp.SDPUtils
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.int

class CommonUtils {
  companion object {
    fun extractRtpCapabilities(sdpObject: SdpObject): RtpCapabilities {
      val codecsMap: MutableMap<Int, LocalRtpCodecParameters> = mutableMapOf()
      val headerExtensions: MutableList<RtpHeaderExtension> = mutableListOf()
      var gotAudio: Boolean = false
      var gotVideo: Boolean = false

      for (m: MediaObject in sdpObject.media) {
        val kind: String = requireNotNull(m.type)

        when (kind) {
          "audio" -> {
            if (gotAudio) continue
            gotAudio = true
          }
          "video" -> {
            if (gotVideo) continue
            gotVideo = true
          }
          else -> continue
        }

        for (rtp: Rtp in requireNotNull(m.rtp)) {
          val codec =
            LocalRtpCodecParameters(
              kind = RTCRtpMediaType.fromString(kind),
              mimeType = "$kind/${rtp.codec}",
              preferredPayloadType = rtp.payload,
              clockRate = rtp.rate,
              numChannels = rtp.encoding,
              parameters = mutableMapOf(),
              rtcpFeedback = mutableListOf(),
            )

          codecsMap[requireNotNull(codec.preferredPayloadType)] = codec
        }

        for (fmtp: Fmtp in m.fmtp ?: emptyList()) {
          val parameters = SDPUtils.parseParams(fmtp.config).toMutableMap()
          val codec: LocalRtpCodecParameters? = codecsMap[fmtp.payload]

          if (codec == null) continue

          if (parameters["profile-level-id"] != null)
            parameters["profile-level-id"] = JsonPrimitive("${parameters["profile-level-id"]}")

          codec.parameters = parameters
        }

        for (fb: RtcpFb in m.rtcpFb ?: emptyList()) {
          val codec: LocalRtpCodecParameters? = codecsMap[fb.payload]

          if (codec == null) continue

          val feedback = RtcpFeedback(type = fb.type, parameter = fb.subtype)

          codec.rtcpFeedback.add(feedback)
        }

        for (ext: Ext in m.ext ?: emptyList()) {
          if (!ext.encryptUri.isNullOrEmpty()) continue

          val headerExtension =
            RtpHeaderExtension(
              kind = RTCRtpMediaType.fromString(kind),
              uri = ext.uri,
              preferredId = ext.value,
            )

          headerExtensions.add(headerExtension)
        }
      }

      val rtpCapabilities =
        RtpCapabilities(
          codecs = codecsMap.values.toMutableList(),
          headerExtensions = headerExtensions,
        )

      return rtpCapabilities
    }

    fun extractDtlsParameters(sdpObject: SdpObject): DtlsParameters {
      val mediaObject: MediaObject =
        sdpObject.media.firstOrNull { m ->
          !m.iceUfrag.isNullOrEmpty() && m.port != null && m.port != 0
        } ?: throw Exception("No active media section found")

      val fingerprint: Fingerprint =
        requireNotNull(mediaObject.fingerprint ?: sdpObject.fingerprint)

      var role: DtlsRole = DtlsRole.auto

      if (mediaObject != null) {
        when (mediaObject.setup) {
          "active" -> role = DtlsRole.client
          "passive" -> role = DtlsRole.server
          "actpass" -> role = DtlsRole.auto
        }
      }

      val dltsParameters =
        DtlsParameters(
          role = role,
          fingerprints =
            listOf(DtlsFingerprint(algorithm = fingerprint.type, value = fingerprint.hash)),
        )

      return dltsParameters
    }

    fun getCname(offerMediaObject: MediaObject): String {
      val ssrcCnameLine: Ssrc =
        (offerMediaObject.ssrcs ?: emptyList()).firstOrNull { it.attribute == "cname" }
          ?: Ssrc(value = "")

      return ssrcCnameLine.value
    }

    fun applyCodecParameters(
      offerRtpParameters: LocalRtpParameters,
      answerMediaObject: MediaObject,
    ) {
      for (codec in offerRtpParameters.codecs) {
        val mimeType: String = requireNotNull(codec.mimeType).lowercase()

        if (mimeType != "audio/opus") continue

        val rtp: Rtp? =
          (answerMediaObject.rtp ?: emptyList()).firstOrNull { it.payload == codec.payloadType }

        if (rtp == null) continue

        answerMediaObject.fmtp = (answerMediaObject.fmtp ?: emptyList()) as MutableList<Fmtp>

        var fmtp: Fmtp? =
          (answerMediaObject.fmtp ?: emptyList()).firstOrNull { it.payload == codec.payloadType }

        if (fmtp == null) {
          fmtp = Fmtp(payload = codec.payloadType!!, config = "")

          answerMediaObject.fmtp!!.add(fmtp)
        }

        val parameters = SDPUtils.parseParams(fmtp.config).toMutableMap()

        if (mimeType == "audio/opus") {
          val spropStereo = codec.parameters["sprop-stereo"]?.int

          if (spropStereo != null)
            parameters["stereo"] = JsonPrimitive(if (spropStereo > 0) 1 else 0)
        }

        fmtp.config = ""

        for (key: Any in parameters.keys) {
          if (fmtp.config.isNotEmpty() == true) fmtp.config += ";"

          fmtp.config += "$key=${parameters[key]}"
        }
      }
    }
  }
}
