package io.dyte.media.common

import io.dyte.media.utils.IMediaClientLogger
import io.dyte.webrtc.CommonRtpEncodingParameters
import io.dyte.webrtc.MediaStreamTrack
import io.dyte.webrtc.MediaStreamTrackKind
import io.dyte.webrtc.MediaStreamTrackState
import io.dyte.webrtc.RtcStatsReport
import io.dyte.webrtc.onEnded
import io.dyte.webrtc.readyState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.launch

class MediaProducer(
  private val options: MediaInternalProducerOptions,
  private val coroutineScope: CoroutineScope,
  private val logger: IMediaClientLogger,
) {
  /** Producer Id */
  val id = options.id

  /** Local Id */
  val localId = options.localId

  /** Closed flag */
  var closed = false
    private set

  /** Associated RtpSender */
  private val handler = options.handler

  /** Local track */
  var track = options.track
    private set

  /** Producer kind */
  val kind = track.kind

  /** Paused flag */
  var paused = if (options.disableTrackOnPause) !track.enabled else false
    private set

  /** Video max spatial layer */
  var maxSpatialLayer: Long? = null
    private set

  /** Whether the Producer should call stop() in the given tracks */
  private val stopTracks = options.stopTracks

  /** Whether the Producer should set track.enabled = false when paused */
  private val disableTrackOnPause = options.disableTrackOnPause

  /** Whether we should replace the RTCRtpSender.track with null when paused */
  private val zeroRtpOnPause = options.zeroRtpOnPause

  /** Custom data */
  val appData = options.appData

  /** Observer for external libs listen for few events */
  val observer = MutableSharedFlow<SfuMediaEmitData>()

  init {
    coroutineScope.launch { handleTrack() }
  }

  /** Closes the Producer */
  suspend fun close(reason: String?) {
    if (closed) return

    logger.traceLog("DyteMediaClient: BaseProducer: close() with reason: $reason")
    closed = true
    destroyTrack()

    if (reason == REASON_TRANSPORT_CLOSED || reason == REASON_DISCONNECTION_CLEANUP) {
      observer.emit(SfuMediaEmitData(SfuMediaEvents.Close(reason)))
      return
    }

    val handlerResult = handler.stopSending(localId)
    observer.emit(SfuMediaEmitData(SfuMediaEvents.Close(reason, handlerResult)))
  }

  /** Get associated RtpSender stats */
  suspend fun getStats(): RtcStatsReport? {
    if (closed) throw IllegalStateException("Producer closed")
    return handler.getSenderStats(localId)
  }

  /** Pauses sending media */
  suspend fun pause() {
    logger.traceLog("DyteMediaClient: BaseProducer: pause() for $id")
    if (closed) logger.traceLog("DyteMediaClient: BaseProducer: pause(): Producer closed")

    paused = true
    if (disableTrackOnPause) track.enabled = false
    if (zeroRtpOnPause) handler.replaceTrack(localId, null)

    observer.emit(SfuMediaEmitData(SfuMediaEvents.Pause))
  }

  /** Resumes sending media */
  suspend fun resume() {
    logger.traceLog("DyteMediaClient: BaseProducer: resume() for $id")
    if (closed) {
      logger.traceLog("DyteMediaClient: BaseProducer: resume(): Producer closed")
      return
    }

    paused = false
    if (disableTrackOnPause) track.enabled = true
    if (zeroRtpOnPause) handler.replaceTrack(localId, track)

    observer.emit(SfuMediaEmitData(SfuMediaEvents.Resume))
  }

  /** Replaces the current track with a new one or null */
  suspend fun replaceTrack(newTrack: MediaStreamTrack?) {
    logger.traceLog("DyteMediaClient: BaseProducer: replaceTrack() with ${newTrack?.id}")

    if (closed) {
      if (newTrack != null && stopTracks) {
        try {
          newTrack.stop()
        } catch (e: Exception) {
          logger.traceError("DyteMediaClient: BaseProducer: error on replaceTrack(): ${e.message}")
        }
      }
      throw IllegalStateException("Producer closed")
    } else if (newTrack != null && newTrack.readyState is MediaStreamTrackState.Ended) {
      throw IllegalStateException("Track ended")
    }

    if (newTrack == track) {
      logger.traceLog("DyteMediaClient: BaseProducer: replaceTrack(): Same track, ignored")
      return
    }

    if (!zeroRtpOnPause || !paused) handler.replaceTrack(localId, newTrack)

    destroyTrack()
    track = newTrack!!

    if (disableTrackOnPause) {
      track.enabled = !paused
    }

    handleTrack()
  }

  /** Sets the video max spatial layer to be sent */
  suspend fun setMaxSpatialLayer(spatialLayer: Long) {
    if (closed) throw IllegalStateException("closed")
    else if (kind != MediaStreamTrackKind.Video)
      throw UnsupportedOperationException("Not a Video Producer")

    if (spatialLayer == maxSpatialLayer) return
    handler.setMaxSpatialLayer(localId, spatialLayer)
    maxSpatialLayer = spatialLayer
  }

  /** Sets the DSCP value */
  suspend fun setRtpEncodingParameters(params: CommonRtpEncodingParameters) {
    if (closed) throw IllegalStateException("Producer closed")
    handler.setRtpEncodingParameters(localId, params)
  }

  private suspend fun onTrackEnded() {
    logger.traceLog("DyteMediaClient: BaseProducer: Track ended event")
    observer.emit(SfuMediaEmitData(SfuMediaEvents.TrackEnded))
  }

  private suspend fun handleTrack() {
    track.onEnded.collect { onTrackEnded() }
  }

  private fun destroyTrack() {
    try {
      if (stopTracks) track.stop()
    } catch (e: Exception) {
      logger.traceError(e.toString())
    }
  }
}
