package io.dyte.media.common

import io.dyte.webrtc.CommonRtpEncodingParameters
import io.dyte.webrtc.DataChannel
import io.dyte.webrtc.IceServer
import io.dyte.webrtc.IceTransportPolicy
import io.dyte.webrtc.MediaStream
import io.dyte.webrtc.MediaStreamTrack
import io.dyte.webrtc.PeerConnection
import io.dyte.webrtc.RtcStatsReport
import io.dyte.webrtc.RtpCodecParameters
import io.dyte.webrtc.RtpReceiver
import io.dyte.webrtc.RtpTransceiver
import io.dyte.webrtc.RtpTransceiverDirection
import io.dyte.webrtc.SessionDescription
import io.dyte.webrtc.TrackEvent

open class MediaGenericHandlerResult(
  open val offerSdp: SessionDescription,
  open val callback: suspend (answer: SessionDescription) -> Any,
)

class MediaHandlerRunOptions(
  val direction: RtpTransceiverDirection,
  val iceServers: List<IceServer>?,
  val iceTransportPolicy: IceTransportPolicy?,
  val additionalSettings: Map<String, Any>?,
  val proprietaryConstraints: Any?,
  val onTrackHandler: suspend (event: TrackEvent) -> Unit,
)

class MediaCodecOptions(val name: String, val parameters: List<String>?)

class MediaHandlerSendOptions(
  val track: MediaStreamTrack,
  val stream: MediaStream,
  val screenShare: Boolean?,
  val encodings: List<CommonRtpEncodingParameters>? = null,
  val codecs: List<RtpCodecParameters>? = null, // Same as CodecCapability?
  val codecOptions: List<MediaCodecOptions>? = null,
)

class MediaHandlerSendResult(
  override val offerSdp: SessionDescription,
  override val callback: suspend (answer: SessionDescription) -> Any,
) : MediaGenericHandlerResult(offerSdp, callback)

class MediaHandlerReceiveResult(val track: MediaStreamTrack, val localId: String)

class MediaHandlerReceiveDataChannelResult(val dataChannel: DataChannel)

abstract class MediaHandlerInterface {
  /** Map of RtpTransceivers indexed by MID */
  val mapMidTransceiver: MutableMap<String, RtpTransceiver> = mutableMapOf()
  val mapMidReceiver: MutableMap<String, RtpReceiver> = mutableMapOf()

  abstract fun getName(): String

  abstract fun getPc(): PeerConnection

  abstract fun close()

  abstract suspend fun init(options: MediaHandlerRunOptions)

  abstract suspend fun connect(): MediaGenericHandlerResult

  abstract suspend fun updateIceServers(iceServers: List<IceServer>)

  abstract suspend fun restartIce(): MediaGenericHandlerResult

  abstract suspend fun getTransportStats(): RtcStatsReport?

  abstract suspend fun send(options: MediaHandlerSendOptions): MediaHandlerSendResult

  abstract suspend fun stopSending(localId: String): MediaGenericHandlerResult

  abstract suspend fun replaceTrack(localId: String, track: MediaStreamTrack?)

  abstract suspend fun setMaxSpatialLayer(localId: String, spatialLayer: Long)

  abstract suspend fun setRtpEncodingParameters(localId: String, params: Any)

  abstract suspend fun getSenderStats(localId: String): RtcStatsReport?

  //    We don't need this right now as receiving happens async with DataChannel base negotiation
  //    abstract fun receive(answer: SessionDescription): HandlerReceiveResult

  abstract suspend fun stopReceiving(localId: String): MediaGenericHandlerResult

  abstract suspend fun pauseReceiving(localId: String): MediaGenericHandlerResult

  abstract suspend fun resumeReceiving(localId: String): MediaGenericHandlerResult

  abstract suspend fun getReceiverStats(localId: String): RtcStatsReport?
}

typealias MediaHandlerFactory = () -> MediaHandlerInterface
