package io.dyte.media.common

import io.dyte.media.utils.IMediaClientLogger
import io.dyte.webrtc.RtcStatsReport
import io.dyte.webrtc.onEnded
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.launch

class MediaConsumer(
  private val options: MediaInternalConsumerOptions,
  private val coroutineScope: CoroutineScope,
  private val logger: IMediaClientLogger,
) {
  /** Consumer Id */
  private val _id = options.id!!

  /** Local Id */
  private val _localId = options.localId

  /** Associated producer Id */
  private val _producerId = options.producerId

  /** Closed flag */
  private var _closed = false

  /** Associated RtpReceiver */
  private val _handler = options.handler

  /** Remote track */
  private val _track = options.track!!

  /** Destroy track on close flag */
  private val _reuseTrack = options.reuseTrack ?: false

  /** Paused flag */
  private var _paused = options.paused ?: false

  /** Peer Id of consumer */
  private var _peerId = options.producingPeerId

  /** Transport Id of consumer */
  var transportId = options.producingTransportId
    private set

  /** App custom data */
  private val _appData = options.appData

  private val _kind = this._track.kind

  private val _ssrc = options.ssrc

  /** Observer is for external libs to listen for some events */
  val observer = MutableSharedFlow<SfuMediaEmitData>()

  init {
    coroutineScope.launch { _handleTrack() }
  }

  /** Consumer Id */
  fun getId() = this._id

  /** Peer Id of consumer */
  fun getPeerId() = this._peerId

  /** Local Id */
  fun getLocalId() = this._localId

  /** Associated producer Id */
  fun getProducerId() = this._producerId

  /** Whether the Consumer is closed */
  fun getClosed() = this._closed

  /** Media kind */
  fun getKind() = this._kind

  /** The associated track */
  fun getTrack() = this._track

  /** Whether the Consumer is paused */
  fun getPaused() = this._paused

  fun getRemotelyPaused() = this.options.paused

  /** Custom data */
  fun getAppData() = this._appData

  fun getSsrc() = this._ssrc

  /** Closes the Consumer */
  suspend fun close(reason: String) {
    if (this._closed) return

    logger.traceLog("DyteMediaClient: MediaConsumer: close() with reason: $reason")

    this._closed = true

    if (!this._reuseTrack) this._destroyTrack()

    this.observer.emit(SfuMediaEmitData(SfuMediaEvents.Close(reason)))
  }

  /** Get associated RtpReceiver stats */
  suspend fun getStats(): RtcStatsReport? {
    if (this._closed) throw IllegalStateException("Consumer closed")

    return this._handler.getReceiverStats(this._localId)
  }

  /** Pauses receiving media */
  suspend fun pause() {
    logger.traceLog("DyteMediaClient: MediaConsumer: pause() for ${this._id}")

    if (this._closed) {
      logger.traceLog("DyteMediaClient: MediaConsumer: pause(): Consumer closed")

      return
    }

    this._paused = true
    this._track.enabled = false

    this.observer.emit(SfuMediaEmitData(SfuMediaEvents.Pause))
  }

  /** Resumes receiving media */
  suspend fun resume() {
    logger.traceLog("DyteMediaClient: MediaConsumer: resume() for ${this._id}")

    if (this._closed) {
      logger.traceLog("DyteMediaClient: MediaConsumer: resume(): Consumer closed")

      return
    }

    this._paused = false
    this._track.enabled = true

    this.observer.emit(SfuMediaEmitData(SfuMediaEvents.Resume))
  }

  private suspend fun _onTrackEnded() {
    logger.traceLog("DyteMediaClient: MediaConsumer: Track ended event")

    this.observer.emit(SfuMediaEmitData(SfuMediaEvents.TrackEnded))
  }

  private suspend fun _handleTrack() {
    this._track.onEnded.collect { _onTrackEnded() }
  }

  private fun _destroyTrack() {
    try {
      this._track.stop()
    } catch (e: Error) {
      logger.traceError(e.toString())
    }
  }
}
