package io.dyte.media.common

import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put

/** DC Message is type for data channel message and consistent with Hive node DataChannelMessage */
@Serializable
data class DCMessage(val type: String, val payload: Map<String, JsonElement>) {
  /**
   * Get the bolt info from the message
   *
   * @return [Bolt] if present, null otherwise
   */
  val bolt: Bolt?
    get() =
      payload["_bolt"]?.jsonObject?.let {
        Bolt(
          id = it["id"]?.jsonPrimitive?.content.orEmpty(),
          type = BoltSendType.fromString(it["type"]?.jsonPrimitive?.content.orEmpty()),
          version = it["version"]?.jsonPrimitive?.content?.toDoubleOrNull() ?: Bolt.VERSION,
          timestamp = it["timestamp"]?.jsonPrimitive?.content?.toLongOrNull(),
        )
      }

  fun withBolt(bolt: Bolt): DCMessage {
    val newPayload = payload.toMutableMap()

    newPayload["_bolt"] = buildJsonObject {
      put("id", bolt.id.ifEmpty { generateId() })
      put("type", bolt.type.toString())
      put("version", bolt.version)
      bolt.timestamp?.let { put("timestamp", it) }
    }

    return DCMessage(type, newPayload)
  }

  companion object {
    fun generateId(len: Int = 8): String {
      return (1..len)
        .map { ('a'..'z') + ('0'..'9') }
        .flatten()
        .shuffled()
        .take(len)
        .joinToString("")
    }
  }
}

@Serializable
data class DCMessageChunked(val id: String, val count: Int, val chunkIndex: Int, val chunk: String)

@Serializable
data class Bolt(
  val id: String,
  val type: BoltSendType,
  val version: Double = VERSION,
  val timestamp: Long? = null,
) {
  companion object {
    const val VERSION = 1.0
  }
}

enum class BoltSendType {
  REQUEST,
  RESPONSE,
  NOTIFY;

  override fun toString(): String {
    return super.toString().uppercase()
  }

  companion object {
    fun fromString(value: String): BoltSendType {
      return valueOf(value.uppercase())
    }
  }
}
