import io.dyte.media.*
import io.dyte.media.handlers.sdp.*
import io.dyte.media.utils.RtpHeaderDirection

val grammar = """
    {
        "data": {
            "v": [
                {
                    "name": "version",
                    "reg": "(\\d*)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "value"
                        }
                    ]
                }
            ],
            "o": [
                {
                    "name": "origin",
                    "reg": "(\\S*) (\\d*) (\\d*) (\\S*) IP(\\d) (\\S*)",
                    "format": "{username} {sessionId} {sessionVersion} {netType} IP{ipVer} {address}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "username"
                        },
                        {
                            "type": "Long",
                            "name": "sessionId"
                        },
                        {
                            "type": "Int",
                            "name": "sessionVersion"
                        },
                        {
                            "type": "String",
                            "name": "netType"
                        },
                        {
                            "type": "Int",
                            "name": "ipVer"
                        },
                        {
                            "type": "String",
                            "name": "address"
                        }
                    ]
                }
            ],
            "s": [
                {
                    "name": "name",
                    "reg": "(.*)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                }
            ],
            "i": [
                {
                    "name": "description",
                    "reg": "(.*)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                }
            ],
            "u": [
                {
                    "name": "uri",
                    "reg": "(.*)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                }
            ],
            "e": [
                {
                    "name": "email",
                    "reg": "(.*)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                }
            ],
            "p": [
                {
                    "name": "phone",
                    "reg": "(.*)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                }
            ],
            "z": [
                {
                    "name": "timezones",
                    "reg": "(.*)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                }
            ],
            "r": [
                {
                    "name": "repeats",
                    "reg": "(.*)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                }
            ],
            "t": [
                {
                    "name": "timing",
                    "reg": "(\\d*) (\\d*)",
                    "format": "{start} {stop}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "start"
                        },
                        {
                            "type": "Int",
                            "name": "stop"
                        }
                    ]
                }
            ],
            "c": [
                {
                    "name": "connection",
                    "reg": "IN IP(\\d) (\\S*)",
                    "format": "IN IP{version} {ip}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "version"
                        },
                        {
                            "type": "String",
                            "name": "ip"
                        }
                    ]
                }
            ],
            "b": [
                {
                    "push": "bandwidth",
                    "reg": "(TIAS|AS|CT|RR|RS):(\\d*)",
                    "format": "{type}:{limit}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "type"
                        },
                        {
                            "type": "String",
                            "name": "limit"
                        }
                    ]
                }
            ],
            "m": [
                {
                    "reg": "(\\w*) (\\d*) ([\\w/]*)(?: (.*))?",
                    "format": "{type} {port} {protocol} {payloads}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "type"
                        },
                        {
                            "type": "Int",
                            "name": "port"
                        },
                        {
                            "type": "String",
                            "name": "protocol"
                        },
                        {
                            "type": "String",
                            "name": "payloads"
                        }
                    ]
                }
            ],
            "a": [
                {
                    "push": "rtp",
                    "reg": "rtpmap:(\\d*) ([\\w\\-.]*)(?:\\s*/(\\d*)(?:\\s*/(\\S*))?)?",
                    "format": "rtpmap:{payload} {codec}{if (item.rate != null) (\"/\" + item.rate) else \"\"}{if (item.encoding != null) (\"/\" + item.encoding) else \"\"}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "payload"
                        },
                        {
                            "type": "String",
                            "name": "codec"
                        },
                        {
                            "type": "Int",
                            "name": "rate",
                            "optional": true
                        },
                        {
                            "type": "String",
                            "name": "encoding",
                            "optional": true
                        }
                    ]
                },
                {
                    "push": "fmtp",
                    "reg": "fmtp:(\\d*) ([\\S| ]*)",
                    "format": "fmtp:{payload} {config}"
                    "fields": [
                        {
                            "type": "Int",
                            "name": "payload"
                        },
                        {
                            "type": "String",
                            "name": "config"
                        }
                    ]
                },
                {
                    "name": "control",
                    "reg": "control:(.*)",
                    "format": "control:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "rtcp",
                    "reg": "rtcp:(\\d*)(?: (\\S*) IP(\\d) (\\S*))?",
                    "format": "rtcp:{port}{if (item.netType != null) (\" \" + item.netType) else \"\"}{if (item.ipVer != null) (\" IP\" + item.ipVer) else \"\"}{if (item.address != null) (\" \" + item.address) else \"\"}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "port"
                        },
                        {
                            "type": "String",
                            "name": "netType",
                            "optional": true
                        },
                        {
                            "type": "Int",
                            "name": "ipVer",
                            "optional": true
                        },
                        {
                            "type": "String",
                            "name": "address",
                            "optional": true
                        }
                    ]
                },
                {
                    "push": "rtcpFbTrrInt",
                    "reg": "rtcp-fb:(\\*|\\d*) trr-int (\\d*)",
                    "format": "rtcp-fb:{payload} trr-int {value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "payload"
                        },
                        {
                            "type": "Int",
                            "name": "value"
                        }
                    ]
                },
                {
                    "push": "rtcpFb",
                    "reg": "rtcp-fb:(\\*|\\d*) ([\\w-_]*)(?: ([\\w-_]*))?",
                    "format": "rtcp-fb:{payload} {type}{if (item.subtype != null) (\" \" + item.subtype) else \"\"}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "payload"
                        },
                        {
                            "type": "String",
                            "name": "type"
                        },
                        {
                            "type": "String",
                            "name": "subtype",
                            "optional": true
                        }
                    ]
                },
                {
                    "push": "ext",
                    "reg": "extmap:(\\d+)(?:/(\\w+))?(?: (urn:ietf:params:rtp-hdrext:encrypt))? (\\S*)(?: (\\S*))?",
                    "format": "extmap:{value}{if (item.direction != null) (\"/\" + item.direction) else \"\"}{if (item.encryptUri != null) (\" \" + item.encryptUri) else \"\"}{uri}{if (item.config != null) (\" \" + item.config) else \"\"}"
                    "fields": [
                        {
                            "type": "Int",
                            "name": "value"
                        },
                        {
                            "type": "String",
                            "name": "direction",
                            "optional": true
                        },
                        {
                            "type": "String",
                            "name": "encryptUri",
                            "optional": true
                        },
                        {
                            "type": "String",
                            "name": "uri"
                        },
                        {
                            "type": "String",
                            "name": "config",
                            "optional": true
                        }
                    ]
                },
                {
                    "name": "extmapAllowMixed",
                    "reg": "(extmap-allow-mixed)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "push": "crypto",
                    "reg": "crypto:(\\d*) ([\\w_]*) (\\S*)(?: (\\S*))?",
                    "format": "crypto:{id} {suite} {config}{if (item.sessionConfig!=null) (\" \" + item.sessionConfig) else \"\"}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "id"
                        },
                        {
                            "type": "String",
                            "name": "suite"
                        },
                        {
                            "type": "String",
                            "name": "config"
                        },
                        {
                            "type": "String",
                            "name": "sessionConfig",
                            "optional": true
                        }
                    ]
                },
                {
                    "name": "setup",
                    "reg": "setup:(\\w*)",
                    "format": "setup:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "connectionType",
                    "reg": "connection:(new|existing)",
                    "format": "connection:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "mid",
                    "reg": "mid:([^\\s]*)",
                    "format": "mid:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "msid",
                    "reg": "msid:(.*)",
                    "format": "msid:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "ptime",
                    "reg": "ptime:(\\d*(?:\\.\\d*)*)",
                    "format": "ptime:{value}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "maxptime",
                    "reg": "maxptime:(\\d*(?:\\.\\d*)*)",
                    "format": "maxptime:{value}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "direction",
                    "reg": "(sendrecv|recvonly|sendonly|inactive)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "icelite",
                    "reg": "(ice-lite)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "iceUfrag",
                    "reg": "ice-ufrag:(\\S*)",
                    "format": "ice-ufrag:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "icePwd",
                    "reg": "ice-pwd:(\\S*)",
                    "format": "ice-pwd:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "fingerprint",
                    "reg": "fingerprint:(\\S*) (\\S*)",
                    "format": "fingerprint:{type} {hash}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "type"
                        },
                        {
                            "type": "String",
                            "name": "hash"
                        }
                    ]
                },
                {
                    "push": "candidates",
                    "reg": "candidate:(\\S*) (\\d*) (\\S*) (\\d*) (\\S*) (\\d*) typ (\\S*)(?: raddr (\\S*) rport (\\d*))?(?: tcptype (\\S*))?(?: generation (\\d*))?(?: network-id (\\d*))?(?: network-cost (\\d*))?",
                    "format": "candidate:{foundation} {component} {transport} {priority} {ip} {port} typ {type}{if (item.raddr!=null&&item.rport!=null) (\" raddr \" + item.raddr + \" rport \" + item.rport) else \"\" }{if (item.tcptype!=null) (\" tcptype \" + item.tcptype) else \"\" }{if (item.generation!=null) (\" generation \" + item.generation) else \"\" }{if (item.networkId!=null) (\" network-id \" + item.networkId) else \"\" }{if (item.networkCost!=null) (\" network-cost \" + item.networkCost) else \"\" }",
                    "fields": [
                        {
                            "type": "String",
                            "name": "foundation"
                        },
                        {
                            "type": "Int",
                            "name": "component"
                        },
                        {
                            "type": "String",
                            "name": "transport"
                        },
                        {
                            "type": "Int",
                            "name": "priority"
                        },
                        {
                            "type": "String",
                            "name": "ip"
                        },
                        {
                            "type": "Int",
                            "name": "port"
                        },
                        {
                            "type": "String",
                            "name": "type"
                        },
                        {
                            "type": "String",
                            "name": "raddr",
                            "optional": true
                        },
                        {
                            "type": "Int",
                            "name": "rport",
                            "optional": true
                        },
                        {
                            "type": "String",
                            "name": "tcptype",
                            "optional": true
                        },
                        {
                            "type": "Int",
                            "name": "generation",
                            "optional": true
                        },
                        {
                            "type": "Int",
                            "name": "networkId",
                            "optional": true
                        },
                        {
                            "type": "Int",
                            "name": "networkCost",
                            "optional": true
                        }
                    ]
                },
                {
                    "name": "endOfCandidates",
                    "reg": "(end-of-candidates)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "remoteCandidates",
                    "reg": "remote-candidates:(.*)",
                    "format": "remote-candidates:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "iceOptions",
                    "reg": "ice-options:(\\S*)",
                    "format": "ice-options:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "push": "ssrcs",
                    "reg": "ssrc:(\\d*) ([\\w_-]*)(?::(.*))?",
                    "format": "ssrc:{id}{if (item.attribute!=null) (\" \" + item.attribute + (if (item.value!=null) \":\" + item.value else \"\" )) else \"\"}",
                    "fields": [
                        {
                            "type": "Long",
                            "name": "id"
                        },
                        {
                            "type": "String",
                            "name": "attribute",
                            "optional": true
                        },
                        {
                            "type": "String",
                            "name": "value",
                            "optional": true
                        }
                    ]
                },
                {
                    "push": "ssrcGroups",
                    "reg": "ssrc-group:([\\x21\\x23\\x24\\x25\\x26\\x27\\x2A\\x2B\\x2D\\x2E\\w]*) (.*)",
                    "format": "ssrc-group:{semantics} {ssrcs}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "semantics"
                        },
                        {
                            "type": "String",
                            "name": "ssrcs"
                        }
                    ]
                },
                {
                    "name": "msidSemantic",
                    "reg": "msid-semantic:\\s?(\\w*) (\\S*)",
                    "format": "msid-semantic: {semantic} {token}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "semantic"
                        },
                        {
                            "type": "String",
                            "name": "token"
                        }
                    ]
                },
                {
                    "push": "groups",
                    "reg": "^group:(\\w*) (.*)",
                    "format": "group:{type} {mids}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "type"
                        },
                        {
                            "type": "String",
                            "name": "mids"
                        }
                    ]
                },
                {
                    "name": "rtcpMux",
                    "reg": "(rtcp-mux)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "rtcpRsize",
                    "reg": "(rtcp-rsize)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "sctpmap",
                    "reg": "sctpmap:([\\w_/]*) (\\S*)(?: (\\S*))?",
                    "format": "sctpmap:{sctpmapNumber}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "sctpmapNumber"
                        }
                    ]
                },
                {
                    "name": "xGoogleFlag",
                    "reg": "x-google-flag:([^\\s]*)",
                    "format": "x-google-flag:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "push": "rids",
                    "reg": "rid:([\\d\\w]+) (\\w+)(?: ([\\S| ]*))?",
                    "format": "rid:{id}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "id"
                        }
                    ]
                },
                {
                    "push": "imageattrs",
                    "reg": "imageattr:(\\d+|\\*)[\\s\\t]+(send|recv)[\\s\\t]+(\\*|[\\S+](?:[\\s\\t]+[\\S+])*)(?:[\\s\\t]+(recv|send)[\\s\\t]+(\\*|[\\S+](?:[\\s\\t]+[\\S+])*))?",
                    "format": "imageattr:{pt}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "pt"
                        }
                    ]
                },
                {
                    "name": "simulcast",
                    "reg": "simulcast:(send|recv) ([a-zA-Z0-9\\-_~;,]+)(?:\\s?(send|recv) ([a-zA-Z0-9\\-_~;,]+))?${'$'}",
                    "format": "simulcast:{dir1}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "dir1"
                        }
                    ]
                },
                {
                    "name": "simulcast03",
                    "reg": "simulcast:[\\s\\t]+([\\S+\\s\\t]+)${'$'}",
                    "format": "simulcast:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "framerate",
                    "reg": "framerate:(\\d+(?:${'$'}|\\.\\d+))",
                    "format": "framerate:{value}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "sourceFilter",
                    "reg": "source-filter: *(excl|incl) (\\S*) (IP4|IP6|\\*) (\\S*) (.*)",
                    "format": "source-filter: {filterMode} {netType} {addressTypes} {destAddress} {srcList}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "filterMode"
                        },
                        {
                            "type": "String",
                            "name": "netType"
                        },
                        {
                            "type": "String",
                            "name": "addressTypes"
                        },
                        {
                            "type": "String",
                            "name": "destAddress"
                        },
                        {
                            "type": "String",
                            "name": "srcList"
                        }
                    ]
                },
                {
                    "name": "bundleOnly",
                    "reg": "(bundle-only)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "label",
                    "reg": "label:(.+)",
                    "format": "label:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "sctpPort",
                    "reg": "sctp-port:(\\d+)${'$'}",
                    "format": "sctp-port:{value}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "maxMessageSize",
                    "reg": "max-message-size:(\\d+)${'$'}",
                    "format": "max-message-size:{value}",
                    "fields": [
                        {
                            "type": "Int",
                            "name": "value"
                        }
                    ]
                },
                {
                    "push": "tsRefClocks",
                    "reg": "ts-refclk:([^\\s=]*)(?:=(\\S*))?",
                    "format": "ts-refclk:{clksrc}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "clksrc"
                        }
                    ]
                },
                {
                    "name": "mediaClk",
                    "reg": "mediaclk:(?:id=(\\S*))? *([^\\s=]*)(?:=(\\S*))?(?: *rate=(\\d+)/(\\d+))?",
                    "format": "mediaclk:{id}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "id"
                        }
                    ]
                },
                {
                    "name": "keywords",
                    "reg": "keywds:(.+)'$'",
                    "format": "keywds:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "content",
                    "reg": "content:(.+)",
                    "format": "content:{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "bfcpFloorCtrl",
                    "reg": "floorctrl:(c-only|s-only|c-s)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "bfcpConfId",
                    "reg": "confid:(\\d+)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "bfcpUserId",
                    "reg": "userid:(\\d+)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                },
                {
                    "name": "bfcpFloorId",
                    "reg": "floorid:(.+) (?:m-stream|mstrm):(.+)",
                    "format": "floorid:{id} mstrm:{mStream}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "id"
                        },
                        {
                            "type": "String",
                            "name": "mStream"
                        }
                    ]
                },
                {
                    "push": "invalid",
                    "reg": "(.*)",
                    "format": "{value}",
                    "fields": [
                        {
                            "type": "String",
                            "name": "value"
                        }
                    ]
                }
            ]
        }
    }
"""

class SdpGrammar {
companion object {
    fun writeVersion(item: Int): String {
        return "${item}"
    }

    fun writeOrigin(item: Origin): String {
        return "${item.username} ${item.sessionId} ${item.sessionVersion} ${item.netType} IP${item.ipVer} ${item.address}"
    }


    fun writeName(item: String): String {
        return "${item}"
    }

    fun writeDescription(item: String): String {
        return "${item}"
    }

    fun writeUri(item: String): String {
        return "${item}"
    }

    fun writeEmail(item: String): String {
        return "${item}"
    }

    fun writePhone(item: String): String {
        return "${item}"
    }

    fun writeTimezones(item: String): String {
        return "${item}"
    }


    fun writeRepeats(item: String): String {
        return "${item}"
    }

    fun writeTiming(item: Timing): String {
        return "${item.start} ${item.stop}"
    }


    fun writeConnection(item: Connection): String {
        return "IN IP${item.version} ${item.ip}"
    }

    fun writeBandwidth(item: Bandwidth): String {
        return "${item.type}:${item.limit}"
    }

    fun writeMline(type: String, port: Int, protocol: String, payloads: String): String {
        return "${type} ${port} ${protocol} ${payloads}"
    }

    fun writeRtp(item: Rtp): String {
        if (item.codec == "opus") {
            return "rtpmap:${item.payload} ${item.codec}${if (item.rate != null) ("/" + item.rate) else ""}${("/2")}"
        }

        return "rtpmap:${item.payload} ${item.codec}${if (item.rate != null) ("/" + item.rate) else ""}${if (item.encoding != null) ("/" + item.encoding) else ""}"
    }

    fun writeFmtp(item: Fmtp): String {
        return "fmtp:${item.payload} ${item.config}"
    }

    fun writeControl(item: String): String {
        return "control:${item}"
    }

    fun writeRtcp(item: Rtcp): String {
        return "rtcp:${item.port}${if (item.netType != null) (" " + item.netType) else ""}${if (item.ipVer != null) (" IP" + item.ipVer) else ""}${if (item.address != null) (" " + item.address) else ""}"
    }

    fun writeRtcpfbtrrint(item: RtcpFbTrrInt): String {
        return "rtcp-fb:${item.payload} trr-int ${item.value}"
    }

    fun writeRtcpfb(item: RtcpFb): String {
        return "rtcp-fb:${item.payload} ${item.type}${if (item.subtype != null) (" " + item.subtype) else ""}"
    }

    fun writeExt(item: Ext): String {
        return "extmap:${item.value}${if (item.direction != null) ("/" + item.direction) else ""}${if (item.encryptUri != null) (" " + item.encryptUri) else ""} ${item.uri}${if (item.config != null) (" " + item.config) else ""}"
    }

    fun writeCrypto(item: Crypto): String {
        return "crypto:${item.id} ${item.suite} ${item.config}${if (item.sessionConfig != null) (" " + item.sessionConfig) else ""}"
    }

    fun writeSetup(item: String): String {
        return "setup:${item}"
    }

    fun writeConnectiontype(item: String): String {
        return "connection:${item}"
    }

    fun writeMid(item: String): String {
        return "mid:${item}"
    }

    fun writeMsid(item: String): String {
        return "msid:${item}"
    }

    fun writePtime(item: Int): String {
        return "ptime:${item}"
    }

    fun writeMaxptime(item: Int): String {
        return "maxptime:${item}"
    }

    fun writeDirection(item: RtpHeaderDirection): String {
        return "${RtpHeaderDirection.value(item)}"
    }

    fun writeIcelite(item: String): String {
        return "${item}"
    }

    fun writeIceufrag(item: String): String {
        return "ice-ufrag:${item}"
    }

    fun writeIcepwd(item: String): String {
        return "ice-pwd:${item}"
    }

    fun writeFingerprint(item: Fingerprint): String {
        return "fingerprint:${item.type} ${item.hash}"
    }

    fun writeCandidates(item: IceCandidate): String {
        return "candidate:${item.foundation} ${item.component} ${item.transport} ${item.priority} ${item.ip} ${item.port} typ ${item.type}${if (item.raddr != null && item.rport != null) (" raddr " + item.raddr + " rport " + item.rport) else ""}${if (item.tcpType != null) (" tcptype " + item.tcpType) else ""}${if (item.generation != null) (" generation " + item.generation) else ""}${if (item.networkId != null) (" network-id " + item.networkId) else ""}${if (item.networkCost != null) (" network-cost " + item.networkCost) else ""}"
    }

    fun writeEndofcandidates(item: String): String {
        return "${item}"
    }

    fun writeRemotecandidates(item: RemoteSdp): String {
        return "remote-candidates:${item}"
    }

    fun writeIceoptions(item: String): String {
        return "ice-options:${item}"
    }

    fun writeSsrcs(item: Ssrc): String {
        return "ssrc:${item.id}${if (item.attribute != null) (" " + item.attribute + (if (item.value != null) ":" + item.value else "")) else ""}"
    }

    fun writeSsrcgroups(item: SsrcGroup): String {
        return "ssrc-group:${item.semantics} ${item.ssrcs}"
    }

    fun writeMsidsemantic(item: MsidSemantic): String {
        return "msid-semantic: ${item.semantic} ${item.token}"
    }

    fun writeGroups(item: Group): String {
        return "group:${item.type} ${item.mids}"
    }

    fun writeRtcpmux(item: String): String {
        return "${item}"
    }

    fun writeRtcprsize(item: String): String {
        return "${item}"
    }

    fun writeSctpmap(item: Sctpmap): String {
        return "sctpmap:${item.sctpmanNumber}"
    }

    fun writeXgoogleflag(item: String): String {
        return "x-google-flag:${item}"
    }

    fun writeRids(item: Rid): String {
        return "rid:${item.id}"
    }

    fun writeImageattrs(item: Imageattrs): String {
        return "imageattr:${item.pt}"
    }

    fun writeSimulcast(item: Simulcast): String {
        return "simulcast:${item.dir1}"
    }

    fun writeSimulcast03(item: Simulcast_03): String {
        return "simulcast:${item}"
    }

    fun writeFramerate(item: String): String {
        return "framerate:${item}"
    }

    fun writeSourcefilter(item: SourceFilter): String {
        return "source-filter: ${item.filterMode} ${item.netType} ${item.addressTypes} ${item.destAddress} ${item.srcList}"
    }

    fun writeBundleonly(item: String): String {
        return "${item}"
    }

    fun writeLabel(item: Int): String {
        return "label:${item}"
    }

    fun writeSctpport(item: Int): String {
        return "sctp-port:${item}"
    }

    fun writeMaxmessagesize(item: Int): String {
        return "max-message-size:${item}"
    }

//    fun writeTsrefclocks(item: ): String {
//        return "ts-refclk:${item.clksrc}"
//    }
//
//    fun writeMediaclk(item: SdpMediaclk): String {
//        return "mediaclk:${item.id}"
//    }

    fun writeKeywords(item: String): String {
        return "keywds:${item}"
    }

    fun writeContent(item: String): String {
        return "content:${item}"
    }

    fun writeBfcpfloorctrl(item: String): String {
        return "${item}"
    }

    fun writeBfcpconfid(item: String): String {
        return "${item}"
    }

    fun writeBfcpuserid(item: String): String {
        return "${item}"
    }

//
//    fun writeBfcpfloorid(item: ): String {
//        return "floorid:${item.id} mstrm:${item.mStream}"
//    }
//
//    fun writeInvalid(item: SdpInvalid): String {
//        return "${item}"
//    }
}
}
