package io.dyte.media.handlers.sdp

import io.dyte.media.*
import io.dyte.media.utils.*
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonNames
import kotlinx.serialization.json.JsonPrimitive

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Rtp(
    @JsonNames("payload")
    val payload: Int,
    @JsonNames("codec")
    val codec: String,
    @JsonNames("rate")
    val rate: Int,
    @JsonNames("encoding")
    var encoding: Int? = null,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Fmtp(
    @JsonNames("paylaod")
    val payload: Int,
    @JsonNames("config")
    var config: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Connection(
    @JsonNames("version")
    val version: Int,
    @JsonNames("ip")
    val ip: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Rtcp(
    @JsonNames("port")
    val port: Int,
    @JsonNames("netType")
    val netType: String,
    @JsonNames("ipVer")
    val ipVer: Int,
    @JsonNames("address")
    val address: String
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Fingerprint(
    @JsonNames("type")
    val type: String,
    @JsonNames("hash")
    val hash: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Ext(
    @JsonNames("value")
    val value: Int? = null,
    @JsonNames("direction")
    val direction: String? = null,
    @JsonNames("uri")
    val uri: String? = null,
    @JsonNames("config")
    val config: String? = null,
    @JsonNames("encrypt-uri")
    val encryptUri: String? = null,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class RtcpFb(
    @JsonNames("payload")
    val payload: Int,
    @JsonNames("type")
    val type: String,
    @JsonNames("subtype")
    val subtype: String? = null,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Ssrc(
    @JsonNames("id")
    val id: Long = 0,
    @JsonNames("attribute")
    val attribute: String? = null,
    @JsonNames("value")
    val value: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class SsrcGroup(
    @JsonNames("semantics")
    val semantics: String,
    @JsonNames("ssrcs")
    val ssrcs: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Sctpmap(
    @JsonNames("app")
    val app: String,
    @JsonNames("sctpmanNumber")
    val sctpmanNumber: Int,
    @JsonNames("maxMesasgeSize")
    val maxMessageSize: Int,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Rid(
    @JsonNames("id")
    val id: Int,
    @JsonNames("direction")
    val direction: String,
    @JsonNames("params")
    val params: String? = null
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Simulcast(
    @JsonNames("dir1")
    val dir1: String? = null,
    @JsonNames("list1")
    val list1: String? = null,
    @JsonNames("dir2")
    val dir2: String? = null,
    @JsonNames("list2")
    val list2: String? = null,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Simulcast_03(
    @JsonNames("value")
    val value: String
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class RtcpFbTrrInt(
    @JsonNames("payload")
    val payload: Int,
    @JsonNames("value")
    val value: Int,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Crypto(
    @JsonNames("id")
    val id: Int,
    @JsonNames("suite")
    val suite: String,
    @JsonNames("config")
    val config: String,
    @JsonNames("sessionConfig")
    var sessionConfig: String? = null, // Type Any?
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Bandwidth(
    @JsonNames("type")
    val type: String,
    @JsonNames("limit")
    val limit: Int,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class Imageattrs(
    @JsonNames("pt")
    val pt: Int,
    @JsonNames("dir1")
    val dir1: String,
    @JsonNames("attrs1")
    val attrs1: String,
    @JsonNames("dir2")
    val dir2: String,
    @JsonNames("attrs2")
    val attrs2: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class SourceFilter(
    @JsonNames("filterMode")
    val filterMode: String,
    @JsonNames("netType")
    val netType: String,
    @JsonNames("addressTypes")
    val addressTypes: String,
    @JsonNames("destAddress")
    val destAddress: String,
    @JsonNames("srcList")
    val srcList: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class MediaObject(
    @JsonNames("candiates")
    var candidates: List<IceCandidate>? = emptyList(),
    @JsonNames("iceUfrag")
    var iceUfrag: String? = null,
    @JsonNames("icePwd")
    var icePwd: String? = null,
    @JsonNames("endOfCandiates")
    var endOfCandidates: String? = null,
    @JsonNames("iceOptions")
    var iceOptions: String? = null,
    @JsonNames("setup")
    var setup: String? = null,
    @JsonNames("mid")
    var mid: String? = null,
    @JsonNames("port")
    var port: Int? = null,
    @JsonNames("direction")
    var direction: RtpHeaderDirection? = null,
    @JsonNames("rtp")
    var rtp: MutableList<Rtp>? = mutableListOf(),
    @JsonNames("fmtp")
    var fmtp: MutableList<Fmtp>? = mutableListOf(),
    @JsonNames("type")
    var type: String? = null,
    @JsonNames("protocol")
    var protocol: String? = null,
    @JsonNames("payloads")
    var payloads: String? = null,
    @JsonNames("connection")
    var connection: Connection? = null,
    @JsonNames("rtcp")
    var rtcp: Rtcp? = null,
    @JsonNames("ext")
    var ext: MutableList<Ext>? = mutableListOf(),
    @JsonNames("msid")
    var msid: String? = null,
    @JsonNames("rtcpMux")
    var rtcpMux: String? = null,
    @JsonNames("rtcpFb")
    var rtcpFb: MutableList<RtcpFb>? = mutableListOf(),
    @JsonNames("ssrcs")
    var ssrcs: MutableList<Ssrc>? = mutableListOf(),
    @JsonNames("ssrcGroups")
    var ssrcGroups: MutableList<SsrcGroup>? = mutableListOf(),
    @JsonNames("simulcast")
    var simulcast: Simulcast? = null,
    @JsonNames("simulcast_03")
    var simulcast_03: Simulcast_03? = null,
    @JsonNames("rids")
    var rids: MutableList<Rid>? = mutableListOf(),
    @JsonNames("extmapAllowMixed")
    var extmapAllowMixed: Boolean? = false,
    @JsonNames("rtcpRsize")
    var rtcpRsize: String? = null,
    @JsonNames("sctpPort")
    var sctpPort: Int? = null,
    @JsonNames("maxMessageSize")
    var maxMessageSize: Int? = null,
    @JsonNames("sctpmap")
    var sctpmap: Sctpmap? = null,
    @JsonNames("xGoogleFlag")
    var xGoogleFlag: String? = null,
    @JsonNames("fingerprint")
    var fingerprint: Fingerprint? = null,
    @JsonNames("rtcpFbTrrInt")
    var rtcpFbTrrInt: List<RtcpFbTrrInt>? = emptyList(),
    @JsonNames("crypto")
    var crypto: List<Crypto>? = emptyList(),
    @JsonNames("invalid")
    var invalid: List<Invalid>? = emptyList(),
    @JsonNames("ptime")
    var ptime: Int? = null,
    @JsonNames("maxptime")
    var maxptime: Int? = null,
    @JsonNames("label")
    var label: Int? = null,
    @JsonNames("bandwidth")
    var bandwidth: List<Bandwidth>? = emptyList(),
    @JsonNames("framerate")
    var framerate: String? = null,
    @JsonNames("bundleOnly")
    var bundleOnly: String? = null,
    @JsonNames("imageattrs")
    var imageattrs: List<Imageattrs>? = emptyList(),
    @JsonNames("sourceFilter")
    var sourceFilter: SourceFilter? = null,
    @JsonNames("description")
    var description: String? = null,
)

abstract class MediaSection(
    iceParameters: IceParameters? = null,
    iceCandidates: List<IceCandidate>? = null,
    dtlsParameters: DtlsParameters? = null,
    planB: Boolean
) {
    var _mediaObject: MediaObject = MediaObject()
    private var _planB: Boolean? = planB
    abstract var mid: String?

    init {
        _mediaObject = MediaObject()
        if (iceParameters != null) setIceParameters(iceParameters)
        if (iceCandidates != null) _mediaObject?.candidates = iceCandidates
        if (dtlsParameters != null) setDtlsRole(dtlsParameters.role)

        mid =_mediaObject?.mid
    }

//    fun fromMap(data: Map<String, Any?>): MediaSection {
//        val iceParameters: IceParameters = IceParameters.fromMap(data["iceParameters"] as Map<String, Any?>)
//        var iceCandidates: MutableList<IceCandidate> = mutableListOf()
//
//        if (data["iceCandidates"] != null) iceCandidates = data["iceCandidates"] as MutableList<IceCandidate>
//
//        _planB = (data["planB"] as Boolean) == true
//
//        _mediaObject = MediaObject(
//            candidates = iceCandidates,
//            iceUfrag = iceParameters.usernameFragment,
//            icePwd = iceParameters.password
//        )
//
//        return this
//    }

    fun setIceParameters(iceParameters: IceParameters) {
        _mediaObject?.iceUfrag = iceParameters.usernameFragment
        _mediaObject?.icePwd = iceParameters.password
    }

    var getObject: MediaObject? = _mediaObject

    abstract fun setDtlsRole(role: DtlsRole)

    var closed: Boolean = (_mediaObject?.port == 0)

    fun disable() {
        _mediaObject?.direction = RtpHeaderDirection.Inactive
        _mediaObject?.ext = null
        _mediaObject?.ssrcs = null
        _mediaObject?.ssrcGroups = null
        _mediaObject?.simulcast = null
        _mediaObject?.simulcast_03 = null
        _mediaObject?.rids = null
    }

    fun close() {
        _mediaObject?.direction = RtpHeaderDirection.Inactive
        _mediaObject?.port = 0
        _mediaObject?.ext = null
        _mediaObject?.ssrcs = null
        _mediaObject?.ssrcGroups = null
        _mediaObject?.simulcast = null
        _mediaObject?.simulcast_03 = null
        _mediaObject?.rids = null
        _mediaObject?.extmapAllowMixed = null
    }
}

class AnswerMediaSection(
    var mIceParameters: IceParameters,
    var iceCandidates: List<IceCandidate>,
    var dtlsParameters: DtlsParameters,
    var planB: Boolean = false,
    var sctpParameters: SctpParameters? = null,
    var plainRtpParameters: PlainRtpParameters? = null,
    var offerMediaObject: MediaObject,
    var offerRtpParameters: LocalRtpParameters? = null,
    var answerRtpParameters: LocalRtpParameters? = null,
    var codecOptions: ProducerCodecOptions? = null,
    var extmapAllowMixed: Boolean = false
): MediaSection(
    iceParameters = mIceParameters,
    iceCandidates = iceCandidates,
    dtlsParameters = dtlsParameters,
    planB = planB
) {
    private var _planB = planB
    override var mid: String? = null

    init {
        mid = offerMediaObject.mid
        _mediaObject.mid = offerMediaObject.mid
        _mediaObject.type = offerMediaObject.type
        _mediaObject.protocol = offerMediaObject.protocol

        if (plainRtpParameters == null) {
            _mediaObject.connection = Connection(
                ip = "127.0.0.1",
                version = 4
            )

            _mediaObject.port = 7
        } else {
            _mediaObject.connection = Connection(
                ip = plainRtpParameters!!.ip,
                version = plainRtpParameters!!.ipVersion
            )

            _mediaObject.port = plainRtpParameters!!.port
        }

        when (offerMediaObject.type) {
            "audio", "video" -> {
                _mediaObject.direction = RtpHeaderDirection.fromString("recvonly")
                _mediaObject.rtp = mutableListOf()
                _mediaObject.rtcpFb = mutableListOf()
                _mediaObject.fmtp = mutableListOf()

                for (codec in answerRtpParameters?.codecs ?: emptyList()) {
                    val rtp = Rtp(
                        payload = codec.payloadType!!,
                        codec = getCodecName(codec),
                        rate = codec.clockRate as Int
                    )

                    if (codec.numChannels != null && codec.numChannels!! > 1) rtp.encoding = codec.numChannels

                    _mediaObject.rtp!!.add(rtp)

                    val codecParameters = codec.parameters.toMutableMap()

                    if (codecOptions != null) {
                        val opusStereo: Int? = codecOptions?.opusStereo
                        val opusFec: Int? = codecOptions?.opusFec
                        val opusDtx: Int? = codecOptions?.opusDtx
                        val opusMaxPlaybackRate = codecOptions?.opusMaxPlaybackRate
                        val opusMaxAverageBitrate = codecOptions?.opusMaxAverageBitrate
                        val opusPtime = codecOptions?.opusPtime
                        val videoGoogleStartBitrate: Int? = codecOptions?.videoGoogleStartBitrate
                        val videoGoogleMaxBitrate: Int? = codecOptions?.videoGoogleMaxBitrate
                        val videoGoogleMinBitrate: Int? = codecOptions?.videoGoogleMinBitrate

                        var offerCodec = offerRtpParameters?.codecs?.first {
                            it.payloadType == codec.payloadType
                        }

                        when (codec.mimeType?.lowercase()) {
                            "audio/opus" -> {
                                // [WIP] WebRTC-KMP parameters to mutable map
                                offerCodec?.parameters?.put("sprop-stereo", JsonPrimitive(if (opusStereo != null) opusStereo else 0))
                                codecParameters["stereo"] = JsonPrimitive((if (opusStereo != null) opusStereo else 0).toString())

                                offerCodec?.parameters?.put("useinbandfec", JsonPrimitive(if (opusFec != null) opusFec else 0))
                                codecParameters["useinbandfec"] = JsonPrimitive((if (opusFec != null) opusFec else 0).toString())

                                offerCodec?.parameters?.put("useinbandfec", JsonPrimitive((if (opusFec != null) opusFec else 0).toString()))
                                codecParameters["useinbandfec"] = JsonPrimitive((if (opusFec != null) opusFec else 0).toString())

                                if (opusMaxPlaybackRate != null) {
                                    codecParameters["maxplaybackrate"] = JsonPrimitive(opusMaxPlaybackRate.toString())
                                }

                                if (opusMaxAverageBitrate != null) {
                                    codecParameters["maxaveragebitrate"] = JsonPrimitive(opusMaxAverageBitrate.toString())
                                }

                                if (opusPtime != null) {
                                    offerCodec?.parameters?.put("ptime", JsonPrimitive(opusPtime))
                                    codecParameters["ptime"] = JsonPrimitive(opusPtime .toString())
                                }
                            }

                            "video/vp8", "video/vp9", "video/h264", "video/h265" -> {
                                if (videoGoogleStartBitrate != null) codecParameters["x-google-start-bitrate"] =
                                    JsonPrimitive(videoGoogleStartBitrate.toString())

                                if (videoGoogleMaxBitrate != null) codecParameters["x-google-max-bitrate"] =
                                    JsonPrimitive(videoGoogleMaxBitrate.toString())

                                if (videoGoogleMinBitrate != null) codecParameters["x-google-min-bitrate"] =
                                    JsonPrimitive(videoGoogleMinBitrate.toString())
                            }
                        }
                    }

                    val fmtp = Fmtp(
                        payload = codec.payloadType!!,
                        config = ""
                    )

                    for (key: String in codecParameters.keys) {
                        if (fmtp.config.isNotEmpty()) fmtp.config += ";"

                        fmtp.config += "$key=${codecParameters[key]}"
                    }

                    if (fmtp.config.isNotEmpty()) _mediaObject.fmtp!!.add(fmtp)

                    for (fb: RtcpFeedback in codec.rtcpFeedback) {
                        _mediaObject.rtcpFb!!.add(
                            RtcpFb(
                                payload = codec.payloadType!!,
                                type = fb.type!!,
                                subtype = fb.parameter
                            )
                        )
                    }
                }

                _mediaObject.payloads = answerRtpParameters?.codecs?.map {it ->
                    it.payloadType
                }?.toList()?.joinToString(" ")

                for (ext in answerRtpParameters?.headerExtension ?: emptyList()) {
                    var found: Boolean = (offerMediaObject.ext ?: emptyList()).any { it ->
                        it.uri == ext.uri
                    }

                    if (!found) continue

                    _mediaObject.ext?.add(Ext(
                        uri = ext.uri,
                        value = ext.id
                    ))
                }

                if (extmapAllowMixed && offerMediaObject.extmapAllowMixed == true) _mediaObject.extmapAllowMixed = true

                if (offerMediaObject.simulcast != null) {
                    _mediaObject.simulcast = Simulcast(
                        dir1 = "recv",
                        list1 = offerMediaObject.simulcast!!.list1
                    )

                    _mediaObject.rids = mutableListOf()

                    for (rid: Rid in offerMediaObject.rids ?: emptyList()) {
                        if (rid.direction != "send") continue

                        _mediaObject.rids!!.add(Rid(
                            id = rid.id,
                            direction = "recv"
                        ))
                    }
                } else if (offerMediaObject.simulcast_03 != null) {
                    _mediaObject.simulcast_03 = Simulcast_03(
                        value = offerMediaObject.simulcast_03!!.value.replace(Regex("/send/g"), "recv")
                    )

                    _mediaObject.rids = mutableListOf()

                    for (rid: Rid in offerMediaObject.rids ?: emptyList()) {
                        if (rid.direction != "send") continue

                        _mediaObject.rids?.add(Rid(
                            id = rid.id,
                            direction = "recv"
                        ))
                    }
                }

                _mediaObject.rtcpMux = "rtcp-mux"
                _mediaObject.rtcpRsize = "rtcp-rsize"

                if (_planB && _mediaObject.type == "video") _mediaObject.xGoogleFlag = "conference"
            }

            "application" -> {
                if (offerMediaObject.sctpPort !is Int) {
                    _mediaObject.payloads = "webrtc-datachannel"
                    _mediaObject.sctpPort = sctpParameters?.port
                    _mediaObject.maxMessageSize = sctpParameters?.maxMessageSize
                } else if (offerMediaObject.sctpmap != null) {
                    _mediaObject.payloads = sctpParameters?.port as String
                    _mediaObject.sctpmap = Sctpmap(
                        app = "webrtc-datachannel",
                        sctpmanNumber = sctpParameters!!.port,
                        maxMessageSize = sctpParameters!!.maxMessageSize
                    )
                }
            }
        }
    }

    override fun setDtlsRole(role: DtlsRole) {
        println("Mediasoup: Setting setDtlsRole ${role}")
        when (role) {
            DtlsRole.client -> _mediaObject.setup = "active"
            DtlsRole.server -> _mediaObject.setup = "passive"
            DtlsRole.auto -> _mediaObject.setup = "actpass"
        }
    }
}

class OfferMediaSection(
    var mIceParameters: IceParameters,
    var iceCandidates: List<IceCandidate>,
    var dtlsParameters: DtlsParameters,
    var sctpParameters: SctpParameters? = null,
    var plainRtpParameters: PlainRtpParameters? = null,
    var planB: Boolean = false,
    override var mid: String?,
    var kind: String,
    var offerRtpParameters: LocalRtpParameters? = null,
    var streamId: String? = null,
    var trackId: String? = null,
    var oldDataChannelSpec: Boolean = false
): MediaSection(
    planB = planB,
    dtlsParameters = dtlsParameters,
    iceCandidates = iceCandidates,
    iceParameters = mIceParameters
) {
    private val _planB = planB

    init {
        _mediaObject.mid = mid
        _mediaObject.type = kind

        if (plainRtpParameters == null) {
            _mediaObject.connection = Connection(
                ip = "127.0.0.1",
                version = 4
            )

            if (sctpParameters == null) {
                _mediaObject.protocol = "UDP/TLS/RTP/SAVPF"
            } else {
                _mediaObject.protocol = "UDP/DTLS/SCTP"
            }

            _mediaObject.port = 7
        } else {
            _mediaObject.connection = Connection(
                ip = plainRtpParameters!!.ip,
                version = plainRtpParameters!!.ipVersion
            )

            _mediaObject.protocol = "RTP/AVP"
            _mediaObject.port = plainRtpParameters!!.port
        }

        when (kind) {
            "audio", "video" -> {
                _mediaObject.direction = RtpHeaderDirection.fromString("sendonly")
                _mediaObject.rtp = mutableListOf()
                _mediaObject.rtcpFb = mutableListOf()
                _mediaObject.fmtp = mutableListOf()

                if (!_planB) _mediaObject.msid = "${streamId ?: "-"} $trackId"

                for (codec in offerRtpParameters?.codecs ?: emptyList()) {
                    val rtp = Rtp(
                        //payload = if (kind == "audio") 111 else codec.payloadType!!,
                        payload = codec.payloadType!!,
                        codec = getCodecName(codec),
                        rate = codec.clockRate as Int
                    )

                    if (codec.numChannels != null && codec.numChannels!! > 1) rtp.encoding = codec.numChannels

                    _mediaObject.rtp!!.add(rtp)

                    val fmtp = Fmtp(
                        //payload = if (kind == "audio") 111 else codec.payloadType!!,
                        payload = codec.payloadType!!,
                        config = ""
                    )

                    for (key: String in codec.parameters.keys) {
                        if (fmtp.config.isNotEmpty()) fmtp.config += ";"

                        fmtp.config += "$key=${codec.parameters[key]}"
                    }

                    if (fmtp.config.isNotEmpty()) _mediaObject.fmtp!!.add(fmtp)

                    for (fb: RtcpFeedback in codec.rtcpFeedback) {
                        _mediaObject.rtcpFb!!.add(RtcpFb(
                            payload = codec.payloadType!!,
                            //payload = if (kind == "audio") 111 else codec.payloadType!!,
                            type = fb.type!!,
                            subtype = fb.parameter
                        ))
                    }
                }

                _mediaObject.payloads = offerRtpParameters?.codecs?.map { it ->
                    it.payloadType
                }?.toList()?.joinToString(" ")

                //if (kind == "audio") _mediaObject.payloads = "111"

                _mediaObject.ext = mutableListOf()

                for (ext in offerRtpParameters?.headerExtension ?: emptyList()) {
                    _mediaObject.ext!!.add(Ext(
                        uri = ext.uri,
                        value = ext.id
                    ))
                }

                _mediaObject.rtcpMux = "rtcp-mux"
                _mediaObject.rtcpRsize = "rtcp-rsize"

                val encoding = offerRtpParameters?.encodings?.first()
                var ssrc = encoding?.ssrc
//                [WIP] WebRTC-KMP does not have rtx
//                var rtxSsrc = if (encoding.rtx != null && encoding.rtx.ssrc != null) encoding.rtx.ssrc else null

                _mediaObject.ssrcs = mutableListOf()
                _mediaObject.ssrcGroups = mutableListOf()

                if (offerRtpParameters?.rtcp?.cname != null && offerRtpParameters!!.rtcp?.cname?.isNotEmpty() == true)
                    _mediaObject.ssrcs!!.add(Ssrc(
                        id = ssrc!!,
                        attribute = "cname",
                        value = offerRtpParameters!!.rtcp!!.cname
                    ))

                if (_planB)
                    _mediaObject.ssrcs!!.add(Ssrc(
                        id = ssrc!!,
                        attribute = "msid",
                        value = offerRtpParameters?.rtcp?.cname.toString()
                    ))

                // if (rtxSsrc != null) {...} // WIP as mentioned in line 875
            }

            "application" -> {
                if (!oldDataChannelSpec) {
                    _mediaObject.payloads = "webrtc-datachannel"
                    _mediaObject.sctpPort = sctpParameters?.port
                    _mediaObject.maxMessageSize = sctpParameters?.maxMessageSize
                } else {
                    _mediaObject.payloads = sctpParameters?.port.toString()
                    _mediaObject.sctpmap = Sctpmap(
                        app = "webrtc-datachannel",
                        sctpmanNumber = sctpParameters?.port as Int,
                        maxMessageSize = sctpParameters?.maxMessageSize as Int
                    )
                }
            }
        }
    }

    override fun setDtlsRole(role: DtlsRole) {
        _mediaObject.setup = "actpass"
    }

}

fun getCodecName(codec: LocalRtpCodecParameters): String {
    val mimeTypeRegex = Regex("^(audio|video)/(.+)")
    val mimeTypeMatch = mimeTypeRegex.findAll(codec.mimeType as String).toList() //check null stuff

    if (mimeTypeMatch.isEmpty()) throw Exception("Invalid codec.mimeType")

    return mimeTypeMatch.elementAt(0).groups[2]?.value.toString()
}