package io.dyte.media

import io.dyte.webrtc.*
import io.dyte.media.utils.LocalRtpCodecParameters
import io.dyte.media.utils.LocalRtpParameters
import io.github.aakira.napier.Napier
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.collectLatest

class ProducerCodecOptions(
    val opusStereo: Int?,
    val opusFec: Int?,
    val opusDtx: Int?,
    val opusMaxPlaybackRate: Int?,
    val opusMaxAverageBitrate: Int?,
    val opusPtime: Int?,
    val videoGoogleStartBitrate: Int?,
    val videoGoogleMaxBitrate: Int?,
    val videoGoogleMinBitrate: Int?
) {
    fun toMap(): Map<String, Int?> {
        val result = mutableMapOf<String, Int?>()

        if (opusStereo != null) result["opusStereo"] = opusStereo
        if (opusFec != null) result["opusFec"] = opusFec
        if (opusDtx != null) result["opusDtx"] = opusDtx
        if (opusMaxPlaybackRate != null) result["opusMaxPlaybackRate"] = opusMaxPlaybackRate
        if (opusMaxAverageBitrate != null) result["opusMaxAverageBitrate"] = opusMaxAverageBitrate
        if (opusPtime != null) result["opusPtime"] = opusPtime
        if (videoGoogleStartBitrate != null) result["videoGoogleStartBitrate"] = videoGoogleStartBitrate
        if (videoGoogleMaxBitrate != null) result["videoGoogleMaxBitrate"] = videoGoogleMaxBitrate
        if (videoGoogleMinBitrate != null) result["videoGoogleMinBitrate"] = videoGoogleMinBitrate

        return result
    }
}

class ProducerOptions(
    val track: MediaStreamTrack,
    val encodings: List<CommonRtpEncodingParameters>,
    val codecOptions: ProducerCodecOptions,
    val codec: LocalRtpCodecParameters,
    val stopTracks: Boolean,
    val disableTrackOnPause: Boolean,
    val zeroRtpOnPause: Boolean,
    val appData: Map<String, Any>
)

data class Producer(
    val id: String,
    val localID: String,
    var closed: Boolean = false,
    var rtpSender: RtpSender?,
    var track: MediaStreamTrack,
    val rtpParameters: LocalRtpParameters,
    val stopTracks: Boolean,
    val disableTrackOnPause: Boolean,
    val zeroRtpOnPause: Boolean,
    val appData: Map<String, Any>,
    val stream: MediaStream,
    val source: String,
    var paused: Boolean = (if (disableTrackOnPause) !track.enabled else false)
) {
    val observer = MutableSharedFlow<EmitData>()
    val kind: MediaStreamTrackKind = requireNotNull(track.kind)
    var maxSpatialLayer: Int? = null

    suspend fun close(): Unit {
        if (closed) return

        println("MEDIASOUP: Producer: close()")
        closed = true
        destroyTrack()
        observer.emit(EmitData("@close"))
    }

    suspend fun closeCopy(): Producer {
        if (closed) return this

        println("MEDIASOUP: Producer: closeCopy()")
        destroyTrack()
        observer.emit(EmitData("@close"))

        return this.copy(closed = true)
    }

    suspend fun transportClosed(): Unit {
        if (closed) return

        println("MEDIASOUP: Producer: transportClosed()")
        closed = true
        destroyTrack()
        observer.emit(EmitData("@close"))
    }

    suspend fun getStats() {
        if (closed) throw Exception("closed")

        return observer.emit(EmitData("@getstats"))
    }

    suspend fun pause(): Unit {
        println("MEDIASOUP: Producer: pause()")

        if (closed) {
            Napier.e("Producer: pause() | Producer closed.")
            return
        }

        paused = true

        if (disableTrackOnPause) track.enabled = false

        if (zeroRtpOnPause) observer.emit(EmitData("@replaceTrack"))

        observer.emit(EmitData("@pause"))
    }

    suspend fun pauseCopy(): Producer {
        println("MEDIASOUP: Producer: pauseCopy()")

        if (closed) {
            Napier.e("Producer: pauseCopy() | Producer closed.")
            return this
        }

        if (disableTrackOnPause) track.enabled = false;

        if (zeroRtpOnPause) observer.emit(EmitData("@replaceTrack"))

        observer.emit(EmitData("@pause"))

        return this.copy(paused = true)
    }

    suspend fun resume(): Unit {
        println("MEDIASOUP: Producer: resume()")

        if (closed) {
            Napier.e("Producer: resume() | Producer closed")
            return
        }

        paused = false

        if (disableTrackOnPause) track.enabled = true

        if (zeroRtpOnPause) observer.emit(EmitData("@replacetrack", mapOf("track" to track)))

        observer.emit(EmitData("@resume"))
    }

    suspend fun resumeCopy(): Producer {
        println("MEDIASOUP: Producer: resumeCopy()")

        if (closed) {
            Napier.e("Producer: resumeCopy() | Producer closed")
            return this
        }

        if (disableTrackOnPause) track.enabled = true

        if (zeroRtpOnPause) observer.emit(EmitData("@replacetrack", mapOf("track" to track)))

        observer.emit(EmitData("@resume"))

        return this.copy(paused = false)
    }

    suspend fun replaceTrack(track: MediaStreamTrack) {
        println("MEDIASOUP: Producer: replaceTrack() ${track.toString()}")

        if (closed) {
            if (stopTracks) {
                try {
                    track.stop()
                } catch (e: Exception) {}
            }
            throw Exception("Closed")
        }

        if (track == this.track) {
            println("MEDIASOUP: Producer: replaceTrack() | Same track, ignored.")
            return
        }

        if (zeroRtpOnPause || paused) observer.emit(EmitData("@replacetrack", mapOf("track" to track)))

        destroyTrack()

        this.track = track

        if (disableTrackOnPause) {
            if (!paused) {
                track.enabled = true
            } else if (paused) {
                track.enabled = false
            }
        }

        handleTrack()
    }

    suspend fun setMaxSpatialLayer(spatialLayer: Int) {
        if (closed) {
            throw Exception("Closed")
        } else if (kind.toString() != "video") {
            throw Exception("Not a video producer")
        }

        if (spatialLayer == maxSpatialLayer) return

        observer.emit(EmitData("@setmaxspatiallayer", mapOf("spatialLayer" to spatialLayer)))

        maxSpatialLayer = spatialLayer
    }

    suspend fun setRtpEncodingParameters(params: RtpEncodingParameters) {
        if (closed) {
            throw Exception("Closed")
        } else if (params == null) {
            throw Exception("Invalid params")
        }

        observer.emit(EmitData("@setrtpencodingparams", mapOf("params" to params)))
    }

    private suspend fun onTrackEnded(): Unit {
        println("MEDIASOUP: Producer: track ended event")

        observer.emit(EmitData("@trackended"))
    }

    private suspend fun handleTrack(): Unit {
        track.onEnded.collectLatest {
            onTrackEnded()
        }
    }

    private suspend fun destroyTrack(): Unit {
        try {
            // track.onEnded = null

            if (stopTracks) {
                track.stop()
                // stream.dispose() [Missing?]
            }
        } catch (e: Exception) {}
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true

        return (
            other is Producer &&
            other.id == id &&
            other.localID == localID &&
            other.closed == closed &&
            other.rtpSender == rtpSender &&
            other.track == track &&
            other.kind == kind &&
            other.rtpParameters == rtpParameters &&
            other.paused == paused &&
            other.maxSpatialLayer == maxSpatialLayer &&
            other.stopTracks == stopTracks &&
            other.disableTrackOnPause == disableTrackOnPause &&
            other.zeroRtpOnPause == zeroRtpOnPause &&
            other.appData == appData &&
            other.stream == stream &&
            other.source == source
        )
    }

    override fun hashCode(): Int {
        return id.hashCode() xor
                localID.hashCode() xor
                closed.hashCode() xor
                rtpSender.hashCode() xor
                track.hashCode() xor
                kind.hashCode() xor
                rtpParameters.hashCode() xor
                paused.hashCode() xor
                maxSpatialLayer.hashCode() xor
                stopTracks.hashCode() xor
                disableTrackOnPause.hashCode() xor
                zeroRtpOnPause.hashCode() xor
                appData.hashCode() xor
                stream.hashCode() xor
                source.hashCode()
    }
}
