package io.dyte.media.handlers.sdp

import io.dyte.media.*
import io.dyte.media.utils.LocalRtpParameters
import io.dyte.media.utils.RTCRtpMediaType
import io.dyte.media.utils.sdp.SDPUtils
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonNames

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class MediaSectionIdx(
  @JsonNames("idx") val idx: Int,
  @JsonNames("reuseMid") val reuseMid: String? = null,
)

class RemoteSdp {
  private var _iceParameters: IceParameters? = null
  private var _iceCandidates: List<IceCandidate> = emptyList()
  private var _dtlsParameters: DtlsParameters? = null
  private var _sctpParameters: SctpParameters? = null
  private var _plainRtpParameters: PlainRtpParameters? = null
  private var _planB: Boolean = false
  private var _mediaSections: MutableList<MediaSection> = mutableListOf()
  private var _midToIndex: MutableMap<String, Int> = mutableMapOf()
  private var _firstMid: String? = null
  private var _sdpObject: SdpObject? = null

  constructor(
    iceParameters: IceParameters,
    iceCandidates: List<IceCandidate>,
    dtlsParameters: DtlsParameters,
    sctpParameters: SctpParameters? = null,
    plainRtpParameters: PlainRtpParameters? = null,
    planB: Boolean = false,
  ) {
    _iceParameters = iceParameters
    _iceCandidates = iceCandidates
    _dtlsParameters = dtlsParameters
    _sctpParameters = sctpParameters
    _plainRtpParameters = plainRtpParameters
    _planB = planB
    _sdpObject =
      SdpObject(
        version = 0,
        origin =
          Origin(
            address = "0.0.0.0",
            ipVer = 4,
            netType = "IN",
            sessionId = 10000,
            sessionVersion = 0,
            username = "mediasoup-client",
          ),
        name = "-",
        timing = Timing(start = 0, stop = 0),
        media = mutableListOf(),
      )

    if (iceParameters.iceLite) _sdpObject!!.icelite = "ice-lite"

    _sdpObject!!.msidSemantic = MsidSemantic(semantic = "WMS", token = "*")

    val numFingerprints: Int = _dtlsParameters!!.fingerprints.size

    _sdpObject!!.fingerprint =
      Fingerprint(
        type = dtlsParameters.fingerprints[numFingerprints - 1].algorithm,
        hash = dtlsParameters.fingerprints[numFingerprints - 1].value,
      )

    _sdpObject!!.groups = listOf(Group(type = "BUNDLE", mids = ""))

    if (plainRtpParameters != null) {
      _sdpObject!!.origin?.address = plainRtpParameters.ip
      _sdpObject!!.origin?.ipVer = plainRtpParameters.ipVersion
    }
  }

  fun getSdp(): String {
    _sdpObject?.origin?.sessionVersion = _sdpObject?.origin?.sessionVersion!! + 1

    return SDPUtils.write(_sdpObject!!)
  }

  fun updateIceParameters(iceParameters: IceParameters) {
    _iceParameters = iceParameters
    _sdpObject?.icelite = if (iceParameters.iceLite) "ice-lite" else null

    for (mediaSection: MediaSection in _mediaSections) mediaSection.setIceParameters(iceParameters)
  }

  fun updateDtlsRole(role: DtlsRole) {
    _dtlsParameters?.role = role

    for (mediaSection: MediaSection in _mediaSections) mediaSection.setDtlsRole(role)
  }

  fun disableMediaSection(mid: String) {
    val idx: Int? = _midToIndex[mid]

    if (idx == null) throw Exception("no media section found with mid $mid")

    val mediaSection: MediaSection = _mediaSections[idx]
    mediaSection.disable()
  }

  fun closeMediaSection(mid: String): Boolean {
    val idx: Int? = _midToIndex[mid]

    if (idx == null) throw Exception("no media section found with mid $mid")

    val mediaSection: MediaSection = _mediaSections[idx]

    if (mid == _firstMid) {
      disableMediaSection(mid)

      return false
    }

    mediaSection.close()

    _regenerateBundleMids()

    return true
  }

  fun send(
    offerMediaObject: MediaObject,
    reuseMid: String? = null,
    offerRtpParameters: LocalRtpParameters,
    answerRtpParameters: LocalRtpParameters,
    codecOptions: ProducerCodecOptions? = null,
    extmapAllowMixed: Boolean = false,
  ) {
    val mediaSection =
      AnswerMediaSection(
        mIceParameters = _iceParameters!!,
        iceCandidates = _iceCandidates,
        dtlsParameters = _dtlsParameters!!,
        plainRtpParameters = _plainRtpParameters,
        planB = _planB,
        offerMediaObject = offerMediaObject,
        offerRtpParameters = offerRtpParameters,
        answerRtpParameters = answerRtpParameters,
        codecOptions = codecOptions,
        extmapAllowMixed = extmapAllowMixed,
      )

    if (reuseMid != null) {

      _replaceMediaSection(mediaSection, reuseMid)
    } else if (!_midToIndex.containsKey(mediaSection.mid)) {

      _addMediaSection(mediaSection)
    } else {

      _replaceMediaSection(mediaSection, null)
    }
  }

  fun receive(
    mid: String,
    kind: RTCRtpMediaType,
    offerRtpParameters: LocalRtpParameters,
    streamId: String,
    trackId: String,
  ) {
    val idx: Int? = _midToIndex[mid]
    var mediaSection: OfferMediaSection? = null

    if (idx != null) mediaSection = _mediaSections[idx] as OfferMediaSection

    if (mediaSection == null) {
      mediaSection =
        OfferMediaSection(
          mIceParameters = _iceParameters!!,
          iceCandidates = _iceCandidates,
          dtlsParameters = _dtlsParameters!!,
          plainRtpParameters = _plainRtpParameters,
          planB = _planB,
          mid = mid,
          kind = RTCRtpMediaType.value(kind),
          offerRtpParameters = offerRtpParameters,
          streamId = streamId,
          trackId = trackId,
        )

      val oldMediaSection: MediaSection? =
        _mediaSections.firstOrNull { m: MediaSection -> m.closed as Boolean }

      if (oldMediaSection != null) {
        _replaceMediaSection(mediaSection, oldMediaSection.mid)
      } else {
        _addMediaSection(mediaSection)
      }
    }
    //        PlanB stuff
    //        } else {
    //            mediaSection.planBReceive(
    //                offerRtpParameters = offerRtpParameters,
    //                streamId = streamId,
    //                trackId = trackId
    //            )
    //
    //            _replaceMediaSection(mediaSection, null)
    //        }
  }

  fun sendSctpAssociation(offerMediaObject: MediaObject) {
    val mediaSection =
      AnswerMediaSection(
        mIceParameters = _iceParameters!!,
        iceCandidates = _iceCandidates,
        dtlsParameters = _dtlsParameters!!,
        plainRtpParameters = _plainRtpParameters,
        offerMediaObject = offerMediaObject,
      )

    _addMediaSection(mediaSection)
  }

  fun receiveSctpAssociation(oldDataChannelSpec: Boolean = false) {
    val mediaSection =
      OfferMediaSection(
        mIceParameters = _iceParameters!!,
        iceCandidates = _iceCandidates,
        dtlsParameters = _dtlsParameters!!,
        sctpParameters = _sctpParameters,
        plainRtpParameters = _plainRtpParameters,
        mid = "datachannel",
        kind = "application",
        oldDataChannelSpec = oldDataChannelSpec,
      )

    _addMediaSection(mediaSection)
  }

  fun getNextMediaSectionIdx(): MediaSectionIdx {
    for (idx: Int in 0 until _mediaSections.size) {
      val mediaSection: MediaSection = _mediaSections[idx]

      if (mediaSection.closed) {
        return MediaSectionIdx(idx = idx, reuseMid = mediaSection.mid)
      }
    }

    return MediaSectionIdx(idx = _mediaSections.size)
  }

  private fun _regenerateBundleMids() {
    _sdpObject!!.groups[0].mids =
      _mediaSections
        .filter { mediaSection -> !mediaSection.closed }
        .map { mediaSection: MediaSection -> mediaSection.mid }
        .joinToString(" ")
  }

  private fun _addMediaSection(newMediaSection: MediaSection) {
    if (_firstMid == null) _firstMid = newMediaSection.mid

    _mediaSections.add(newMediaSection)
    _midToIndex[newMediaSection.mid.toString()] = _mediaSections.size - 1
    _sdpObject?.media?.add(newMediaSection.getObject!!)
    _regenerateBundleMids()
  }

  private fun _replaceMediaSection(newMediaSection: MediaSection, reuseMid: String?) {
    if (reuseMid != null) {
      val idx: Int? = _midToIndex[reuseMid]

      if (idx == null) throw Exception("no media section found for reuseMid $reuseMid")

      val oldMediaSection: MediaSection = _mediaSections[idx]

      _mediaSections[idx] = newMediaSection
      _midToIndex.remove(oldMediaSection.mid)
      _midToIndex[newMediaSection.mid.toString()] = idx
      _sdpObject?.media?.set(idx, newMediaSection.getObject!!)
      _regenerateBundleMids()
    } else {
      val idx: Int? = _midToIndex[newMediaSection.mid]

      if (idx == null) throw Exception("no media section found with mid ${newMediaSection.mid}")

      _mediaSections[idx] = newMediaSection
      _sdpObject?.media?.set(idx, newMediaSection.getObject!!)
    }
  }
}
