package io.dyte.media.handlers

import io.dyte.media.*
import io.dyte.media.handlers.sdp.*
import io.dyte.media.hive.REASON_DISCONNECTION_CLEANUP
import io.dyte.media.hive.REASON_TRANSPORT_CLOSED
import io.dyte.media.utils.*
import io.dyte.media.utils.sdp.SDPUtils
import io.dyte.webrtc.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async

class UnifiedPlan(private val logger: IMediaClientLogger) : HandlerInterface() {
  private var _direction: Direction? = null
  private var _remoteSdp: RemoteSdp? = null
  private var _sendingRtpParametersByKind: MutableMap<RTCRtpMediaType, LocalRtpParameters?> =
    mutableMapOf()
  private var _sendingRemoteRtpParametersByKind: MutableMap<RTCRtpMediaType, LocalRtpParameters?> =
    mutableMapOf()
  private var _forcedLocalDtlsrole: DtlsRole? = null
  private var _pc: PeerConnection? = null
  private var _mapMidTransceiver: HashMap<String, RtpTransceiver> = hashMapOf()
  private var _hasDataChannelMediaSection: Boolean = false
  private var _nextSendSctpStreamId: Int = 0
  private var _transportReady: Boolean = false
  private var _lastStream: MediaStream? = null

  private suspend fun _setupTransport(localDtlsRole: DtlsRole, localSdpObject: SdpObject? = null) {
    //        val tempSdpObject = if (localSdpObject == null)
    // SdpObject.fromMap(SDPUtils.parse(requireNotNull(_pc?.localDescription?.sdp)))
    //                            else localSdpObject

    val tempSdpObject =
      if (localSdpObject == null) SDPUtils.parse(requireNotNull(_pc?.localDescription?.sdp))
      else localSdpObject

    val dtlsParameters: DtlsParameters = CommonUtils.extractDtlsParameters(tempSdpObject)
    dtlsParameters.role = localDtlsRole

    _remoteSdp?.updateDtlsRole(
      if (localDtlsRole == DtlsRole.client) DtlsRole.server else DtlsRole.client
    )

    observer.emit(EmitData(name = "@connect", data = mapOf("dtlsParameters" to dtlsParameters)))

    _transportReady = true
  }

  private fun _assertSendDirection() {
    if (_direction != Direction.send)
      throw Exception("method can just be called for handlers with send direction")
  }

  private fun _assertRecvDirection() {
    if (_direction != Direction.recv)
      throw Exception("method can just be called for handlers with recv direction")
  }

  override suspend fun close() {
    logger.traceLog("MEDIASOUP: Handlers/UnifiedPlan: close()")

    if (_pc != null)
      try {
        _pc!!.close()
      } catch (e: Error) {}
  }

  override suspend fun getNativeRtpCapabilities(): RtpCapabilities {
    var pc =
      PeerConnection(
        RtcConfiguration(
          bundlePolicy = BundlePolicy.MaxBundle,
          iceTransportPolicy = IceTransportPolicy.All,
          rtcpMuxPolicy = RtcpMuxPolicy.Require,
        )
      )

    try {
      //            pc.addTransceiver(kind = MediaStreamTrackKind.Audio)
      //            pc.addTransceiver(kind = MediaStreamTrackKind.Video)

      var tempDirection: RtpTransceiverDirection? = null

      if (_direction == Direction.recv) {
        tempDirection = RtpTransceiverDirection.RecvOnly
      } // else if (_direction == Direction.send) {
      else {
        tempDirection = RtpTransceiverDirection.SendOnly
      }

      pc.addTransceiver(
        track = null,
        kind = MediaStreamTrackKind.Audio,
        init =
          CommonRtpTransceiverInit(
            direction = tempDirection!!,
            streams = emptyList(),
            sendEncodings = emptyList(),
          ),
      )

      pc.addTransceiver(
        track = null,
        kind = MediaStreamTrackKind.Video,
        init =
          CommonRtpTransceiverInit(
            direction = tempDirection!!,
            streams = emptyList(),
            sendEncodings = emptyList(),
          ),
      )

      val offer: SessionDescription = pc.createOffer(OfferAnswerOptions())
      val parsedOffer = SDPUtils.parse(requireNotNull(offer.sdp))
      val sdpObject: SdpObject = parsedOffer // SdpObject.fromMap(parsedOffer)
      val nativeRtpCapabilities: RtpCapabilities = CommonUtils.extractRtpCapabilities(sdpObject)

      return nativeRtpCapabilities
    } catch (e: Error) {
      try {
        pc.close()
      } catch (e2: Error) {}

      throw e
    }
  }

  override suspend fun getNativeSctpCapabilities(): SctpCapabilities {
    return SctpCapabilities(
      numStreams = NumSctpStreams(mis = SCTP_NUM_STREAMS.MIS, os = SCTP_NUM_STREAMS.OS)
    )
  }

  //    Missing Transceiver.getStats()
  override suspend fun getReceiverStats(localId: String): RtcStatsReport? {
    _assertRecvDirection()

    var transceiver: RtpTransceiver? = _mapMidTransceiver[localId]

    if (transceiver == null) throw Exception("associated RTCRtpTransceiver not found")

    return _pc?.getReceiverStats(transceiver.receiver)
  }

  override suspend fun getSenderStats(localId: String): RtcStatsReport? {
    _assertSendDirection()

    var transceiver: RtpTransceiver? = _mapMidTransceiver[localId]

    if (transceiver == null) throw Exception("associated RTCRtpTransceiver not found")

    return _pc?.getSenderStats(transceiver.sender)
  }

  override suspend fun getTransportStats(): List<RtcStatsReport> {
    val stats = _pc?.getStats()

    return if (stats != null) {
      listOf(stats)
    } else {
      emptyList()
    }
  }

  override var name = "Unified plan handler"

  override suspend fun receive(options: HandlerReceiveOptions): HandlerReceiveResult {
    _assertRecvDirection()
    logger.traceLog(
      "MEDIASOUP: Handlers/UnifiedPlan: receive() [trackId:${options.trackId}, kind:${RTCRtpMediaType.value(options.kind)}]"
    )

    val localId: String = options.rtpParameters.mid ?: _mapMidTransceiver.size.toString()

    _remoteSdp?.receive(
      mid = localId,
      kind = options.kind,
      offerRtpParameters = options.rtpParameters,
      streamId = requireNotNull(options.rtpParameters.rtcp).cname,
      trackId = options.trackId,
    )

    val offer =
      SessionDescription(SessionDescriptionUtils.fromString("offer"), _remoteSdp!!.getSdp())
    _pc?.setRemoteDescription(offer)

    var answer: SessionDescription = _pc?.createAnswer(OfferAnswerOptions())!!

    val localSdpObject: SdpObject = SDPUtils.parse(requireNotNull(answer.sdp))
    val answerMediaObject: MediaObject? = localSdpObject.media.firstOrNull { it.mid == localId }
    CommonUtils.applyCodecParameters(options.rtpParameters, answerMediaObject!!)

    answer =
      SessionDescription(
        SessionDescriptionUtils.fromString("answer"),
        SDPUtils.write(localSdpObject),
      )

    if (!_transportReady)
      _setupTransport(localDtlsRole = DtlsRole.client, localSdpObject = localSdpObject)
    _pc?.setLocalDescription(answer)

    val transceivers = _pc?.getTransceivers()
    val transceiver: RtpTransceiver? = transceivers?.firstOrNull { it.mid == localId }

    if (transceiver == null) throw Exception("new RTCRtpTransceiver not found")

    _mapMidTransceiver.iterator().forEach { (key, _) ->
      transceivers.firstOrNull { it.mid == key }?.apply { _mapMidTransceiver[key] = this }
    }
    _mapMidTransceiver[localId] = transceiver

    val stream: MediaStream? = _lastStream ?: throw Exception("Stream not found")

    return HandlerReceiveResult(
      localId = localId,
      track = requireNotNull(transceiver.receiver.track),
      rtpReceiver = transceiver.receiver,
      stream = stream!!,
    )
  }

  override suspend fun receiveDataChannel(
    options: HandlerReceiveDataChannelOptions
  ): HandlerReceiveDataChannelResult {
    _assertRecvDirection()

    logger.traceLog("MEDIASOUP: Handlers/UnifiedPlan: receiveDataChannel()")

    var dataChannel: DataChannel =
      _pc?.createDataChannel(
        label = options.label,
        id = options.sctpStreamParameters.streamId,
        ordered = options.sctpStreamParameters.ordered!!,
        maxRetransmitTimeMs = options.sctpStreamParameters.maxPacketLifeTime!!,
        maxRetransmits = options.sctpStreamParameters.maxRetransmits!!,
        protocol = options.protocol,
      )!!

    if (!_hasDataChannelMediaSection) {
      _remoteSdp?.receiveSctpAssociation()

      val offer =
        SessionDescription(SessionDescriptionUtils.fromString("offer"), _remoteSdp!!.getSdp())
      _pc?.setRemoteDescription(offer)

      val answer: SessionDescription = _pc?.createAnswer(OfferAnswerOptions())!!

      if (!_transportReady) {
        val localSdpObject: SdpObject = SDPUtils.parse(requireNotNull(answer.sdp))
        _setupTransport(
          localDtlsRole = this._forcedLocalDtlsrole ?: DtlsRole.client,
          localSdpObject = localSdpObject,
        )
      }

      _pc?.setLocalDescription(answer)
      _hasDataChannelMediaSection = true
    }

    return HandlerReceiveDataChannelResult(dataChannel = dataChannel)
  }

  override suspend fun replaceTrack(options: ReplaceTrackOptions) {
    _assertSendDirection()

    logger.traceLog(
      "MEDIASOUP: Handlers/UnifiedPlan: replaceTrack() [localId:${options.localId}, track.id${options.track.id}"
    )

    val transceiver: RtpTransceiver? = _mapMidTransceiver[options.localId]

    if (transceiver == null) throw Exception("associated RTCRtpTransceiver not found")

    transceiver.sender.replaceTrack(options.track)
    _mapMidTransceiver.remove(options.localId)
  }

  override suspend fun restartIce(iceParameters: IceParameters) {
    logger.traceLog("MEDIASOUP: Handlers/UnifiedPlan: restartIce()")

    _remoteSdp?.updateIceParameters(iceParameters)

    if (!_transportReady) return

    if (_direction == Direction.send) {
      val offer: SessionDescription = _pc?.createOffer(OfferAnswerOptions(iceRestart = true))!!
      _pc?.setLocalDescription(offer)

      val answer =
        SessionDescription(SessionDescriptionUtils.fromString("answer"), _remoteSdp!!.getSdp())
      _pc?.setRemoteDescription(answer)
    } else {
      val offer =
        SessionDescription(SessionDescriptionUtils.fromString("offer"), _remoteSdp!!.getSdp())
      _pc?.setRemoteDescription(offer)

      val answer: SessionDescription = _pc?.createAnswer(OfferAnswerOptions())!!
      _pc?.setLocalDescription(answer)
    }
  }

  override suspend fun run(options: HandlerRunOptions) {
    logger.traceLog("MEDIASOUP: Handlers/UnifiedPlan: run()")

    _direction = options.direction

    _remoteSdp =
      RemoteSdp(
        iceParameters = options.iceParameters,
        iceCandidates = options.iceCandidates,
        dtlsParameters = options.dtlsParameters,
        sctpParameters = options.sctpParameters,
      )

    _sendingRtpParametersByKind =
      mutableMapOf(
        RTCRtpMediaType.RTCRtpMediaTypeAudio to
          Ortc.getSendingRtpParameters(
            RTCRtpMediaType.RTCRtpMediaTypeAudio,
            options.extendedRtpCapabilities!!,
          ),
        RTCRtpMediaType.RTCRtpMediaTypeVideo to
          Ortc.getSendingRtpParameters(
            RTCRtpMediaType.RTCRtpMediaTypeVideo,
            options.extendedRtpCapabilities!!,
          ),
      )

    _sendingRemoteRtpParametersByKind =
      mutableMapOf(
        RTCRtpMediaType.RTCRtpMediaTypeAudio to
          Ortc.getSendingRemoteRtpParameters(
            RTCRtpMediaType.RTCRtpMediaTypeAudio,
            options.extendedRtpCapabilities!!,
          ),
        RTCRtpMediaType.RTCRtpMediaTypeVideo to
          Ortc.getSendingRemoteRtpParameters(
            RTCRtpMediaType.RTCRtpMediaTypeVideo,
            options.extendedRtpCapabilities!!,
          ),
      )

    if (options.dtlsParameters.role != DtlsRole.auto)
      this._forcedLocalDtlsrole =
        if (options.dtlsParameters.role == DtlsRole.server) DtlsRole.client else DtlsRole.server

    var _constrains =
      if (options.proprietaryConstraints.isEmpty())
        mutableMapOf<String, Any>(
          "mandatory" to emptyList<Any>(),
          "optional" to listOf(mapOf("DtlsSrtpKeyAgreement" to true)),
        )
      else options.proprietaryConstraints

    _constrains = _constrains.toMutableMap()
    _constrains["optional"] = listOf(_constrains["optional"], mapOf("DtlsSrtpKeyAgreement" to true))

    _pc =
      PeerConnection(
        RtcConfiguration(
          iceServers = options.iceServers,
          iceTransportPolicy = options.iceTransportPolicy ?: IceTransportPolicy.All,
          bundlePolicy = BundlePolicy.MaxBundle,
          rtcpMuxPolicy = RtcpMuxPolicy.Require,
        )
      )

    CoroutineScope(Dispatchers.Default).async {
      _pc!!.onTrack.collect { it.streams.forEach { it2 -> _lastStream = it2 } }
    }

    when (_pc?.iceConnectionState) {
      IceConnectionState.Checking ->
        observer.emit(
          EmitData(name = "@connectionstatechange", data = mapOf("state" to "connecting"))
        )
      IceConnectionState.Completed ->
        observer.emit(
          EmitData(name = "@connectionstatechange", data = mapOf("state" to "connected"))
        )
      IceConnectionState.Failed ->
        observer.emit(EmitData(name = "@connectionstatechange", data = mapOf("state" to "failed")))
      IceConnectionState.Disconnected ->
        observer.emit(
          EmitData(name = "@connectionstatechange", data = mapOf("state" to "disconnected"))
        )
      IceConnectionState.Closed ->
        observer.emit(EmitData(name = "@connectionstatechange", data = mapOf("state" to "closesd")))
      else -> {}
    }
  }

  override suspend fun send(options: HandlerSendOptions): HandlerSendResult {
    _assertSendDirection()

    logger.traceLog(
      "MEDIASOUP: Handlers/UnifiedPlan: send() [kind:${options.track.kind}, track.id:${options.track.id}"
    )

    if (options.encodings.size > 1) {
      var idx: Int = 0
      options.encodings.forEach { encoding -> encoding.rid = "r${idx++}" }
    }

    val sendingRtpParameters =
      _sendingRtpParametersByKind[RTCRtpMediaType.fromString(options.track.kind.toString())]?.copy()

    sendingRtpParameters?.codecs =
      Ortc.reduceCodecs(sendingRtpParameters!!.codecs, options.codec).toMutableList()

    val sendingRemoteRtpParameters =
      _sendingRemoteRtpParametersByKind[RTCRtpMediaType.fromString(options.track.kind.toString())]
        ?.copy()

    sendingRemoteRtpParameters?.codecs =
      Ortc.reduceCodecs(sendingRemoteRtpParameters!!.codecs, options.codec).toMutableList()

    val mediaSectionIdx: MediaSectionIdx = _remoteSdp?.getNextMediaSectionIdx()!!

    val transceiver =
      _pc?.addTransceiver(
        track = options.track,
        kind = options.track.kind,
        init =
          CommonRtpTransceiverInit(
            direction = RtpTransceiverDirection.SendOnly,
            streams = listOf(options.stream),
            sendEncodings = options.encodings,
          ),
      )

    var offer: SessionDescription = _pc?.createOffer(OfferAnswerOptions())!!
    var localSdpObject: SdpObject = SDPUtils.parse(offer.sdp)
    var offerMediaObject: MediaObject

    if (!_transportReady)
      _setupTransport(localDtlsRole = DtlsRole.client, localSdpObject = localSdpObject)

    var hackVp9Svc: Boolean = false

    val layers: ScalabilityMode = ScalabilityMode.parse(options.encodings)
    //            if (options.encodings.isNotEmpty()) options.encodings
    //            Missing constructor and scalabilityMode
    //            else listOf(RtpEncodingParameters(scalabilityMode =
    // "",))).first().scalabilityMode)
    //        ))

    if (
      options.encodings.size == 1 &&
        layers.spatialLayers > 1 &&
        sendingRtpParameters.codecs.first().mimeType?.lowercase() == "video/vp9"
    ) {
      logger.traceLog(
        "MEDIASOUP: Handlers/UnifiedPlan: send() | enabling legacy simulcast for VP9 SVC"
      )

      hackVp9Svc = true
      localSdpObject = SDPUtils.parse(offer.sdp)
      offerMediaObject = localSdpObject.media[mediaSectionIdx.idx]

      UnifiedPlanUtils.addLegacySimulcast(offerMediaObject, layers.spatialLayers)

      offer =
        SessionDescription(
          SessionDescriptionUtils.fromString("offer"),
          SDPUtils.write(localSdpObject),
        )
    }

    _pc?.setLocalDescription(
      SessionDescription(
        SessionDescriptionUtils.fromString("offer"),
        SDPUtils.write(SDPUtils.parse(offer.sdp)),
      )
    )

    val localId: String = transceiver?.mid!!

    sendingRtpParameters.mid = localId

    localSdpObject = SDPUtils.parse(_pc!!.localDescription?.sdp!!)
    offerMediaObject = localSdpObject.media[mediaSectionIdx.idx]

    sendingRtpParameters.rtcp?.cname = CommonUtils.getCname(offerMediaObject)
    if (options.encodings.isEmpty()) {
      sendingRtpParameters.encodings = UnifiedPlanUtils.getRtpEncodings(offerMediaObject)
      sendingRtpParameters.encodings!![0].scaleResolutionDownBy = 1.0
    } else if (options.encodings.size == 1) {
      var newEncodings: MutableList<CommonRtpEncodingParameters> =
        UnifiedPlanUtils.getRtpEncodings(offerMediaObject).toMutableList()

      newEncodings[0] = options.encodings[0]

      if (hackVp9Svc) newEncodings = mutableListOf(newEncodings[0])
      sendingRtpParameters.encodings = newEncodings
    } else {
      sendingRtpParameters.encodings = options.encodings
    }

    //        // TODO: understand this ?
    //        if (sendingRtpParameters.encodings!!.size > 1
    //            && (sendingRtpParameters.codecs[0].mimeType?.lowercase() == "video/vp8"
    //            || sendingRtpParameters.codecs[0].mimeType?.lowercase() == "video/h264"))
    //            for (encoding in sendingRtpParameters.encodings!!)
    //                 encoding.scalabilityMode = "S1T3" Missing scalabilityMode

    _remoteSdp?.send(
      offerMediaObject = offerMediaObject,
      reuseMid = mediaSectionIdx.reuseMid,
      offerRtpParameters = sendingRtpParameters,
      answerRtpParameters = sendingRemoteRtpParameters,
      codecOptions = options.codecOptions,
      extmapAllowMixed = true,
    )

    var answer =
      SessionDescription(SessionDescriptionUtils.fromString("answer"), _remoteSdp!!.getSdp())
    _pc?.setRemoteDescription(answer)

    _mapMidTransceiver[localId] = transceiver

    return HandlerSendResult(
      localId = localId,
      rtpParameters = sendingRtpParameters,
      // rtpSender = transceiver.sender
      rtpSender = transceiver.sender,
    )
  }

  override suspend fun sendDataChannel(
    options: SendDataChannelArguments
  ): HandlerSendDataChannelResult {
    _assertSendDirection()

    logger.traceLog("MEDIASOUP: Handlers/UnifiedPlan: sendDataChannel()")

    var dataChannel: DataChannel =
      _pc?.createDataChannel(
        label = options.label!!,
        id = _nextSendSctpStreamId,
        ordered = options.ordered!!,
        maxRetransmitTimeMs = options.maxPacketLifeTime!!,
        maxRetransmits = options.maxRetransmits!!,
        protocol = options.protocol!!,
        negotiated = true,
      )!!

    _nextSendSctpStreamId = ++_nextSendSctpStreamId % SCTP_NUM_STREAMS.MIS

    if (!_hasDataChannelMediaSection) {
      var offer: SessionDescription = _pc?.createOffer(OfferAnswerOptions())!!
      var localSdpObject: SdpObject = SDPUtils.parse(offer.sdp)
      var offerMediaObject: MediaObject? =
        localSdpObject.media.firstOrNull { m: MediaObject -> m.type == "application" }

      if (!_transportReady)
        _setupTransport(
          localDtlsRole = this._forcedLocalDtlsrole ?: DtlsRole.client,
          localSdpObject = localSdpObject,
        )

      _pc?.setLocalDescription(offer)
      _remoteSdp?.sendSctpAssociation(offerMediaObject!!)

      var answer =
        SessionDescription(SessionDescriptionUtils.fromString("answer"), _remoteSdp!!.getSdp())
      _pc?.setRemoteDescription(answer)
      _hasDataChannelMediaSection = true
    }

    var sctpStreamParameters =
      SctpStreamParameters(
        streamId = _nextSendSctpStreamId,
        ordered = options.ordered!!,
        maxPacketLifeTime = options.maxPacketLifeTime!!,
        maxRetransmits = options.maxRetransmits!!,
      )

    return HandlerSendDataChannelResult(
      dataChannel = dataChannel,
      sctpStreamParameters = sctpStreamParameters,
    )
  }

  override suspend fun setMaxSpatialLayers(options: SetMaxSpatialLayerOptions) {
    _assertSendDirection()

    logger.traceLog(
      "MEDIASOUP: Handlers/UnifiedPlan: setMaxSpatialLayer() [localId:${options.localId}, spatialLayer:${options.spatialLayer}"
    )

    var transceiver: RtpTransceiver? = _mapMidTransceiver[options.localId]

    if (transceiver == null) throw Exception("associated RTCRtpTransceiver not found")

    var parameters: RtpParameters = transceiver.sender.parameters
    var idx: Int = 0

    parameters.encodings.forEach { encoding ->
      if (idx <= options.spatialLayer) {
        encoding.active = true
      } else {
        encoding.active = false
      }

      idx++
    }

    transceiver.sender.parameters = parameters
  }

  override suspend fun setRtpEncodingParameters(options: SetRtpEncodingParametersOptions) {
    _assertSendDirection()

    logger.traceLog(
      "MEDIASOUP: Handlers/UnifiedPlan: setRtpEncodingParameters() [localId:${options.localId}, params:${options.params}]"
    )

    var transceiver: RtpTransceiver? = _mapMidTransceiver[options.localId]

    if (transceiver == null) throw Exception("associated RTCRtpTransceiver not found")

    var parameters: RtpParameters = transceiver.sender.parameters
    var idx: Int = 0

    parameters.encodings.forEach { encoding ->
      val tempEncoding: RtpEncodingParameters? = null

      tempEncoding?.setRtpEncodingParameters(
        active = if (options.params.active != null) options.params.active else encoding.active,
        maxBitrateBps = options.params.maxBitrateBps ?: encoding.maxBitrateBps,
        minBitrateBps = options.params.minBitrateBps ?: encoding.minBitrateBps,
        maxFramerate = options.params.maxFramerate ?: encoding.maxFramerate,
        numTemporalLayers = options.params.numTemporalLayers ?: encoding.numTemporalLayers,
        scaleResolutionDownBy =
          options.params.scaleResolutionDownBy ?: encoding.scaleResolutionDownBy,
        ssrc = options.params.ssrc ?: encoding.ssrc,
        // rid = options.params.rid ?: encoding.rid,
      )
    }

    transceiver.sender.parameters = parameters
    transceiver.sender.parameters = parameters
  }

  override suspend fun stopReceiving(localId: String) {
    _assertRecvDirection()

    logger.traceLog("MEDIASOUP: Handlers/UnifiedPlan: stopReceiving() [localId:$localId")

    var transceiver: RtpTransceiver? = _mapMidTransceiver[localId]

    if (transceiver == null) throw Exception("associated RTCRtpTransceiveer not found")

    _remoteSdp?.closeMediaSection(transceiver.mid)

    var offer =
      SessionDescription(SessionDescriptionUtils.fromString("offer"), _remoteSdp!!.getSdp())
    _pc?.setRemoteDescription(offer)

    var answer: SessionDescription = _pc?.createAnswer(OfferAnswerOptions())!!
    _pc?.setLocalDescription(answer)

    _mapMidTransceiver.remove(localId)
  }

  override suspend fun stopSending(localId: String, reason: String?) {
    if (reason == REASON_TRANSPORT_CLOSED || reason == REASON_DISCONNECTION_CLEANUP) {
      return
    }

    _assertSendDirection()

    logger.traceLog("MEDIASOUP: Handlers/UnifiedPlan: stopSending() [localId:$localId]")

    val transceiver: RtpTransceiver? = _mapMidTransceiver[localId]

    if (transceiver == null) throw Exception("associated RTCRtpTransceiver not found")

    transceiver.sender.replaceTrack(null)
    _pc?.removeTrack(transceiver.sender)
    val closed = _remoteSdp?.closeMediaSection(transceiver.mid)

    if (closed == true) transceiver.stop()

    val offer: SessionDescription = _pc?.createOffer(OfferAnswerOptions())!!

    _pc?.setLocalDescription(offer)

    val answer =
      SessionDescription(SessionDescriptionUtils.fromString("answer"), _remoteSdp!!.getSdp())

    _pc?.setRemoteDescription(answer)

    _mapMidTransceiver.remove(localId)
  }

  override suspend fun updateIceServers(iceServers: List<IceServer>) {
    logger.traceLog("MEDIASOUP: Handlers/UnifiedPlan: updateIceServers()")

    //        [WIP]
    //        val configuration: MutableMap<String, Any> = _pc.getConfiguration()
    //
    //        configuration["iceServers"] = iceServers
    //
    //        _pc?.setConfiguration(configuration)

    val configuration = _pc?.setConfiguration(RtcConfiguration(iceServers = iceServers))
  }
}
