package io.dyte.media

import io.dyte.webrtc.*
import io.dyte.media.handlers.*
import io.dyte.media.utils.Priority
import io.dyte.media.utils.*
import io.github.aakira.napier.Napier
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonNames


enum class Protocol {
    udp,
    tcp;

    companion object {
        private val types: Map<String, Protocol> = mapOf(
            "tcp" to tcp,
            "udp" to udp,
        )

        val values = mapOf(
            tcp to "tcp",
            udp to "udp",
        )

        fun fromString(i: String): Protocol = requireNotNull(Protocol.types[i])

        fun value(p: Protocol?): String = requireNotNull(Protocol.values[p])
    }
}

class ProduceArguments(
    val track: MediaStreamTrack,
    val stream: MediaStream,
    val encodings: List<CommonRtpEncodingParameters>,
    val codecOptions: ProducerCodecOptions? = null,
    val codec: LocalRtpCodecParameters? = null, // [WIP]
    val stopTracks: Boolean = true,
    val disableTrackOnPause: Boolean = true,
    val zeroRtpOnPause: Boolean = false,
    val appData: Map<String, Any> = emptyMap(),
    val source: String,
)

class ConsumeArguments(
    val id: String,
    val producerId: String,
    val kind: RTCRtpMediaType, // [WIP]
    val rtpParameters: LocalRtpParameters,
    val appData: Map<String, Any>,
    val accept: Function<Unit>?,
    val peerId: String,
)

enum class IceCandidateType {
    host,
    srflx,
    prflx,
    relay;

    companion object {
        private val types: Map<String, IceCandidateType> = mapOf(
            "host" to IceCandidateType.host,
            "srflx" to IceCandidateType.srflx,
            "prflx" to IceCandidateType.prflx,
            "relay" to IceCandidateType.relay,
        )

        val values = mapOf(
            IceCandidateType.host to "host",
            IceCandidateType.srflx to "srflx",
            IceCandidateType.prflx to "prflx",
            IceCandidateType.relay to "relay",
        )

        fun fromString(i: String): IceCandidateType = requireNotNull(IceCandidateType.types[i])

        fun value(i: IceCandidateType?): String = requireNotNull(IceCandidateType.values[i])
    }
}

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class IceParameters(
    @JsonNames("usernameFragment")
    var usernameFragment: String,
    @JsonNames("password")
    var password: String,
    @JsonNames("iceLite")
    var iceLite: Boolean,
)

enum class TcpType {
    active,
    passive,
    so;

    companion object {
        private val types: Map<String, TcpType> = mapOf(
            "active" to TcpType.active,
            "passive" to TcpType.passive,
            "so" to TcpType.so,
        )

        val values = mapOf(
            TcpType.active to "active",
            TcpType.passive to "passive",
            TcpType.so to "so"
        )

        fun fromString(i: String): TcpType = requireNotNull(TcpType.types[i])

        fun value(t: TcpType?): String = requireNotNull(TcpType.values[t])
    }
}

@Serializable
@OptIn(ExperimentalSerializationApi::class)
data class IceCandidate(
    @JsonNames("component")
    var component: Int = 1,
    @JsonNames("foundation")
    var foundation: String? = null, // Type Any?
    @JsonNames("priority")
    var priority: Int,
    @JsonNames("ip")
    var ip: String,
    @JsonNames("protocol")
    var protocol: Protocol? = null,
    @JsonNames("port")
    var port: Int,
    @JsonNames("type")
    var type: IceCandidateType,
    @JsonNames("tcpType")
    var tcpType: TcpType? = null,
    @JsonNames("transport")
    var transport: String,
    @JsonNames("raddr")
    var raddr: String? = null,
    @JsonNames("rport")
    var rport: Int? = null,
    @JsonNames("generation")
    var generation: Int? = null, // Type Any?
    @JsonNames("network-id")
    var networkId: Int? = null, // Type Any?
    @JsonNames("network-cost")
    var networkCost: Int? = null, // Type Any?
)

enum class DtlsRole {
    auto,
    client,
    server;

    companion object {
        private val types: Map<String, DtlsRole> = mapOf(
            "auto" to DtlsRole.auto,
            "client" to DtlsRole.client,
            "server" to DtlsRole.server,
        )

        val values = mapOf(
            DtlsRole.auto to "auto",
            DtlsRole.client to "client",
            DtlsRole.server to "server"
        )

        fun fromString(i: String): DtlsRole = requireNotNull(DtlsRole.types[i])

        fun value(d: DtlsRole?): String = requireNotNull(DtlsRole.values[d])
    }
}

class SendDataChannelArguments(
    var ordered: Boolean?,
    val maxPacketLifeTime: Int?,
    val maxRetransmits: Int?,
    val priority: Priority?,
    val label: String?,
    val protocol: String?,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class DtlsFingerprint(
    @JsonNames("algorithm")
    val algorithm: String,
    @JsonNames("value")
    val value: String,
)

@Serializable
@OptIn(ExperimentalSerializationApi::class)
class DtlsParameters(
    @JsonNames("role")
    var role: DtlsRole,
    @JsonNames("fingerprints")
    val fingerprints: List<DtlsFingerprint> = emptyList()
)

class PlainRtpParameters(
    val ip: String,
    private var _ipVersion: Int,
    val port: Int
) {
    var ipVersion: Int
        get() = _ipVersion
        set(value) {
            if (value == 4 ||value == 6) {
                _ipVersion = value
            } else throw Exception("only 4 or 6")
        }
}

enum class Direction {
    send,
    recv;

    companion object {
        private val types: Map<String, Direction> = mapOf(
            "send" to Direction.send,
            "recv" to Direction.recv
        )

        val values = mapOf(
            Direction.send to "send",
            Direction.recv to "recv"
        )

        fun fromString(i: String): Direction = requireNotNull(Direction.types[i])

        fun value(d: Direction?): String = requireNotNull(Direction.values[d])
    }
}

class CanProduceByKind(
    var audio: Boolean,
    var video: Boolean,
    var tmp: Map<String, Boolean> = emptyMap()
) {
    fun canIt(kind: RTCRtpMediaType): Boolean {
        if (kind == RTCRtpMediaType.RTCRtpMediaTypeAudio) return audio
        return video
    }
}

class EmitData(
    var name: String? = null,
    var data: Map<Any, Any>? = null
)

class Transport(
    var direction: Direction,
    var id: String,
    var iceParameters: IceParameters,
    var iceCandidates: List<IceCandidate>,
    var dtlsParameters: DtlsParameters,
    var sctpParameters: SctpParameters? = null,
    val iceServers: List<IceServer> = emptyList(),
    var iceTransportPolicy: IceTransportPolicy? = null,
    var additionalSettings: MutableMap<String, Any> = mutableMapOf(),
    val proprietaryConstraints: Map<String, Any> = emptyMap(),
    val appData: Map<String, Any> = emptyMap(),
    var extendedRtpCapabilities: ExtendedRtpCapabilities? = null,
    var canProduceByKind: CanProduceByKind,
    var producerCallback: ((Any?) -> Unit)? = null,
    var consumerCallback: ((Any?, Any?) -> Unit)? = null
) {
    private var _id: String? = null
    private var _closed: Boolean = false
    private var _direction: Direction? = null
    private var _extendedRtpCapabilities: ExtendedRtpCapabilities? = null
    private var _canProduceByKind: CanProduceByKind? = null
    private var _maxSctpMessageSize: Int? = null
    private var _handler: HandlerInterface? = null
    private var _connectionState: String = "new"
    private var _appData: Map<String, Any>? = emptyMap()
    private var _producers: MutableMap<String, Producer> = mutableMapOf()
    private var _consumers: MutableMap<String, Consumer> = mutableMapOf()
    private var _probatorConsumerCreated: Boolean = false

    @OptIn(ExperimentalCoroutinesApi::class)

    val dispatcher = Dispatchers.Default.limitedParallelism(1)
    private val transportMutex = Mutex()
    val observer = MutableSharedFlow<EmitData>()

    init {
        println("MEDIASOUP: Transport: constructor() [id:$id, direction:${Direction.value(direction)}]")

        _id = id
        _direction = direction
        _extendedRtpCapabilities = extendedRtpCapabilities
        _canProduceByKind = canProduceByKind
        _maxSctpMessageSize = if (sctpParameters != null) sctpParameters?.maxMessageSize else null

        additionalSettings.remove("iceServers")
        additionalSettings.remove("iceTransportPolicy")
        additionalSettings.remove("bundlePolicy")
        additionalSettings.remove("bundlePolicy")
        additionalSettings.remove("sdpSemantics")

        _handler = HandlerInterface.handlerFactory()

        runBlocking {
            _handler!!.run(
                options = HandlerRunOptions(
                    direction = direction,
                    iceParameters = iceParameters,
                    iceCandidates = iceCandidates,
                    dtlsParameters = dtlsParameters,
                    sctpParameters = sctpParameters,
                    iceServers = iceServers,
                    iceTransportPolicy = iceTransportPolicy,
                    additionalSettings = additionalSettings,
                    proprietaryConstraints = proprietaryConstraints,
                    extendedRtpCapabilities = extendedRtpCapabilities
                )
            )
        }

        _appData = appData

        CoroutineScope(Dispatchers.Default).async {
            _handleHandler()
        }
    }

    suspend fun _handleHandler() {
        val handler: HandlerInterface = _handler!!

        handler.observer.collect {
            when(it.name) {
                "@connect" -> {
                    println("MEDIASOUP: @connect")
                    dtlsParameters = it.data?.get("dtlsParameters") as DtlsParameters
                    println("MEDIASOUP: Emit: $dtlsParameters")
                    //val callback = it.data!!["callback"] as Function<Any?>
                    //println("MEDIASOUP: Emit: $callback")
                    //val errback = it.data!!["errback"] as (String) -> Unit

                    println("MEDIASOUP: Emit: Before IF")
                    if (_closed) {
                        println("MEDIASOUP: CLOSED")
                        //errback("closed")

                        return@collect
                    }
                    println("MEDIASOUP: Emit: After IF")
                    println("MEDIASOUP: Emit: Before Emit")
                    observer.emit(EmitData(
                        name = "connect",
                        data = mapOf(
                            "dtlsParameters" to dtlsParameters,
                            //"callback" to callback,
                            //"errback" to errback
                        )
                    ))
                    println("MEDIASOUP: Emit: Before Emit")
                }

                "@connectionstatechange" -> {
                    val connectionState: String = it.data?.get("state") as String

                    if (connectionState == _connectionState) return@collect

                    observer.emit(EmitData("connectionstatechange", mapOf("state" to connectionState)))

                    println("MEDIASOUP: Transport: connection state changed to $connectionState")
                }
            }
        }
    }

    val closed = _closed
    val handler = _handler
    val connectionState = _connectionState

    suspend fun close() {
        if (_closed) return

        println("MEDIASOUP: Transport: close()")

        _closed = true
        _handler?.close()

        for (producer: Producer in _producers.values) {
            producer.transportClosed()
        }
        _producers.clear()

        for (consumer: Consumer in _consumers.values) {
            consumer.transportClosed()
        }
        _consumers.clear()

        observer.emit(EmitData(name = "close"))
    }

    suspend fun getStats(): List<RtcStatsReport> {
        if (_closed) throw Exception("closed")

        return _handler!!.getTransportStats()
    }

    suspend fun restartIce(iceParameters: IceParameters) {
        println("MEDIASOUP: Transport: restartIce()")

        if (this._closed) {
            throw Exception("closed")
        } else if (iceParameters == null) {
            throw Exception("missing iceParameters")
        }

        dispatcher.run {
            _handler?.restartIce(iceParameters)
            println("MEDIASOUP: Transport: Dispatcher: transport.restartIce()")
        }
    }

    suspend fun updateIceServers(iceServers: List<IceServer>) {
        println("MEDIASOUP: Transport: updateIceServers()")

        if (this._closed) {
            throw Exception("closed")
        } else if (iceParameters == null) {
            throw Exception("missing iceServers")
        }

        dispatcher.run {
            _handler?.updateIceServers(iceServers)
            println("MEDIASOUP: Transport: Dispatcher: transport.updateIceServers()")
        }
    }

    private fun _handleProducer(producer: Producer) {
        CoroutineScope(Dispatchers.Default).async {
            producer.observer.collect {
                when (it.name) {
                    "@close" -> {
                        _producers.remove(producer.id)

                        if (_closed) return@collect

                        dispatcher.run {
                            try {
                                _handler?.stopSending(producer.localID)
                            } catch (e: Error) {
                                Napier.w("Transport: Dispatcher: producer.close() failed:${e.toString()}")
                            }

                            println("MEDIASOUP: Transport: Dispatcher: producer @close event")
                        }
                    }

                    "@replacetrack" -> {
                        val track: MediaStreamTrack = it.data?.get("track") as MediaStreamTrack
                        val callback = it.data!!["callback"] as (Any?) -> Unit
                        val errback = it.data!!["errback"] as (String) -> Unit

                        dispatcher.run {
                            try {
                                _handler?.replaceTrack(
                                    ReplaceTrackOptions(
                                        localId = producer.localID,
                                        track = track
                                    )
                                )

                                callback.invoke(it.name)
                            } catch (e: Error) {
                                errback.invoke(e.toString())
                            }
                            println("MEDIASOUP: Transport: Dispatcher: producer @replacetrack event")
                        }
                    }

                    "@setmaxspatiallayer" -> {
                        val spatialLayer = it.data?.get("spatialLayer") as Int
                        val callback = it.data!!["callback"] as (Any?) -> Unit
                        val errback = it.data!!["errback"] as (String) -> Unit

                        dispatcher.run {
                            try {
                                _handler?.setMaxSpatialLayers(
                                    SetMaxSpatialLayerOptions(
                                        localId = producer.localID,
                                        spatialLayer = spatialLayer
                                    )
                                )
                            } catch (e: Error) {
                                errback.invoke(e.toString())
                            }

                            println("MEDIASOUP: Transport: Dispatcher: producer @setmaxspatiallayer event")
                        }
                    }

                    "@setrtpencodingparameters" -> {
                        val params = it.data?.get("params") as RtpEncodingParameters
                        val callback = it.data?.get("callback") as (Any?) -> Unit
                        val errback = it.data?.get("errback") as (String) -> Unit

                        dispatcher.run {
                            try {
                                _handler?.setRtpEncodingParameters(
                                    SetRtpEncodingParametersOptions(
                                        localId = producer.localID,
                                        params = params
                                    )
                                )
                            } catch (e: Error) {
                                errback.invoke(e.toString())
                            }

                            println("MEDIASOUP: Transport: Dispatcher: producer @setrtpencodingparameters event")
                        }
                    }

                    "@getstats" -> {
                        val callback = it.data?.get("callback") as (Any?) -> Unit
                        val errback = it.data?.get("errback") as (String) -> Unit

                        if (_closed) return@collect errback("close")

                        _handler?.getSenderStats(producer.localID)?.asFlow()?.onCompletion {
                            callback.invoke(it?.message.toString())
                        }?.catch {
                            errback.invoke(it.message.toString())
                        }
                    }
                }
            }
        }
    }

    private fun _handleconsumer(consumer: Consumer) {
        CoroutineScope(Dispatchers.Default).async {
            consumer.observer.collect {
                when (it.name) {
                    "@close" -> {
                        _consumers.remove(consumer.id)

                        if (_closed) return@collect

                        dispatcher.run {
                            _handler?.stopReceiving(consumer.localID)

                            println("MEDIASOUP: Transport: Dispatcher: consumer @close event")
                        }
                    }

                    "@getstats" -> {
                        val callback = it.data?.get("callback") as (Any?) -> Unit
                        val errback = it.data?.get("errback") as (String) -> Unit

                        if (_closed) return@collect errback("closed")

                        _handler?.getReceiverStats(consumer.localID)?.asFlow()?.onCompletion {
                            callback.invoke(it?.message.toString())
                        }?.catch {
                            errback.invoke(it.message.toString())
                        }
                    }
                }
            }
        }
    }

    private suspend fun __produce(arguments: ProduceArguments): Producer {

        var returnProducer: Producer? = null

        try {
            var normalizedEncodings: MutableList<CommonRtpEncodingParameters> = mutableListOf()

            if (arguments.encodings != null && arguments.encodings.isEmpty()) {
                normalizedEncodings = mutableListOf()
            } else if (arguments.encodings != null && arguments.encodings.isNotEmpty()) {
//                [WIP] Missing constructor in WebRTC? De-commonize?
                arguments.encodings.forEach {
                    //val normalizedEncoding = RtpEncodingParameters(active = true)

                    var normalizedEncoding = CommonRtpEncodingParameters()
                    normalizedEncoding.active = true


                    normalizedEncoding.active = it.active ?: true
                    normalizedEncoding.scaleResolutionDownBy = it.scaleResolutionDownBy
                    normalizedEncoding.maxBitrateBps = it.maxBitrateBps
                    normalizedEncoding.maxFramerate = it.maxFramerate
                    normalizedEncoding.bitratePriority = it.bitratePriority
                    normalizedEncoding.networkPriority = it.networkPriority


                    normalizedEncodings.add(normalizedEncoding)
                }
            }
            println("MEDIASOUP: _produce() 0 ")
            val sendResult: HandlerSendResult = _handler!!.send(HandlerSendOptions(
                track = arguments.track,
                encodings = normalizedEncodings,
                codecOptions = arguments.codecOptions,
                codec = arguments.codec,
                stream = arguments.stream
            ))
            println("FixMid: Mid = ${sendResult.rtpParameters.mid} for ${arguments.track.kind}")
            println("MEDIASOUP: _produce() 1")
            try {
                Ortc.validateRtpParameters(sendResult.rtpParameters)

                val id: String = observer.emit(EmitData(
                    name = "produce",
                    data = mapOf(
                        "kind" to arguments.track.kind,
                        "rtpParameters" to sendResult.rtpParameters,
                        "appData" to arguments.appData
                    )
                )).toString()

                println("MEDIASOUP: Produce: Produce event emitted")

                val producer = Producer(
                    id = id,
                    localID = sendResult.localId,
                    rtpSender = sendResult.rtpSender,
                    track = arguments.track,
                    rtpParameters = sendResult.rtpParameters,
                    stopTracks = arguments.stopTracks,
                    disableTrackOnPause = arguments.disableTrackOnPause,
                    zeroRtpOnPause = arguments.zeroRtpOnPause,
                    appData = arguments.appData,
                    stream = arguments.stream,
                    source = arguments.source
                )
                println("MEDIASOUP: _produce() 2")

                _producers[producer.id] = producer
                _handleProducer(producer)
                observer.emit(EmitData(
                    name = "newProducer",
                    data = mapOf(
                        "producer" to producer
                    )
                ))
                println("MEDIASOUP: _produce() pre")
                producerCallback?.let { it(producer) }
                returnProducer = producer

                println("MEDIASOUP: _produce() post")
            } catch (e: Error) {
                _handler!!.stopSending(sendResult.localId)

                throw e
            }
        } catch (e: Error) {
            if (arguments.stopTracks) {
                try {
                    arguments.track.stop()
                } catch (e2: Error) {}
            }

            throw e
        }

        return returnProducer
    }

    private suspend fun _produce(arguments: ProduceArguments): Producer {
        return transportMutex.withLock {
            __produce(arguments)
        }
    }


    suspend fun produce(
        track: MediaStreamTrack,
        stream: MediaStream,
        encodings: List<CommonRtpEncodingParameters> = emptyList(),
        codecOptions: ProducerCodecOptions? = null,
        codec: LocalRtpCodecParameters? = null,
        stopTracks: Boolean = true,
        disableTrackOnPause: Boolean = true,
        zeroRtpOnPause: Boolean = false,
        appData: Map<String, Any> = emptyMap(),
        source: String
    ): Producer {
        println("MEDIASOUP: Transport: produce() [track:${track.toString()}")

        if (_direction != Direction.send) {
            throw Exception("Not a sending transport")
        } else if (track.kind == null || !_canProduceByKind?.canIt(RTCRtpMediaType.fromString(track.kind.toString()))!!) {
            throw Exception("cannot produce ${track.kind}")
        }
//        Listeners? How?
//        else if (listeners("connect").isEmpty() && _connectionState == "new") {
//            throw Exception("no connect listener set into this transport")
//        } else if (listeners("produce").isEmpty()) {
//            throw Exception("no pruduce listener set into this transport")
//        }

        var returnProducer: Producer? = null

        dispatcher.run {
            returnProducer = _produce(ProduceArguments(
                track = track,
                stream = stream,
                encodings = encodings,
                codecOptions = codecOptions,
                codec = codec,
                stopTracks = stopTracks,
                disableTrackOnPause = disableTrackOnPause,
                zeroRtpOnPause = zeroRtpOnPause,
                appData = appData,
                source = source
            ))

            println("MEDIASOUP: Transport: Dispatcher: transport.produce()")
        }

        return returnProducer!!
    }

    suspend fun __consume(arguments: ConsumeArguments): Consumer {
        println("FixConsumer: inside _consume")

        var returnConsumer: Consumer? = null

        val canConsume: Boolean = Ortc.canReceive(arguments.rtpParameters, _extendedRtpCapabilities)

        println("FixConsumer: canConsumer $canConsume")

        if (!canConsume) throw Exception("Cannot consume this Consumer")

        val receiveResult: HandlerReceiveResult = _handler!!.receive(
            HandlerReceiveOptions(
                trackId = arguments.id,
                kind = arguments.kind,
                rtpParameters = arguments.rtpParameters
            )
        )

        println("FixConsumer: Received result from handler")

        val consumer = Consumer(
            id = arguments.id,
            localID = receiveResult.localId,
            producerID = arguments.producerId,
            rtpParameters = arguments.rtpParameters,
            appData = arguments.appData,
            track = receiveResult.track,
            rtpReceiver = receiveResult.rtpReceiver,
            stream = receiveResult.stream,
            peerID = arguments.peerId
        )

        _consumers[consumer.id] = consumer

        _handleconsumer(consumer)

        println("FixConsumer: Handled Consumer")

//        if (!_probatorConsumerCreated && arguments.kind == RTCRtpMediaType.RTCRtpMediaTypeVideo) {
//            try {
//                val probatorRtpParameters = Ortc.generateProbatorRtpParameters(consumer.rtpParameters)
//
//                _handler!!.receive(HandlerReceiveOptions(
//                    trackId = "probator",
//                    kind = RTCRtpMediaType.RTCRtpMediaTypeVideo,
//                    rtpParameters = probatorRtpParameters
//                ))
//
//                println("MEDIASOUP: Transport: consume() | Consumer for RTP probation created")
//
//                _probatorConsumerCreated = true
//            } catch (e: Error) {
//                Napier.e("Transport: consume() | failed to create Consumer for RTP probation:${e.toString()}")
//            }
//        }

        println("FixConsumer: Before emit newconsumer")

        observer.emit(EmitData(
            name = "newconsumer",
            data = mapOf(
                "consumer" to consumer
            )
        ))

        println("FixConsumer: After emit newconsumer")

        //consumerCallback!!.invoke(consumer, arguments.accept)

        returnConsumer = consumer

        println("FixConsumer: returnConsumer $returnConsumer.id")

        return returnConsumer
    }

    private suspend fun _consume(arguments: ConsumeArguments): Consumer {
        return transportMutex.withLock {
            __consume(arguments)
        }
    }

    suspend fun consume(
        id: String,
        producerId: String,
        peerId: String,
        kind: RTCRtpMediaType,
        rtpParameters: LocalRtpParameters,
        appData: Map<String, Any> = emptyMap(),
        accept: ((Any?, Any?) -> Unit)? = null
    ): Consumer {
        println("MEDIASOUP: Transport: consume()")

        if (_closed) {
            throw Exception("closed")
        } else if (_direction != Direction.recv) {
            throw Exception("Not a receiving transport")
        } else if (kind != RTCRtpMediaType.RTCRtpMediaTypeAudio && kind != RTCRtpMediaType.RTCRtpMediaTypeVideo) {
            throw Exception("invalid kind ${RTCRtpMediaType.value(kind)}")
        }
//        Listeners? How and why?
//        else if (listeners("connect").isEmpty() && _connectionState == "new") {
//            throw Exception("no connect listener set into this transport")
//        }

        var returnConsumer: Consumer? = null

        dispatcher.run {
             returnConsumer = _consume(ConsumeArguments(
                id = id,
                producerId = producerId,
                kind = kind,
                rtpParameters = rtpParameters,
                appData = appData,
                accept = accept,
                peerId = peerId
            ))

            println("MEDIASOUP: Transport: Dispatcher: transport.consume()")
        }

        return returnConsumer!!
    }
}