package io.dyte.media.utils.sdp

import SdpGrammar
import grammar
import io.dyte.media.SdpObject
import io.dyte.media.utils.append
import io.dyte.media.utils.extend
import io.dyte.media.utils.update
import io.github.aakira.napier.Napier
import kotlinx.serialization.json.*

val json = Json {
    ignoreUnknownKeys = true
}

class ParseRegWrapper(
    var obj: SDPObject,
    var location: JsonObject,
    var content: String
)

class AttachPropertiesWrapper(
    var match: Sequence<MatchResult>,
    var location: JsonObject,
    var rawName: String?,
    var fields: List<Field>?
)

fun toIntIfInt(v: String?): Int? {
    if (v != null) {
        try {
            v.toInt()
        } catch(e: Exception) {
            try {
                v.toLong()
                println("ERROR toInt failed but toLong succeeded for ${v}")
            } catch (e: Exception) {

            }
             return null
        }

        return v.toInt()
    }

    return null
}

class SDPUtils {
    companion object{
        private val responseModel = json.decodeFromString(SDPWrapper.serializer(), grammar)

        private fun attachProperties(ap: AttachPropertiesWrapper) {
            if (ap.rawName != null && (ap.fields == null || ap.fields!!.size <= 1)) {
                ap.match.forEach {
                    val elem = if (it.groups.size == 0) it.value else it.groups[1]?.value;
                    val field = ap.fields!!.get(0);
                    when(field.type) {
                        "String" -> if(elem != null && ap.location[ap.rawName!!] == null) {
                            ap.location = ap.location.update(ap.rawName!!, elem!!)
                        }
                        "Int" -> {
                            val elemInt = toIntIfInt(elem);
                            if(elemInt != null) {
                                ap.location = ap.location.update(ap.rawName!!, elemInt)
                            }
                        }
                    }
                }
            } else if (ap.fields != null) {
                ap.match.forEach {
                    for(i in 0..it.groups.size){
                        val field = ap.fields!!.getOrNull(i);
                        if(field != null){
                            val elem = it.groups.get(i+1)?.value
                            when(field.type) {
                                "String" -> if(elem != null) {
                                    ap.location = ap.location.update(field.name, elem!!)
                                }
                                "Int" -> {
                                    val elemInt = toIntIfInt(elem);
                                    if(elemInt != null) {
                                        ap.location = ap.location.update(field.name, elemInt)
                                    }
                                }
                                "Long" -> {
                                    ap.location = ap.location.update(field.name, elem!!.toLong())
                                }
                            }
                        }
                    }
                }
            }
        }

        private fun parseReg(pg: ParseRegWrapper) {
            var needsBlank = pg.obj.name != null

            if (pg.obj.push != null && pg.location[pg.obj.push] == null) {
                pg.location = pg.location.update(pg.obj.push!!, buildJsonArray {  })
            } else if (needsBlank && pg.location[pg.obj.name] == null) {
                pg.location = pg.location.update(pg.obj.name!!, buildJsonObject {  })
            }
            val emptyJsonObject = buildJsonObject {  }

            var keyLocation: JsonObject
            try {
            keyLocation = if (pg.obj.push != null) emptyJsonObject else
                (if (needsBlank) pg.location[pg.obj.name] else pg.location) as JsonObject
                }catch (e: Exception) {
                    println("SDP duplicate ${pg.obj.name}")
                    return;
                }
            val attachProps = AttachPropertiesWrapper(Regex(pg.obj.reg).findAll(pg.content), keyLocation!!, pg.obj.name, pg.obj.fields)

            attachProperties(attachProps)

            keyLocation = attachProps.location

            if (pg.obj.push != null) {
                //(location[obj.push] as MutableMap<String, Any?>) += keyLocation
                var curr = pg.location[pg.obj.push] as JsonArray;
                curr = curr.append(keyLocation)
                pg.location = pg.location.update(pg.obj.push!!, curr)
            } else {
                if(needsBlank && pg.obj.fields.size > 1) {
                    val newInternal = (pg.location[pg.obj.name] as JsonObject).extend(keyLocation)
                    pg.location = pg.location.update(pg.obj.name!!, newInternal)
                } else {
                    pg.location = pg.location.extend(keyLocation)
                }
            }
        }

        fun parse(sdp: String): SdpObject {
            var session = buildJsonObject {  }
            var medias = buildJsonArray {  }
            var location: JsonObject = session
            var mIndex = -1

            sdp.split("\n","\r\n").forEach {
                if (it != "") {
                    val type = it[0].toString()
                    var content = it.substring(2)

                    if (type == "m") {
                        if(mIndex == -1) {
                            session = session.extend(location)
                        }
                        var media = buildJsonObject {  }

                        media = media.update("rtp", buildJsonArray {  })
                        media = media.update("fmtp", buildJsonArray {  })
                        mIndex = medias.size
                        medias = medias.append(media)

                        location = media

                    }

                    if (responseModel.data[type] != null) {
                        for (j in 0 until (responseModel.data[type]!!.size)) {
                            var obj = responseModel.data[type]?.get(j)

                            if (obj?.reg == null) {
                                if (obj?.name != null) {
                                    location = location.update(obj.name!!, content)
                                } else {
                                    println("MEDIASOUP: SDPUtils: Trying to add null key")
                                }

                                continue
                            }

                            if (Regex(obj.reg).containsMatchIn(content)) {
                                val parseReg = ParseRegWrapper(obj, location, content)

                                parseReg(parseReg)
                                obj = parseReg.obj
                                location = parseReg.location
                                content = parseReg.content
                            }
                        }

                        if (location["invalid"] == null) location = location.update("invalid", buildJsonArray {  })

                        val tmp = buildJsonObject {
                            put("value", content)
                        }
//                        var invalidVal = location["invalid"] as JsonArray;
//                        invalidVal = invalidVal.append(tmp)
//                        location = location.update("invalid", invalidVal)
                        if(mIndex >= 0) medias = medias.update(mIndex, location)
                    } else {
                        println("MEDIASOUP: SDPUtils: ERROR unknown grammar type $type")
                    }
                }
            }

            session = session.update("media", medias);
            return Json{ ignoreUnknownKeys = true }.decodeFromJsonElement<SdpObject>(session)
        }

        fun parseParams(str: String): MutableMap<String, JsonPrimitive?> {
            val params: MutableMap<String, JsonPrimitive?> = mutableMapOf()

            str.split(Regex(";").pattern).forEach {
                val idx: Int = it.indexOf("=")
                val key: String
                var value: String? = null

                if (idx == -1) {
                    key = it
                } else {
                    key = it.substring(0, idx).trim()
                    value = it.substring(idx + 1, it.length).trim()
                }
                val intValue = toIntIfInt(value)
                if(intValue != null) {
                    params[key] = JsonPrimitive(intValue)
                } else {
                    params[key] = JsonPrimitive(value)
                }
            }

            return params
        }

        private val innerOrder = listOf<String>("i'", "c", "b", "a")
        private val outerOrder = listOf<String>("v", "o", "s", "i", "u", "e", "p", "c", "b", "t", "r", "z", "a")

        fun format(formatString: String, args: String): String {
            return formatString.replace(Regex("%d|%v|%s"), args)
        }

        fun makeLine(type: String, obj: SDPObject, location: JsonObject): String {
            var str: String = ""

            if(obj.format != null) {
                var format = obj.format
                if(format == "{value}") str = "${location[obj.name]}"
                else {
                    str = format.toString();
                    obj.fields.forEach {
                        val e = location[it.name]
                        if(e != null) {
                            str = str.replace("{${it.name}}", e.toString())
                        }
                    }
                }
            } else {
                try {
                    str = "${location[obj.name]}"
                } catch (e: Error) {
                    Napier.e("SDPUtils: makeline(): ${e.toString()}")
                }
            }

            var formatStr = "$type=$str"
            var args = "${location[obj.name]}" ?: ""

            return format(formatStr, args)
        }

        fun write(sdpObject: SdpObject): String {
            return sdpObject.write()
        }
    }
}
