package io.dyte.media.hive.handlers

import io.dyte.webrtc.*
import io.dyte.media.hive.HiveEmitData
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.runBlocking

class HiveUnifiedPlan: HiveHandlerInterface() {
  /** Handler Direction */
  private lateinit var _direction: RtpTransceiverDirection

  /** RTCPeerConnection instance */
  private lateinit var _pc: PeerConnection

  /** Local stream for sending general media tracks */
  private lateinit var _sendStream: MediaStream

  /** Local stream for sending screen share tracks */
  private lateinit var _sendScreenShareStream: MediaStream

  /** Got transport local and remote parameters */
  private var _transportReady = false

  val observer = MutableSharedFlow<HiveEmitData>()

  init {
    runBlocking {
      _sendStream = MediaDevices.getUserMedia(audio = true, video = true)
      _sendScreenShareStream = MediaDevices.getDisplayMedia()
    }
  }

  companion object {
    fun createFactory(): HiveHandlerFactory = { HiveUnifiedPlan() }
  }

  override fun getName() = "UnifiedPlan"

  override fun getPc() = this._pc

  override fun close() {
    println("DyteMediaClient: UnifiedPlan: close()")

    if (this::_pc.isInitialized) {
      try {
        this._pc.close()
      } catch (e: Error) {
        println("DyteMediaClient: UnifiedPlan: pc.close() $e")
      }
    }
  }

  override suspend fun init(options: HiveHandlerRunOptions) {
    println("DyteMediaClient: UnifiedPlan: init()")

    this._direction = options.direction

    @Suppress("UNCHECKED_CAST")
    this._pc = PeerConnection(RtcConfiguration(
      iceServers = options.additionalSettings?.get("iceServers") as? List<IceServer>
        ?: options.iceServers ?: emptyList(),
      iceTransportPolicy = options.additionalSettings?.get("iceTransportPolicy") as? IceTransportPolicy
        ?: options.iceTransportPolicy ?: IceTransportPolicy.All,
      bundlePolicy = options.additionalSettings?.get("bundlePolicy") as? BundlePolicy ?: BundlePolicy.MaxBundle,
      rtcpMuxPolicy = options.additionalSettings?.get("rtcpMuxPolicy") as? RtcpMuxPolicy  ?: RtcpMuxPolicy.Require,
//      sdpSemantics = "unified-plan"
//      proprietaryConstraints
    ))

    this._pc.onTrack.collect {
      options.onTrackHandler(it)
    }

    this._addEventListeners()
  }

  override suspend fun connect(): HiveGenericHandlerResult {
    /** Since we have enabled bundlePolicy to MaxBundle,
     *  we need to have max-bundle attribute in SDP and to generate
     *  that we are using a hacky fix by creating a temp data channel
     */
    val dc = this._pc.createDataChannel(
      label = "dyte"
    )

    val offer = this._pc.createOffer(OfferAnswerOptions())

    println("DyteMediaClient: UnifiedPlan: connect  offer with sdp = ${offer.sdp}")

    this._pc.setRemoteDescription(offer)

    val callback: suspend (SessionDescription) -> Unit = { answer ->
      println("DyteMediaClient: UnifiedPlan: connect(): Calling pc.setRemoteDescription() with answer=${answer.sdp}")

      this._pc.setRemoteDescription(answer)
      dc?.close()
    }

    return HiveGenericHandlerResult(
      offerSdp = offer,
      callback = callback
    )
  }

  override suspend fun updateIceServers(iceServers: List<IceServer>) {
    println("DyteMediaClient: UnifiedPlan: updateIceServers()")

//    val configuration = this._pc.getConfiguration()
//    configuration.iceServers = iceServers
//    this._pc.setConfiguration(configuration)

    // TODO: Modify to retain other configuration parameters or implement getConfiguration
    this._pc.setConfiguration(RtcConfiguration(iceServers = iceServers))
  }

  override suspend fun restartIce(): HiveGenericHandlerResult {
    println("DyteMediaClient: UnifiedPlan: restartIce()")

    val offer = this._pc.createOffer(OfferAnswerOptions(iceRestart = true))

    val callback: suspend (SessionDescription) -> Unit = { answer ->
      println("DyteMediaClient: UnifiedPlan: restartIce(): Calling pc.setRemoteDescription() with answer=${answer.sdp}")

      this._pc.setRemoteDescription(answer)
    }

    return HiveGenericHandlerResult(
      offerSdp = offer,
      callback = callback
    )
  }

  override suspend fun getTransportStats(): RtcStatsReport = this._pc.getStats()!!

  override suspend fun send(options: HiveHandlerSendOptions): HiveHandlerSendResult {
    this._assertSendDirection()

    println("DyteMediaClient: UnifiedPlan: send() with kind = ${options.track.kind} with trackId = ${options.track.id}")

    println("DyteMediaClient: UnifiedPlan: send(): Creating new transceiver")

    val commonEncodings = mutableListOf<CommonRtpEncodingParameters>()

    options.encodings?.forEach {
      val encoding = CommonRtpEncodingParameters()

      encoding.rid = it.rid
      encoding.active = it.active
      encoding.bitratePriority = it.bitratePriority
      encoding.networkPriority = it.networkPriority
      encoding.maxBitrateBps = it.maxBitrateBps
      encoding.minBitrateBps = it.minBitrateBps
      encoding.maxFramerate = it.maxFramerate
      encoding.numTemporalLayers = it.numTemporalLayers
      encoding.scaleResolutionDownBy = it.scaleResolutionDownBy
      encoding.ssrc = it.ssrc

      commonEncodings.add(encoding)
    }

    // TODO: Return the created transceiver from addTransceiver() itself
    this._pc.addTransceiver(
      track = options.track,
      kind = options.track.kind,
      init = CommonRtpTransceiverInit(
        direction = RtpTransceiverDirection.SendOnly,
        streams = if (options.screenShare != null && options.screenShare) listOf(this._sendScreenShareStream)
        else listOf(this._sendStream),
        sendEncodings = commonEncodings
      )
    )

    val trans = this._pc.getTransceivers().last()

    // Note(anunaym14): Similar to Firefox, WebRTC-KMP doesn't support RtpTransceiver.setCodecPreferences()
    // So, let's enforce SFU to only use VP8 in server side.

    val offer = this._pc.createOffer(OfferAnswerOptions())

    if (!this._transportReady) throw Error("WebRTC Transport not connected")

    this._pc.setLocalDescription(offer)

    /** mangle offer with new fmtp parameters */
    val updatedOffer = SessionDescription(
      type = offer.type,
      sdp = offer.sdp.replace("minptime=10;useinbandfec=1",
        "minptime=10;useinbandfec=1;usedtx=1;maxaveragebitrate=16000")
    )

    val setAnswer: suspend (SessionDescription) -> String = { answer ->
      println("DyteMediaClient: UnifiedPlan: send(): Calling pc.setRemoteDescription() with answer=${answer.sdp}")

      // Store in the map
      this.mapMidTransceiver[trans.mid] = trans

      trans.mid
    }

    return HiveHandlerSendResult(
      offerSdp = updatedOffer,
      callback = setAnswer
    )
  }

  override suspend fun stopSending(localId: String): HiveGenericHandlerResult {
    this._assertSendDirection()

    println("DyteMediaClient: UnifiedPlan: stopSending for localId: $localId")

    val transceiver = this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    transceiver.sender.replaceTrack(null)
    this._pc.removeTrack(transceiver.sender)

    transceiver.direction = RtpTransceiverDirection.Inactive

    val offer = this._pc.createOffer(OfferAnswerOptions())

    println("DyteMediaClient: calling pc.setLocalDescription() with offer = ${offer.sdp}")

    this._pc.setLocalDescription(offer)

    val callback: suspend (SessionDescription) -> Unit = { answer ->
      println("DyteMediaClient: UnifiedPlan: stopSending(): Calling pc.setRemoteDescription() with answer=${answer.sdp}")

      this._pc.setRemoteDescription(answer)

      this.mapMidTransceiver.remove(localId)
    }

    return HiveGenericHandlerResult(
      offerSdp = offer,
      callback = callback
    )
  }

  override suspend fun replaceTrack(localId: String, track: MediaStreamTrack?) {
    this._assertSendDirection()

    if (track != null) println("DyteMediaClient: UnifiedPlan: replaceTrack() with localId=$localId, trackId=${track.id}")
    else println("DyteMediaClient: UnifiedPlan: replaceTrack() localId=$localId with no track")

    val transceiver = this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    transceiver.sender.replaceTrack(track)
  }

  override suspend fun setMaxSpatialLayer(localId: String, spatialLayer: Long) {
    this._assertSendDirection()

    println("DyteMediaClient: UnifiedPlan: setMaxSpatialLayer() with localId=$localId spatialLayer=$spatialLayer")

    val transceiver = this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    val parameters = transceiver.sender.parameters

    parameters.encodings.forEachIndexed { idx, encoding ->
      // TODO: Verify if changing active to var changes it internally
      encoding.active = idx <= spatialLayer
    }

    transceiver.sender.parameters = parameters
  }

  override suspend fun setRtpEncodingParameters(localId: String, params: Any) {
    this._assertSendDirection()

    println("DyteMediaClient: UnifiedPlan: setRtpEncodingParameters() with localId=$localId params=$params")

    val transceiver = this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    val parameters = transceiver.sender.parameters

    // TODO: This is required for simulcast which Hive doesn't support right now, fix for future
    //parameters.encodings.add(params as RtpEncodingParameters)

    transceiver.sender.parameters = parameters
  }

  override fun getSenderStats(localId: String): List<RtcStatsReport> {
    this._assertSendDirection()

    val transceiver = this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    // TODO: Implement sender getStats()
//    return transceiver.sender.getStats()
    return emptyList()
  }

  override suspend fun stopReceiving(localId: String): HiveGenericHandlerResult {
    this._assertRecvDirection()

    println("DyteMediaClient: UnifiedPlan: stopReceiving() with localId=$localId")

    throw Error("Method not implemented")
  }

  override suspend fun pauseReceiving(localId: String): HiveGenericHandlerResult {
    this._assertRecvDirection()

    println("DyteMediaClient: UnifiedPlan: pauseReceiving() with localId=$localId")

    throw Error("Method not implemented")
  }

  override suspend fun resumeReceiving(localId: String): HiveGenericHandlerResult {
    this._assertRecvDirection()

    println("DyteMediaClient: UnifiedPlan: resumeReceiving() with localId=$localId")

    throw Error("Method not implemented")
  }

  override suspend fun getReceiverStats(localId: String): List<RtcStatsReport> {
    this._assertRecvDirection()

    val transceiver = this.mapMidTransceiver[localId] ?: throw Error("Associated RtpTransceiver not found")

    // TODO: Implement receiver getStats()
//    return transceiver.receiver.getStats()
    return emptyList()
  }

  private fun _assertSendDirection() {
    if (this._direction != RtpTransceiverDirection.SendOnly) {
      throw Error("Method can just be called for handlers with send direction")
    }
  }

  private fun _assertRecvDirection() {
    if (this._direction != RtpTransceiverDirection.RecvOnly) {
      throw Error("Method can just be called for handlers with recv direction")
    }
  }

  private suspend fun _generateOffer(): SessionDescription {
    val offer = this._pc.createOffer(OfferAnswerOptions())

    this._pc.setLocalDescription(offer)

    return offer
  }

  private suspend fun _setAnswer(answer: SessionDescription) {
    println("DyteMediaClient: UnifiedPlan: _setAnswer() | calling pc.setRemoteDescription() with answer=${answer.sdp}")

    this._pc.setRemoteDescription(answer)
  }

  private suspend fun _addEventListeners() {
    this._pc.onIceConnectionStateChange.collect {
      when (it) {
        IceConnectionState.Checking -> this.observer.emit(HiveEmitData(
          eventName = "@connectionstatechange",
          data = "connecting"
        ))

        IceConnectionState.Connected -> this.observer.emit(HiveEmitData(
          eventName = "@connectionstatechange",
          data = "connected"
        ))

        IceConnectionState.Completed -> this.observer.emit(HiveEmitData(
          eventName = "@connectionstatechange",
          data = "connected"
        ))

        IceConnectionState.Failed -> this.observer.emit(HiveEmitData(
          eventName = "@connectionstatechange",
          data = "failed"
        ))

        IceConnectionState.Disconnected -> this.observer.emit(HiveEmitData(
          eventName = "@connectionstatechange",
          data = "failed"
        ))

        IceConnectionState.Closed -> this.observer.emit(HiveEmitData(
          eventName = "@connectionstatechange",
          data = "closed"
        ))

        else -> println("Unknown IceConnectionState")
      }
    }

    this._pc.onIceCandidate.collect {
      this.observer.emit(HiveEmitData(
        eventName = "@icecandidate",
        data = it.candidate
      ))
    }

    this._pc.onNegotiationNeeded.collect {
      this.observer.emit(HiveEmitData("@negotiationneeded"))
    }

    this._pc.onIceGatheringState.collect {
      when (it) {
        IceGatheringState.Gathering -> {
          println("IceGatheringStateChange | Gathering")

          this.observer.emit(HiveEmitData(
            eventName = "@icegatheringstatechange",
            data = IceGatheringState.Gathering
          ))
        }

        IceGatheringState.Complete -> {
          println("IceGatheringStateChange | Complete")

          this.observer.emit(HiveEmitData(
            eventName = "@icegatheringstatechange",
            data = IceGatheringState.Complete
          ))
        }

        else -> println("Unknown IceGatheringState")
      }
    }

    // this._pc.onIceCandidateError.collect {}

    this._pc.onDataChannel.collect {
      println("Data channel created: ${it.label}")

      val channel = it

      channel.onOpen.collect {
        println("Data channel open: ${channel.label}")
      }

      channel.onClose.collect {
        println("Data channel closed: ${channel.label}")
      }

      channel.onError.collect {error ->
        println("Data channel error: $error for ${channel.label}")
      }

      channel.onMessage.collect {msg ->
        println("Data channel meswsage: $msg for ${channel.label}")

        this.observer.emit(HiveEmitData(
          eventName = "datachannel",
          data = mapOf(
            "dataChannel" to channel,
            "message" to msg.toString()
          )
        ))
      }
    }
  }
}