package io.dyte.media.handlers

import io.dyte.media.*
import io.dyte.media.IceCandidate
import io.dyte.media.utils.*
import io.dyte.webrtc.*
import kotlinx.coroutines.flow.MutableSharedFlow

class SCTP_NUM_STREAMS {
  companion object {
    const val OS: Int = 1024
    const val MIS: Int = 1024
  }
}

class RTCOAuthCredential(val accessToken: String, val macKey: String)

enum class RTCIceCredentialType {
  oauth,
  password;

  companion object {
    private val types: Map<String, RTCIceCredentialType> =
      mapOf(
        "oauth" to RTCIceCredentialType.oauth,
        "password" to RTCIceCredentialType.password,
      )

    val values =
      mapOf(RTCIceCredentialType.oauth to "oauth", RTCIceCredentialType.password to "password")

    operator fun get(i: String): RTCIceCredentialType =
      requireNotNull(RTCIceCredentialType.types[i])

    fun value(r: RTCIceCredentialType?): String = requireNotNull(RTCIceCredentialType.values[r])
  }
}

enum class RTCIceTransportPolicy {
  all,
  relay;

  companion object {
    private val types: Map<String, RTCIceTransportPolicy> =
      mapOf("all" to RTCIceTransportPolicy.all, "relay" to RTCIceTransportPolicy.relay)

    val values =
      mapOf(
        RTCIceTransportPolicy.all to "all",
        RTCIceTransportPolicy.relay to "relay",
      )

    operator fun get(i: String): RTCIceTransportPolicy =
      requireNotNull(RTCIceTransportPolicy.types[i])

    fun value(r: RTCIceCredentialType?): String = requireNotNull(RTCIceCredentialType.values[r])
  }
}

class RTCIceServer(
  val credential: Any? = null,
  val credentialType: RTCIceCredentialType,
  val urls: List<String> = emptyList(),
  val username: String
) {
  fun toMap(): Map<String, Any?> {
    val result: MutableMap<String, Any?> = mutableMapOf()

    if (credential != null) result["credential"] = credential
    result["credentialType"] = RTCIceCredentialType.value(credentialType)
    result["urls"] = urls
    result["username"] = username

    return result
  }
}

class HandlerRunOptions(
  var direction: Direction,
  var iceParameters: IceParameters,
  var iceCandidates: List<IceCandidate>,
  var dtlsParameters: DtlsParameters,
  var sctpParameters: SctpParameters? = null,
  var iceServers: List<IceServer>,
  var iceTransportPolicy: IceTransportPolicy? = null,
  var additionalSettings: Any? = null,
  var proprietaryConstraints: Map<String, Any>,
  var extendedRtpCapabilities: ExtendedRtpCapabilities? = null
)

class HandlerSendResult(
  val localId: String,
  val rtpParameters: LocalRtpParameters,
  val rtpSender: RtpSender? = null,
)

class HandlerSendOptions(
  val track: MediaStreamTrack,
  val encodings: List<CommonRtpEncodingParameters> = emptyList(),
  val codecOptions: ProducerCodecOptions? = null,
  val codec: LocalRtpCodecParameters? = null,
  val stream: MediaStream
)

class HandlerSendDataChannelResult(
  val dataChannel: DataChannel,
  val sctpStreamParameters: SctpStreamParameters
)

class HandlerReceiveResult(
  val localId: String,
  val track: MediaStreamTrack,
  val rtpReceiver: RtpReceiver? = null,
  val stream: MediaStream
)

class HandlerReceiveOptions(
  val trackId: String,
  val kind: RTCRtpMediaType,
  val rtpParameters: LocalRtpParameters
)

class HandlerReceiveDataChannelOptions(
  val sctpStreamParameters: SctpStreamParameters,
  val label: String,
  val protocol: String
)

class ReplaceTrackOptions(val localId: String, val track: MediaStreamTrack)

class HandlerReceiveDataChannelResult(
  val dataChannel: DataChannel,
)

class SetMaxSpatialLayerOptions(
  val localId: String,
  val spatialLayer: Int,
)

class SetRtpEncodingParametersOptions(val localId: String, val params: RtpEncodingParameters)

abstract class HandlerInterface {
  val observer = MutableSharedFlow<EmitData>()
  abstract var name: String

  abstract suspend fun close()

  abstract suspend fun getNativeRtpCapabilities(): RtpCapabilities

  abstract suspend fun getNativeSctpCapabilities(): SctpCapabilities

  abstract suspend fun run(options: HandlerRunOptions)

  abstract suspend fun updateIceServers(iceServer: List<IceServer>)

  abstract suspend fun restartIce(iceParameters: IceParameters)

  abstract suspend fun getTransportStats(): List<RtcStatsReport>

  abstract suspend fun send(options: HandlerSendOptions): HandlerSendResult

  abstract suspend fun stopSending(localId: String)

  abstract suspend fun replaceTrack(options: ReplaceTrackOptions)

  abstract suspend fun setMaxSpatialLayers(options: SetMaxSpatialLayerOptions)

  abstract suspend fun setRtpEncodingParameters(options: SetRtpEncodingParametersOptions)

  abstract suspend fun getSenderStats(localId: String): List<RtcStatsReport>

  abstract suspend fun sendDataChannel(
    options: SendDataChannelArguments
  ): HandlerSendDataChannelResult

  abstract suspend fun receive(options: HandlerReceiveOptions): HandlerReceiveResult

  abstract suspend fun stopReceiving(localId: String)

  abstract suspend fun getReceiverStats(localId: String): List<RtcStatsReport>

  abstract suspend fun receiveDataChannel(
    options: HandlerReceiveDataChannelOptions
  ): HandlerReceiveDataChannelResult

  companion object {
    fun handlerFactory(logger: IMediaClientLogger): HandlerInterface = UnifiedPlan(logger)
  }
}
