package io.dyte.media

import io.dyte.webrtc.*
import io.dyte.media.utils.*
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.int

val RTP_PROBATOR_MID: String = "probator"
val RTP_PROBATOR_SSRC: Int = 1234
val RTP_PROBATOR_CODEC_PAYLOAD_TYPE: Int = 127

class Ortc {
    companion object {
        fun validateRtcpFeedback(fb: RtcpFeedback?) {
            if (fb == null) throw Exception("Feedback is not an object")

            if (fb.type!!.isEmpty()) throw Exception("Missing fb.type")

            if (fb.parameter == null || fb.parameter?.isEmpty() == true) fb.parameter = ""
        }

        fun validateRtpCodecCompatibility(codec: LocalRtpCodecParameters?) {
            val mimeTypeRegex = Regex("^(audio|video)/(.+)")

            if (codec == null) throw Exception("Codec is not an object")

            if (codec.mimeType.isEmpty()) throw Exception("Missing codec.mimeType")

            val mimeTypeMatch = mimeTypeRegex.findAll(codec.mimeType).toList()

            //if (mimeTypeMatch.isNotEmpty()) throw Exception("Invalid codec.mimeType")

            codec.kind = RTCRtpMediaType.fromString(requireNotNull(mimeTypeMatch[0].groups[1]?.value.toString()).lowercase())

            if (codec.kind == RTCRtpMediaType.RTCRtpMediaTypeAudio) {
                if (codec == null) {
                    codec.numChannels = 1
                } else {
                    codec.numChannels = null
                }
            }

            for(key in codec.parameters.keys) {
                var value = codec.parameters[key]

                if (value == null) {
                    codec.parameters[key] = JsonPrimitive("")
                    value = JsonPrimitive("")
                }

                //if (value !is String && value !is Int) throw Exception("Invalid codec parameter [key:${key}s, value:$value")

                for (fb: RtcpFeedback in codec.rtcpFeedback) {
                    validateRtcpFeedback(fb)
                }
            }
        }

        fun validateRtpHeaderExtention(ext: RtpHeaderExtension?) {
            if (ext == null) throw Exception("ext is not an object")

            if (ext.kind != RTCRtpMediaType.RTCRtpMediaTypeAudio && ext.kind != RTCRtpMediaType.RTCRtpMediaTypeVideo)
                throw Exception("Invalid ext.kind")

            if (ext.uri == null) throw Exception("Missing ext.uri")

            if (ext.preferredId == null) throw Exception("Missing ext.preferredID")

            if (ext.preferredEncrypt == null) ext.preferredEncrypt = false

            if (ext.direction == null) ext.direction = RtpHeaderDirection.SendRecv
        }

        fun validateRtpCapabilities(caps: RtpCapabilities?) {
            if (caps == null) throw Exception("caps is not an object")

            for (codec: LocalRtpCodecParameters in caps.codecs!!) {
                validateRtpCodecCompatibility(codec)
            }

            for (ext: RtpHeaderExtension in caps.headerExtensions) {
                validateRtpHeaderExtention(ext)
            }
        }

        fun validateRtpHeaderExtensionParameters(ext: RtpHeaderExtensionParameters) {
            if (ext == null) throw Exception("ext is not an object")

            if (ext.uri == null) throw Exception("Missing ext.uri")

            if (ext.id == null) throw Exception("Missing ext.id")

            if (ext.encrypted == null) ext.encrypted = false

//            [Check] No paramteres in WebRTC-KMP HeaderExtension
//            for (key in ext.parameters.keys) {
//                var value = ext.parameters[key]
//
//                if (value == null) {
//                    ext.parameters[key] = ""
//                    value = ""
//                }
//
//                if (value !is String && value !is Int) throw Exception("Invalid header extension parameter")
//            }
        }

        fun validateRtpEncodingParameters(encoding: CommonRtpEncodingParameters) {
            if (encoding == null) throw Exception("encoding is not an object")

//            [WIP] WebRTC-KMP Missing entity rtx
//            if (encoding.rtx != null)
//                if (encoding.rtx?.ssrc == null) throw Exception("Missing encoding.rtx.ssrc")
//
//            if (encoding.dtx == null) encoding.dtx = false
        }

        fun validateRtcpParameters(rtcp: LocalRtcpParameters?) {
            if (rtcp == null) throw Exception("rtcp is not an object")

            if (rtcp.reducedSize == null) rtcp.reducedSize = true
        }

        fun validateRtpCodecParameters(codec: LocalRtpCodecParameters?) {
            val mimeTypeRegex = Regex("^(audio|video)/(.+)")

            if (codec == null) throw Exception("codec is not an object")

            val mimeTypeMatch = mimeTypeRegex.findAll(codec.mimeType.toString()).toList()

            if (mimeTypeMatch == null) throw Exception("Invalid codec.mimetpye")

            var kind: RTCRtpMediaType = RTCRtpMediaType.fromString(
                requireNotNull(mimeTypeMatch[0].groups[1]?.value).lowercase()
            )

            if (kind == RTCRtpMediaType.RTCRtpMediaTypeAudio) {
                if (codec.numChannels == null) {
                    codec.numChannels = 1
                } else {
                    codec.numChannels = null
                }
            }

            for (key: String in codec.parameters.keys) {
                var value = codec.parameters[key]

                if (value == null) {
                    codec.parameters[key] = JsonPrimitive("")
                    value = JsonPrimitive("")
                }

//                Seems Unnecessary
//                if (value !is String && value !is Int) throw Exception("Invalid codec parameter " +
//                        "[key:${key}s, value:$value]")


//                Missing feedback in WebRTC-KMP
//                for (fb: RtcpFeedback in codec.rtcpFeedback)
//                    validateRtcpFeedback(fb)
            }
        }

        fun validateRtpParameters(params: LocalRtpParameters) {
            for (codec: LocalRtpCodecParameters in params.codecs) validateRtpCodecParameters(codec)

            for (ext in params.headerExtension) validateRtpHeaderExtensionParameters(ext)

            for (encoding in params.encodings!!) validateRtpEncodingParameters(encoding)

            if (params.rtcp == null)
//
//                [WIP] No default constructor for this in WebRTC-KMP
//                params.rtcp = RtcpParameters(
//                    reducedSize =  true,
//                    cname = "",
//                    mux = false
//                )

            validateRtcpParameters(params.rtcp!!)
        }

        fun validateNumSctpStreams(numStreams: NumSctpStreams?) {
            if (numStreams == null) throw Exception("numStreams is not an object")
        }

        fun validateSctpCapabilities(caps: SctpCapabilities?) {
            if (caps == null) throw Exception("caps is not an object")

            validateNumSctpStreams(caps.numStreams)
        }

        fun validateSctpParameter(params: SctpParameters?) {
            if (params == null) throw Exception("params is not an object")
        }

        fun validateSctpStreamParameters(params: SctpStreamParameters?) {
            if (params == null) throw Exception("Params is not an object")

            var orderGiven: Boolean = false

            if (params.ordered != null) {
                orderGiven = true
            } else {
                params.ordered = true
            }

            if (orderGiven && params.ordered == true &&
                (params.maxPacketLifeTime != null || params.maxRetransmits != null)) {
                throw Exception("cannot be ordered with maxPacketLifeTime or maxRetransmits")
            } else if (!orderGiven && (params.maxPacketLifeTime != null || params.maxRetransmits != null)) {
                params.ordered = false
            }
        }

        fun isRtxCodec(codec: LocalRtpCodecParameters?): Boolean {
            if (codec == null) return false

            return Regex(".+\\/rtx\$").containsMatchIn(codec.mimeType)
        }

        fun isRtxCodec(codec: RtpCodecParameters?): Boolean {
            if (codec == null) return false

            return Regex(".+\\/rtx\$").containsMatchIn(codec.mimeType!!)
        }

        fun matchCodecs(aCodec: RtpCodecParameters?, bCodec: RtpCodecParameters?, strict: Boolean = false, modify: Boolean = false): Boolean {
            var aMimeType: String = aCodec?.mimeType!!.lowercase()
            var bMimeType: String = bCodec?.mimeType!!.lowercase()

            if (aMimeType != bMimeType) return false

            if (aCodec.clockRate != bCodec.clockRate) return false

            if (aCodec.numChannels != bCodec.numChannels) return false

            when (aMimeType) {
                "video/h264" -> {
                    var aPacketizationMode = aCodec.parameters["packetization-mode"]?.toInt() ?: 0
                    var bPacketizationMode = bCodec.parameters["packetization-mode"]?.toInt() ?: 0

                    if (aPacketizationMode != bPacketizationMode) return false

                    if (strict)
                        if(H264Utils.isSameProfile(aCodec.parameters, bCodec.parameters))
                            return false

                    var selectedProfileLevelId: String?

                    try {
                        selectedProfileLevelId = H264Utils.generateProfileLevelIdForAnswer(
                            local_supported_params = aCodec.parameters,
                            remote_offered_params = bCodec.parameters
                        )
                    } catch (e: Error) {
                        return false
                    }

                    if (modify) {
                        if (selectedProfileLevelId != null) {
                            aCodec.parameters["profile-level-id"] = selectedProfileLevelId
                            bCodec.parameters["profile-level-id"] = selectedProfileLevelId
                        } else {
                            aCodec.parameters.remove("profile-level-id")
                            bCodec.parameters.remove("profile-level-id")
                        }
                    }
                }

                "video/vp9" -> {
                    if (strict) {
                        var aProfileId = aCodec.parameters["profile-id"] ?: 0
                        var bProfileId = bCodec.parameters["profile-id"] ?: 0

                        if (aProfileId != bProfileId) return false
                    }
                }
            }

            return true
        }

        fun matchCodecs(aCodec: LocalRtpCodecParameters?, bCodec: LocalRtpCodecParameters?, strict: Boolean = false, modify: Boolean = false): Boolean {
            var aMimeType: String = (aCodec?.mimeType as String).lowercase()
            var bMimeType: String = (bCodec?.mimeType as String).lowercase()

            if (aMimeType != bMimeType) return false

            if (aCodec.clockRate != bCodec.clockRate) return false

            if (aCodec.numChannels != bCodec.numChannels) return false

            when (aMimeType) {
                "video/h264" -> {
                    var aPacketizationMode = aCodec.parameters["packetization-mode"]?.int ?: 0
                    var bPacketizationMode = bCodec.parameters["packetization-mode"]?.int ?: 0

                    if (aPacketizationMode != bPacketizationMode) return false

                    if (strict)
                        if(H264Utils.isSameProfileJSON(aCodec.parameters, bCodec.parameters)) return false

                    var selectedProfileLevelId: String?

                    try {
                        selectedProfileLevelId = H264Utils.generateProfileLevelIdForAnswerJSON(
                            local_supported_params = aCodec.parameters,
                            remote_offered_params = bCodec.parameters
                        )
                    } catch (e: Error) {
                        return false
                    }

                    if (modify) {
                        if (selectedProfileLevelId != null) {
                            aCodec.parameters["profile-level-id"] = JsonPrimitive(selectedProfileLevelId)
                            bCodec.parameters["profile-level-id"] = JsonPrimitive(selectedProfileLevelId)
                        } else {
                            aCodec.parameters.remove("profile-level-id")
                            bCodec.parameters.remove("profile-level-id")
                        }
                    }
                }

                "video/vp9" -> {
                    if (strict) {
                        var aProfileId = aCodec.parameters["profile-id"] ?: 0
                        var bProfileId = bCodec.parameters["profile-id"] ?: 0

                        if (aProfileId != bProfileId) return false
                    }
                }
            }

            return true
        }

        fun matchCodecs(aCodec: RtpCodecParameters?, bCodec: LocalRtpCodecParameters?, strict: Boolean = false, modify: Boolean = false): Boolean {
            var aMimeType: String = (aCodec?.mimeType as String).lowercase()
            var bMimeType: String = (bCodec?.mimeType as String).lowercase()

            if (aMimeType != bMimeType) return false

            if (aCodec.clockRate != bCodec.clockRate) return false

            if (aCodec.numChannels != bCodec.numChannels) return false

            when (aMimeType) {
                "video/h264" -> {
                    var aPacketizationMode = aCodec.parameters["packetization-mode"]?.toInt() ?: 0
                    var bPacketizationMode = bCodec.parameters["packetization-mode"]?.int ?: 0

                    if (aPacketizationMode != bPacketizationMode) return false

                    if (strict)
                        if(H264Utils.isSameProfileJSON2(aCodec.parameters, bCodec.parameters)) return false

                    var selectedProfileLevelId: String?

                    try {
                        selectedProfileLevelId = H264Utils.generateProfileLevelIdForAnswerJSON2(
                            local_supported_params = aCodec.parameters,
                            remote_offered_params = bCodec.parameters
                        )
                    } catch (e: Error) {
                        return false
                    }

                    if (modify) {
                        if (selectedProfileLevelId != null) {
                            aCodec.parameters["profile-level-id"] = selectedProfileLevelId
                            bCodec.parameters["profile-level-id"] = JsonPrimitive(selectedProfileLevelId)
                        } else {
                            aCodec.parameters.remove("profile-level-id")
                            bCodec.parameters.remove("profile-level-id")
                        }
                    }
                }

                "video/vp9" -> {
                    if (strict) {
                        var aProfileId = aCodec.parameters["profile-id"] ?: 0
                        var bProfileId = bCodec.parameters["profile-id"] ?: 0

                        if (aProfileId != bProfileId) return false
                    }
                }
            }

            return true
        }

        fun reduceRtcpFeedback(codecA: LocalRtpCodecParameters, codecB: LocalRtpCodecParameters): List<RtcpFeedback> {
            var reducedRtcpFeedback: MutableList<RtcpFeedback> = mutableListOf()

            for (aFb: RtcpFeedback in codecA.rtcpFeedback) {
                var matchingBFb: RtcpFeedback? = codecB.rtcpFeedback.firstOrNull {
                    it.parameter == aFb.parameter && it.type == aFb.type
                }

                if (matchingBFb != null) reducedRtcpFeedback.add(matchingBFb)
            }

            return reducedRtcpFeedback
        }

        fun matchHeaderExtentions(aExt: RtpHeaderExtension, bExt: RtpHeaderExtension): Boolean {
            if (aExt.kind != null && bExt.kind != null && aExt.kind != bExt.kind) return false

            if (aExt.uri != bExt.uri) return false

            return true
        }

        fun getExtendedRtpCapabilities(localCaps: RtpCapabilities, remoteCaps: RtpCapabilities): ExtendedRtpCapabilities {
            val extendedRtpCapabilities = ExtendedRtpCapabilities(
                codecs = mutableListOf(),
                headerExtensions = mutableListOf()
            )

            for (remoteCodec: LocalRtpCodecParameters in remoteCaps.codecs!!) {
                if (isRtxCodec(remoteCodec)) continue

                val matchingLocalCodec: LocalRtpCodecParameters? = localCaps.codecs!!.firstOrNull {
                    matchCodecs(
                        aCodec = it,
                        bCodec = remoteCodec,
                        strict = true,
                        modify = true
                    )
                }

                if (matchingLocalCodec == null) continue

                val extendedCodec = ExtendedRtpCodec(
                    mimeType = matchingLocalCodec.mimeType,
                    kind = matchingLocalCodec.kind!!,
                    clockRate = matchingLocalCodec.clockRate!!,
                    channels = matchingLocalCodec.numChannels,
                    localPayloadType = matchingLocalCodec.preferredPayloadType,
                    localRtxPayloadType = null,
                    remotePayloadType = remoteCodec.preferredPayloadType,
                    remoteRtxPayloadType = null,
                    localParameters = matchingLocalCodec.parameters,
                    remoteParameters = remoteCodec.parameters,
                    rtcpFeedback = reduceRtcpFeedback(matchingLocalCodec, remoteCodec)
                )

                extendedRtpCapabilities.codecs.add(extendedCodec)
            }

            for (extendedCodec: ExtendedRtpCodec in extendedRtpCapabilities.codecs) {
                var matchingLocalRtxCodec: LocalRtpCodecParameters? = localCaps.codecs!!.firstOrNull {
                    isRtxCodec(it) && it.parameters["apt"].toString() == extendedCodec.localPayloadType.toString()
                }

                val matchingRemoteRtxCodec: LocalRtpCodecParameters? = remoteCaps.codecs.firstOrNull {
                    isRtxCodec(it) && it.parameters["apt"].toString() == extendedCodec.remotePayloadType.toString()
                }

                if (matchingLocalRtxCodec != null && matchingRemoteRtxCodec != null) {
                    extendedCodec.localRtxPayloadType = matchingLocalRtxCodec.preferredPayloadType
                    extendedCodec.remoteRtxPayloadType = matchingRemoteRtxCodec.preferredPayloadType
                }
            }

            for (remoteExt: RtpHeaderExtension in remoteCaps.headerExtensions) {
                val matchingLocalExt: RtpHeaderExtension? = localCaps.headerExtensions.firstOrNull {
                    matchHeaderExtentions(it, remoteExt)
                }

                if (matchingLocalExt == null) continue

                val extendedExt = ExtendedRtpHeaderExtension(
                    kind = remoteExt.kind!!,
                    uri = remoteExt.uri as String,
                    sendId = requireNotNull(matchingLocalExt.preferredId),
                    recvId = requireNotNull(remoteExt.preferredId),
                    encrypt = requireNotNull(matchingLocalExt.preferredEncrypt),
                    direction = RtpHeaderDirection.SendRecv
                )

                when (remoteExt.direction) {
                    RtpHeaderDirection.SendRecv -> extendedExt.direction = RtpHeaderDirection.SendRecv
                    RtpHeaderDirection.RecvOnly -> extendedExt.direction = RtpHeaderDirection.SendOnly
                    RtpHeaderDirection.SendOnly -> extendedExt.direction = RtpHeaderDirection.RecvOnly
                    RtpHeaderDirection.Inactive -> extendedExt.direction = RtpHeaderDirection.Inactive
                    else -> {}
                }

                extendedRtpCapabilities.headerExtensions.add(extendedExt)
            }

            return extendedRtpCapabilities
        }

        fun generateProbatorRtpParameters(videoRtpParameters: LocalRtpParameters): LocalRtpParameters {
            validateRtpParameters(videoRtpParameters)

            var rtpParameters: LocalRtpParameters? = LocalRtpParameters()

            val tempEncoding = CommonRtpEncodingParameters()
            tempEncoding.ssrc = RTP_PROBATOR_SSRC.toLong()

            rtpParameters?.encodings = listOf(tempEncoding!!)
            rtpParameters?.rtcp = LocalRtcpParameters(cname = "probator")

            rtpParameters!!.codecs.add(videoRtpParameters.codecs.first())
            rtpParameters.codecs.first().payloadType = RTP_PROBATOR_CODEC_PAYLOAD_TYPE
            rtpParameters.headerExtension = videoRtpParameters.headerExtension

            return rtpParameters!!
        }

        fun reduceCodecs(codecs: List<LocalRtpCodecParameters>, capCodec: LocalRtpCodecParameters?): List<LocalRtpCodecParameters> {
            var filteredCodecs: MutableList<LocalRtpCodecParameters> = mutableListOf()

            if (capCodec == null) {
                filteredCodecs.add(codecs.first())

                if (codecs.size > 1 && isRtxCodec(codecs[1])) filteredCodecs.add(codecs[1])
            } else {
                for (idx: Int in 0 until codecs.size) {
                    if (matchCodecs(aCodec = codecs[idx], bCodec = capCodec)) {
                        filteredCodecs.add(codecs[idx])

                        if(isRtxCodec(codecs.getOrNull(idx + 1))) filteredCodecs.add(codecs[idx + 1])

                        break
                    }
                }

                if (filteredCodecs.isEmpty()) throw Exception("No matching codec found")
            }

            return filteredCodecs
        }


        fun getSendingRemoteRtpParameters(kind: RTCRtpMediaType,
            extendedRtpCapabilities: ExtendedRtpCapabilities): LocalRtpParameters {
            val rtpParameters = LocalRtpParameters(
                mid = null,
                codecs = mutableListOf(),
                headerExtension = mutableListOf(),
                encodings = emptyList(),
                rtcp = LocalRtcpParameters()
            )

            for (extendedCodec: ExtendedRtpCodec in extendedRtpCapabilities.codecs) {
                if (extendedCodec.kind != kind) continue

                val codec = LocalRtpCodecParameters(
                    payloadType = extendedCodec.localPayloadType!!,
                    mimeType = extendedCodec.mimeType,
                    clockRate = extendedCodec.clockRate,
                    numChannels = extendedCodec.channels,
                    parameters = extendedCodec.remoteParameters
                )

                rtpParameters.codecs.add(codec)

                if (extendedCodec.localRtxPayloadType != null) {

                    val rtxCodec = LocalRtpCodecParameters(
                        mimeType = "${RTCRtpMediaType.value(kind)}/rtx",
                        payloadType = extendedCodec.localRtxPayloadType!!,
                        clockRate = extendedCodec.clockRate,
                        parameters = mutableMapOf("apt" to JsonPrimitive(extendedCodec.localPayloadType.toString()))
                    )

                    rtpParameters.codecs.add(rtxCodec)
                }
            }

            for (extendedExtension: ExtendedRtpHeaderExtension in extendedRtpCapabilities.headerExtensions) {
                if ((extendedExtension.kind != null && extendedExtension.kind != kind)
                    || (extendedExtension.direction != RtpHeaderDirection.SendRecv
                    && extendedExtension.direction != RtpHeaderDirection.SendOnly)) continue

                var ext = RtpHeaderExtensionParameters(
                    uri = extendedExtension.uri,
                    id = extendedExtension.sendId,
                    encrypted = extendedExtension.encrypt,
                    parameters = "",
                )

                rtpParameters!!.headerExtension.add(ext)
            }

            if (rtpParameters.headerExtension.any { ext: RtpHeaderExtensionParameters ->
                    ext.uri == "http://www.ietf.org/id/draft-holmer-rmcat-transport-wide-cc-extensions-01"
                }) {
                for (codec in rtpParameters.codecs) {
                    codec.rtcpFeedback = codec.rtcpFeedback.filter { fb: RtcpFeedback ->
                        fb.type != "goog-remb"
                    }.toMutableList()
                }
            } else if (rtpParameters.headerExtension.any { ext: RtpHeaderExtensionParameters ->
                    ext.uri == "http://www.webrtc.org/experiments/rtp-hdrext/abs-send-time"
                }) {
                for (codec in rtpParameters.codecs) {
                    codec.rtcpFeedback = codec.rtcpFeedback.filter { fb: RtcpFeedback ->
                        fb.type != "transport-cc"
                    }.toMutableList()
                }
            } else {
                for (codec in rtpParameters.codecs) {
                    codec.rtcpFeedback = codec.rtcpFeedback.filter { fb: RtcpFeedback ->
                        fb.type != "transport-cc" && fb.type != "goog-remb"
                    }.toMutableList()
                }
            }

            return rtpParameters!!
        }

        fun getRecvRtpCapabilities(extendedRtpCapabilities: ExtendedRtpCapabilities): RtpCapabilities {
            var rtpCapabilities = RtpCapabilities(
                codecs = mutableListOf(),
                headerExtensions = mutableListOf()
            )

            for (extendedCodec: ExtendedRtpCodec in extendedRtpCapabilities.codecs) {
                var codec = LocalRtpCodecParameters(
                    mimeType = extendedCodec.mimeType,
                    kind = extendedCodec.kind,
                    preferredPayloadType = extendedCodec.remotePayloadType,
                    clockRate = extendedCodec.clockRate,
                    numChannels = extendedCodec.channels,
                    parameters = extendedCodec.localParameters,
                    rtcpFeedback = extendedCodec.rtcpFeedback.toMutableList(),
                )

                rtpCapabilities.codecs!!.add(codec)

                if (extendedCodec.remoteRtxPayloadType == null) continue

                var rtxCodec = LocalRtpCodecParameters(
                    mimeType = "${RTCRtpMediaType.value(extendedCodec.kind)}/rtx",
                    kind = extendedCodec.kind,
                    preferredPayloadType = extendedCodec.remoteRtxPayloadType,
                    clockRate = extendedCodec.clockRate,
                    parameters = mutableMapOf(
                        "apt" to JsonPrimitive(extendedCodec.remotePayloadType!!.toString()),
                    ),
                    rtcpFeedback = mutableListOf(),
                )

                rtpCapabilities.codecs!!.add(rtxCodec)
            }

            for (extendedExtension: ExtendedRtpHeaderExtension in extendedRtpCapabilities.headerExtensions) {
                if (extendedExtension.direction != RtpHeaderDirection.SendRecv
                    && extendedExtension.direction != RtpHeaderDirection.RecvOnly) continue

                var ext = RtpHeaderExtension(
                    kind = extendedExtension.kind,
                    uri = extendedExtension.uri,
                    preferredId = extendedExtension.recvId,
                    preferredEncrypt = extendedExtension.encrypt,
                    direction = extendedExtension.direction
                )

                rtpCapabilities.headerExtensions.add(ext)
            }

            return rtpCapabilities
        }

        fun getSendingRtpParameters(kind: RTCRtpMediaType,
            extendedRtpCapabilities: ExtendedRtpCapabilities): LocalRtpParameters {

            val rtpParameters = LocalRtpParameters(
                mid = null,
                codecs = mutableListOf(),
                headerExtension = mutableListOf(),
                encodings = emptyList(),
                rtcp = LocalRtcpParameters()
            )

            for (extendedCodec: ExtendedRtpCodec in extendedRtpCapabilities.codecs) {
                if (extendedCodec.kind != kind) continue

                val codec = LocalRtpCodecParameters(
                    kind = extendedCodec.kind,
                    mimeType = extendedCodec.mimeType,
                    payloadType = extendedCodec.localPayloadType!!,
                    clockRate = extendedCodec.clockRate,
                    numChannels = extendedCodec.channels,
                    parameters = extendedCodec.localParameters,
                    rtcpFeedback = extendedCodec.rtcpFeedback.toMutableList()
                )

                rtpParameters.codecs.add(codec)

                if (extendedCodec.localRtxPayloadType != null) {
                    val rtxCodec = LocalRtpCodecParameters(
                        mimeType = "${RTCRtpMediaType.value(extendedCodec.kind)}/rtx",
                        payloadType = extendedCodec.localRtxPayloadType!!,
                        clockRate = extendedCodec.clockRate,
                        parameters = mutableMapOf("apt" to JsonPrimitive(extendedCodec.localPayloadType!!.toString()))
                    )

                    rtpParameters.codecs.add(rtxCodec)
                }
            }

            for (extendedExtension: ExtendedRtpHeaderExtension in extendedRtpCapabilities.headerExtensions) {
                if ((extendedExtension.kind != null && extendedExtension.kind != kind)
                    || (extendedExtension.direction != RtpHeaderDirection.SendRecv
                    && extendedExtension.direction != RtpHeaderDirection.SendOnly)) continue

                val ext = RtpHeaderExtensionParameters(
                    uri = extendedExtension.uri,
                    id = extendedExtension.sendId,
                    encrypted = extendedExtension.encrypt,
                    parameters = "",
                )

                rtpParameters.headerExtension.add(ext)
            }

            return rtpParameters
        }

        fun canSend(kind: RTCRtpMediaType, extendedRtpCapabilities: ExtendedRtpCapabilities ): Boolean {
            return extendedRtpCapabilities.codecs.any { codec: ExtendedRtpCodec ->
                codec.kind == kind
            }
        }

        fun canReceive(rtpParameters: LocalRtpParameters, extendedRtpCapabilities: ExtendedRtpCapabilities?): Boolean {
            validateRtpParameters(rtpParameters)

            if (rtpParameters.codecs.isEmpty()) return false

            var firstMediaCodec: LocalRtpCodecParameters = rtpParameters.codecs.first()

            if (extendedRtpCapabilities != null) {
                return extendedRtpCapabilities.codecs.any { codec: ExtendedRtpCodec ->
                    (codec.remotePayloadType == firstMediaCodec.payloadType) ?: false
                }
            }
            return false
        }
    }
}

class ExtendedRtpCapabilities(
    val codecs: MutableList<ExtendedRtpCodec> = mutableListOf(),
    val headerExtensions: MutableList<ExtendedRtpHeaderExtension> = mutableListOf()
)