package io.dyte.core.feat

import io.dyte.core.controllers.DyteStageStatus
import io.dyte.core.controllers.ParticipantController
import io.dyte.core.listeners.DyteParticipantUpdateListener
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.newSingleThreadContext

/**
 * Dyte meeting participant
 *
 * @property id The ID of the participant, this ID is temporary changes every time the participant
 *   rejoins the meeting
 * @property userId The userId of the participant, this ID is tied to [clientSpecificId] and does
 *   not change for a user in a meeting
 * @property name The participant's name
 * @property picture The participant's picture (if any).
 * @property clientSpecificId Identifier provided while adding the participant.
 * @property audioEnabled A boolean value indicating if the audio currently enabled.
 * @property videoEnabled A boolean value indicating if the video currently enabled.
 */
abstract class DyteMeetingParticipant
internal constructor(
  open val id: String,
  open val userId: String,
  open val name: String,
  open val picture: String?,
  open val isHost: Boolean,
  open val clientSpecificId: String?,
  open val flags: ParticipantFlags,
  open val presetName: String,
  internal val participantController: ParticipantController,
) {

  internal var _stageStatus: DyteStageStatus = DyteStageStatus.OFF_STAGE
  open val stageStatus: DyteStageStatus
    get() = _stageStatus

  internal var _audioEnabled: Boolean = false
    set(value) {
      field = value
      participantController.emitEvent { it.onAudioUpdate(field, this) }
    }

  val audioEnabled: Boolean
    get() = _audioEnabled

  internal var _videoEnabled: Boolean = false
    set(value) {
      field = value
      participantController.emitEvent { it.onVideoUpdate(field, this) }
    }

  val videoEnabled: Boolean
    get() = _videoEnabled

  @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
  protected val serialScope = CoroutineScope(newSingleThreadContext("DyteMeetingParticipant"))

  open fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["id"] = id
    map["userId"] = userId
    map["name"] = name
    map["picture"] = picture
    map["isHost"] = isHost
    map["clientSpecificId"] = clientSpecificId
    map["flags"] = flags.toMap()
    map["audioEnabled"] = audioEnabled
    map["videoEnabled"] = videoEnabled
    return map
  }

  fun addParticipantUpdateListener(participantUpdateListener: DyteParticipantUpdateListener) {
    participantController.addParticipantUpdateListener(this, participantUpdateListener)
  }

  fun removeParticipantUpdateListener(participantUpdateListener: DyteParticipantUpdateListener) {
    participantController.removeParticipantUpdateListener(
      this@DyteMeetingParticipant,
      participantUpdateListener,
    )
  }

  fun removeParticipantUpdateListeners() {
    participantController.removeParticipantUpdateListeners(this)
  }
}

data class ParticipantFlags(
  val recorder: Boolean,
  val hiddenParticipant: Boolean,
  val webinarHiddenParticipant: Boolean,
) {
  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["recorder"] = recorder
    map["hiddenParticipant"] = hiddenParticipant
    return map
  }
}
