@file:Suppress("DEPRECATION")

package io.dyte.core.controllers

import io.dyte.core.controllers.IParticipantController.UpdatePeerStageStatusResult
import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.host.IHostController
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DyteParticipantUpdateListener
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteMeetingType
import io.dyte.core.models.DyteParticipant
import io.dyte.core.models.DyteParticipants
import io.dyte.core.models.DyteScreenShareMeetingParticipant
import io.dyte.core.models.DyteSelectedPeers
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDyteVideoUtils
import io.dyte.core.platform.VideoView
import io.dyte.core.room.RoomSocketEvent
import io.dyte.core.room.RoomSocketHandler
import io.dyte.core.socket.RoomNodeSocketService
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketActiveSpeakerModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketGetPageModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerPinnedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketStageStatus
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.core.utils.DyteUtils.supportsVideoProduction
import io.dyte.core.utils.ReadHeavyMutableList
import io.dyte.core.utils.WriteHeavyMutableList
import io.dyte.core.waitingroom.BaseWaitlistController
import io.dyte.webrtc.VideoStreamTrack
import kotlin.math.ceil
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.put
import socket.room.AddedParticipant
import socket.room.GetAllAddedParticipantsResponse
import socket.room.PeerInfoResponse

internal class ParticipantController(
  private val mediaSoupController: MediaSoupController?,
  private val isHiveMeeting: Boolean,
  private val participantInfo: ParticipantInfo,
  private val videoUtils: IDyteVideoUtils?,
  private val socketController: RoomNodeSocketService,
  private val hostController: IHostController,
  private val roomNodeController: IRoomNodeController?,
  private val scope: CoroutineScope,
  private val flutterNotifier: IEventController,
  private val meetingType: DyteMeetingType,
  private val socketService:
    ISockratesSocketService?, // TODO: Remove direct passing of socketService
  private val roomSocketHandler: RoomSocketHandler,
) : IParticipantController, EventEmitter<DyteParticipantEventsListener>(), InternalEvents {

  internal lateinit var selfParticipant: DyteSelfParticipant
  internal lateinit var waitlistController: BaseWaitlistController

  internal val joinedParticipants = WriteHeavyMutableList<DyteJoinedMeetingParticipant>()
  internal val activeParticipants = ReadHeavyMutableList<DyteJoinedMeetingParticipant>()
  private val hiddenParticipants = ReadHeavyMutableList<DyteJoinedMeetingParticipant>()

  internal val screenshareParticipants: List<DyteScreenShareMeetingParticipant>
    get() =
      joinedParticipants
        .filter { it._screenShareTrack != null }
        .map { DyteScreenShareMeetingParticipant.fromJoinedParticipant(it) }

  private var pinnedParticipantId: String? = null
  internal val pinnedParticipant: DyteJoinedMeetingParticipant?
    get() = joinedParticipants.find { it.id == pinnedParticipantId }

  internal var activeSpeaker: DyteJoinedMeetingParticipant? = null

  private val _allParticipants = mutableListOf<DyteParticipant>()
  override val allParticipants: List<DyteParticipant>
    get() = _allParticipants

  override var currentActiveGridParticipantsHive = mutableSetOf<String>()

  private val selectedPeers = DyteSelectedPeers()

  private var gridPagesInfo =
    GridInfo(0, 0, isNextPagePossible = false, isPreviousPagePossible = false)

  private var requestedPageNumber = 0

  private val participantUpdateListeners =
    hashMapOf<String, ArrayList<DyteParticipantUpdateListener>>()

  private val mainScope = MainScope()

  /**
   * (Hive specific) Stores [WebSocketStageStatus] of peers against their Ids till they are added to
   * the joined list.
   */
  private val peerHiveStageStatusMap: MutableMap<String, WebSocketStageStatus> = mutableMapOf()

  override fun getCurrentPageNumber(): Int {
    return gridPagesInfo.currentPageNumber
  }

  init {
    if (meetingType == DyteMeetingType.CHAT) {
      subscribeToChatRoomEvents()
    } else {
      subscribeToMeetingRoomEvents()
    }
  }

  private fun subscribeToMeetingRoomEvents() {
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_JOINED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          onPeerJoined(event.payload as WebSocketMeetingPeerUser)
        }
      },
    )
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_LEFT,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          onPeerLeft(event.payload as WebSocketPeerLeftModel)
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_SELECTED_PEERS,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          onSelectedPeers(event.payload as WebSocketSelectedPeersModel)
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_PINNED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          onPinPeer((event.payload as WebSocketPeerPinnedModel).peerId)
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_ACTIVE_SPEAKER,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          onActiveSpeaker((event.payload as WebSocketActiveSpeakerModel).peerId)
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_NO_ACTIVE_SPEAKER,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          onNoActiveSpeaker()
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_MUTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          onPeerAudioMuted(event.payload as WebSocketPeerMuteModel)
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_UNMUTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          onPeerAudioUnmuted(event.payload as WebSocketPeerMuteModel)
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PAUSE_CONSUMER,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val payload = event.payload as WebSocketConsumerClosedModel
          onPeerVideoMuted(payload)
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_CLOSE_CONSUMER,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val payload = event.payload as WebSocketConsumerClosedModel
          onPeerVideoMuted(payload)
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_REMOVED_FROM_STAGE,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          selfParticipant._stageStatus = DyteStageStatus.OFF_STAGE
        }
      },
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_STARTED_PRESENTING,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          selfParticipant._stageStatus = DyteStageStatus.ON_STAGE
        }
      },
    )

    roomSocketHandler.roomEvents
      .onEach { event ->
        /*
         * NOTE(swapnil): Storing peer's stageStatus, if present, in 'peerHiveStageStatusMap' for Hive meetings.
         * Do not do any long operations here or it will cause a race with the onPeerJoined() method.
         * */
        if (event is RoomSocketEvent.JoinRoom) {
          val peer = event.peerInfo
          if (peer.peerId == selfParticipant.id || peer.stageStatus == null) {
            return@onEach
          }

          peerHiveStageStatusMap[peer.peerId] = peer.stageStatus.toWebSocketStageStatus()
        }
      }
      .launchIn(scope)
  }

  // TODO(swapnil): Listen to roomSocketHandler.roomEvents in-place of socketService
  private fun subscribeToChatRoomEvents() {
    /* No need to unsubscribe explicitly. We clear all the SocketService listeners while leaving call. */
    socketService?.subscribe(
      SocketServiceUtils.RoomEvent.JOIN_ROOM.id,
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          scope.launch { handleSocketServiceEvent(event, payload) }
        }
      },
    )
  }

  override fun getVideoView(dyteMeetingParticipant: DyteJoinedMeetingParticipant): VideoView? {
    return videoUtils?.getVideoView(dyteMeetingParticipant)
  }

  override fun getScreenShareView(dyteMeetingParticipant: DyteJoinedMeetingParticipant): VideoView {
    if (!meetingType.supportsVideoProduction) {
      DyteLogger.error(
        "ParticipantController::getScreenShareView::cannot create screenShareView in a CHAT meeting"
      )
      error("ScreenShareView not available in CHAT meeting")
    }

    return videoUtils?.getScreenShareView(dyteMeetingParticipant)
      ?: kotlin.run {
        DyteLogger.error("ParticipantController::getScreenShareView::videoUtils is null")
        error("Failed to get screenShareView")
      }
  }

  /**
   * On peer joined
   *
   * when a remote peer joins, this does not contain audio track or video track
   *
   * @param meetingPeerUser
   * @param fetchStageStatus should this method fetch peer's stageStatus from SocketService. NOTE:
   *   'fetchStageStatus' applies only to Hive meetings.
   */
  internal fun onPeerJoined(
    meetingPeerUser: WebSocketMeetingPeerUser,
    fetchStageStatus: Boolean = true,
  ) {
    val participant =
      if (isHiveMeeting && fetchStageStatus) {
        getHiveJoinedParticipantWithStageStatus(meetingPeerUser)
      } else {
        WebSocketMeetingPeerUser.fromWebSocketMeetingPeerUser(
          meetingPeerUser,
          selfParticipant,
          this,
          hostController,
        )
      }

    // TODO: Handle this properly.
    // When self is host, flags.recorder is true because it is set based on
    // canRecord() permission, which is wrong. Having the permission to trigger
    // recording and actually being the recorder are two different things.
    // Hence, self was getting added to hiddenParticipants list unnecessarily.
    // Added a temporary fix for now to not change any public facing classes.
    if (
      (participant.flags.hiddenParticipant || participant.flags.recorder) &&
        participant.id != selfParticipant.id
    ) {
      hiddenParticipants.add(participant)
    } else {
      val existing = joinedParticipants.find { it.id == participant.id }
      if (existing == null) {
        joinedParticipants.add(participant)
      }
      updatePageCount()
      emitEvent { it.onParticipantJoin(participant) }
      if (
        joinedParticipants.size < participantInfo.presetInfo.gridInfo.maxParticipantsPerPage &&
          participant.flags.webinarHiddenParticipant.not()
      ) {
        val existingActive = activeParticipants.find { it.id == participant.id }
        if (existingActive == null && !isHiveMeeting) {
          activeParticipants.add(participant)
          emitEvent {
            it.onActiveParticipantsChanged(
              activeParticipants.distinctBy(DyteJoinedMeetingParticipant::id)
            )
          }
        }
      }
    }
  }

  private fun getHiveJoinedParticipantWithStageStatus(
    meetingPeerUser: WebSocketMeetingPeerUser
  ): DyteJoinedMeetingParticipant {
    // No need to evaluate stageStatus for Self
    if (meetingPeerUser.id == selfParticipant.id) {
      return selfParticipant
    }

    val peerStageStatus =
      runBlocking(scope.coroutineContext) {
        peerHiveStageStatusMap[meetingPeerUser.id]?.also {
          // also remove entry from peerHiveStageStatusMap after retrieval
          peerHiveStageStatusMap.remove(meetingPeerUser.id)
        } ?: getPeerWebSocketStageStatus(meetingPeerUser.id)
      }

    if (peerStageStatus == null) {
      DyteLogger.warn(
        "ParticipantController::onPeerJoined::hive stageStatus received for ${meetingPeerUser.id} is null"
      )
    }

    val meetingPeerUserWithStageStatus =
      meetingPeerUser.copy(stageStatus = peerStageStatus ?: WebSocketStageStatus.OFF_STAGE)
    return WebSocketMeetingPeerUser.fromWebSocketMeetingPeerUser(
      meetingPeerUserWithStageStatus,
      selfParticipant,
      this,
      hostController,
    )
  }

  internal fun onPeerLeft(webSocketPeerLeftModel: WebSocketPeerLeftModel) {
    var hiddenParticipantLeft = false
    var peerToRemove = joinedParticipants.find { it.id == webSocketPeerLeftModel.id }
    if (peerToRemove == null) {
      hiddenParticipantLeft = hiddenParticipants.removeAll { it.id == webSocketPeerLeftModel.id }
    }

    var refreshGridHive = false

    if (
      activeParticipants.toSafeList().any { it.id == peerToRemove?.id } &&
        this.getCurrentPageNumber() == 0
    ) {
      refreshGridHive = true
    }

    if (!hiddenParticipantLeft) {
      peerToRemove?.let {
        if (peerToRemove.screenShareTrack != null) {
          videoUtils?.destroyScreenShare(peerToRemove)
          emitEvent {
            it.onScreenShareEnded(peerToRemove)
            it.onScreenSharesUpdated()
          }
        }
        if (pinnedParticipantId != null && peerToRemove.id == pinnedParticipantId) {
          val pinnedParticipant = joinedParticipants.find { it.id == pinnedParticipantId }
          pinnedParticipantId = null
          pinnedParticipant?.let { emitEvent { it.onParticipantUnpinned(pinnedParticipant) } }
        }
        joinedParticipants.remove(peerToRemove)
        videoUtils?.destroyView(peerToRemove)
        val removedFromActive = activeParticipants.removeAll { it.id == peerToRemove.id }
        if (removedFromActive) {
          if (activeParticipants.isEmpty()) {
            scope.launch {
              if (getCurrentPageNumber() > 0) {
                setPage(getCurrentPageNumber().minus(1))
              } else {
                setPage(0)
              }
            }
          } else {
            emitEvent {
              it.onActiveParticipantsChanged(
                activeParticipants.distinctBy(DyteJoinedMeetingParticipant::id)
              )
            }
          }
        }
        updatePageCount()
        emitEvent { it.onParticipantLeave(peerToRemove) }
      }
    }

    if (isHiveMeeting && refreshGridHive) {
      scope.launch { refreshGridParticipantsHive(force = false) }
    }
  }

  private fun updatePageCount(currentPageNumber: Int = gridPagesInfo.currentPageNumber) {
    val remotePeersCount = joinedParticipants.size.minus(1) // removing self

    val nextPages =
      if (remotePeersCount >= participantInfo.presetInfo.gridInfo.maxParticipantsPerPage) {
        ceil(
            joinedParticipants.size /
              participantInfo.presetInfo.gridInfo.maxParticipantsPerPage.toFloat()
          )
          .toInt()
          .plus(1)
      } else {
        1
      }
    val newGridInfo =
      gridPagesInfo.copy(
        pageCount = nextPages,
        currentPageNumber = currentPageNumber,
        isNextPagePossible = currentPageNumber < nextPages.minus(1),
        isPreviousPagePossible = currentPageNumber > 0,
      )
    if (newGridInfo != gridPagesInfo) {
      gridPagesInfo = newGridInfo
    }
  }

  internal fun onPeerAudioMuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant =
      joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
        ?: kotlin.run {
          DyteLogger.warn(
            "ParticipantController::onPeerAudioMuted::joinedParticipant with peerId ${webSocketPeerMuteModel.peerId} not found"
          )
          return
        }
    participant._audioEnabled = false
    onAudioUpdate(participant)
  }

  internal fun onPeerAudioUnmuted(webSocketPeerMuteModel: WebSocketPeerMuteModel) {
    val participant =
      joinedParticipants.find { it.id == webSocketPeerMuteModel.peerId }
        ?: kotlin.run {
          DyteLogger.warn(
            "ParticipantController::onPeerAudioUnmuted::joinedParticipant with peerId ${webSocketPeerMuteModel.peerId} not found"
          )
          return
        }
    participant._audioEnabled = true
    onAudioUpdate(participant)
  }

  fun onPeerVideoMuted(webSocketConsumerClosedModel: WebSocketConsumerClosedModel) {
    val consumerId = webSocketConsumerClosedModel.id

    if (isHiveMeeting) {
      val participant =
        joinedParticipants.find { it.id == consumerId }
          ?: kotlin.run {
            DyteLogger.warn(
              "ParticipantController::onPeerVideoMuted::joinedParticipant with peerId $consumerId not found"
            )
            return
          }
      participant._videoEnabled = false
      videoUtils?.destroyView(participant)
      onVideoUpdate(participant)
      return
    }

    val consumerType = mediaSoupController?.getConsumerType(consumerId)
    if ("audio" == consumerType) {
      return
    }

    val appdata =
      mediaSoupController?.getAppDataFromConsumerId(consumerId)
        ?: kotlin.run {
          DyteLogger.warn(
            "ParticipantController::onPeerVideoMuted::appData for consumerId $consumerId not found"
          )
          return
        }
    val peerId = appdata.peerId

    val participant =
      joinedParticipants.find { it.id == peerId }
        ?: kotlin.run {
          DyteLogger.warn(
            "ParticipantController::onPeerVideoMuted::joinedParticipant with peerId $peerId not found"
          )
          return
        }
    if (!appdata.screenShare) {
      participant._videoEnabled = false
      videoUtils?.destroyView(participant)
      onVideoUpdate(participant)
    } else {
      participant._screenShareTrack = null
      videoUtils?.destroyScreenShare(participant)
      emitEvent {
        it.onScreenShareEnded(participant)
        it.onScreenSharesUpdated()
      }
    }
  }

  fun onPeerVideoUnmuted(webSocketConsumerClosedModel: WebSocketConsumerResumedModel) {
    if (isHiveMeeting) {
      val participant =
        joinedParticipants.find { it.id == webSocketConsumerClosedModel.id }
          ?: kotlin.run {
            DyteLogger.warn(
              "ParticipantController::onPeerVideoUnmuted::joinedParticipant with peerId ${webSocketConsumerClosedModel.id} not found"
            )
            return
          }
      participant._videoEnabled = true
      onVideoUpdate(participant)
    }
  }

  override fun onParticipantVideoMuted(participant: DyteJoinedMeetingParticipant) {
    participant._videoEnabled = false
    videoUtils?.destroyView(participant)
    onVideoUpdate(participant)
  }

  override fun onParticipantVideoUnmuted(
    participant: DyteJoinedMeetingParticipant,
    videoTrack: VideoStreamTrack,
  ) {
    participant._videoEnabled = true
    participant._videoTrack = videoTrack
    onVideoUpdate(participant)
  }

  internal fun onPinPeer(pinnedPeerId: String?) {
    DyteLogger.info("ParticipantController::onPinPeer::$pinnedPeerId")
    if (pinnedPeerId == null) {
      pinnedParticipantId?.let { nonNullPinnedId ->
        val pinnedParticipant = joinedParticipants.find { it.id == nonNullPinnedId }
        pinnedParticipant?.let { pinned ->
          pinnedParticipantId = null
          emitEvent { it.onParticipantUnpinned(pinned) }
        }
      }
    } else {
      pinnedParticipantId = pinnedPeerId
      val participant = joinedParticipants.find { it.id == pinnedParticipantId }
      participant?.let { emitEvent { it.onParticipantPinned(participant) } }
      if (roomNodeController is HiveController) {
        scope.launch { roomNodeController.activatePeer(peerId = pinnedPeerId) }
      }
    }
  }

  private fun onActiveSpeaker(id: String) {
    val participant = joinedParticipants.find { it.id == id }
    activeSpeaker = participant
    activeSpeaker?.let { active -> emitEvent { it.onActiveSpeakerChanged(active) } }
  }

  private fun onNoActiveSpeaker() {
    activeSpeaker = null
    emitEvent { it.onNoActiveSpeaker() }
  }

  override fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel) {
    webSocketRoomStateModel.roomState.peers.forEach { onPeerJoined(it, false) }
  }

  override fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    if (joinedParticipants.find { it.id == selfParticipant.id } == null) {
      joinedParticipants.add(selfParticipant)
    }
    if (webSocketJoinRoomModel.peers.isNotEmpty()) {
      webSocketJoinRoomModel.peers.forEach { onPeerJoined(it, false) }
    }
    if (webSocketJoinRoomModel.pinnedPeerId?.isNotEmpty() == true) {
      pinnedParticipantId = webSocketJoinRoomModel.pinnedPeerId
      val pinnedParticipant = joinedParticipants.find { it.id == pinnedParticipantId }
      pinnedParticipant?.let { emitEvent { it.onParticipantPinned(pinnedParticipant) } }
    }
  }

  internal fun onSelectedPeers(webSocketSelectedPeersModel: WebSocketSelectedPeersModel) {
    if (requestedPageNumber == 0) {
      val peerIds = arrayListOf<String>()
      peerIds.addAll(webSocketSelectedPeersModel.peerIds)

      val compulsoryPeerIds = arrayListOf<String>()
      compulsoryPeerIds.addAll(webSocketSelectedPeersModel.compulsoryPeers ?: emptyList())

      if (!isHiveMeeting) {
        if (
          !webSocketSelectedPeersModel.peerIds.contains(selfParticipant.id) &&
            (webSocketSelectedPeersModel.peerIds.size <
              participantInfo.presetInfo.gridInfo.maxParticipantsPerPage)
        ) {
          if (selfParticipant.stageStatus == DyteStageStatus.ON_STAGE) {
            peerIds.add(selfParticipant.id)
          }
        } else {
          peerIds.add(selfParticipant.id)
        }
        processNewPage(peerIds, 0, gridPagesInfo.currentPageNumber)
      } else {
        val peersWithPriority =
          peerIds.mapIndexed { index, s -> PeerWithPriority(s, (index + 1)) }.toMutableList()

        val compulsoryPeerWithPriority =
          compulsoryPeerIds.mapIndexed { index, s -> PeerWithPriority(s, -(index + 1)) }

        peersWithPriority.addAll(compulsoryPeerWithPriority)

        if (peersWithPriority.size > 0) {
          this.updateActiveParticipantsWithPriorities(peersWithPriority)
        }
      }
    }
  }

  override fun updateActiveParticipantsWithPriorities(
    peersWithPriority: MutableList<PeerWithPriority>,
    refresh: Boolean,
  ) {
    // HIVE TODO: check for roomJoined=true

    peersWithPriority.forEach {
      if (this.joinedParticipants.any { p -> p.id == it.peerId }) {
        this.selectedPeers.add(it.peerId, it.priority)
      }
    }

    if (refresh) {
      scope.launch { refreshGridParticipantsHive(force = false) }
    }
  }

  override suspend fun updateActiveParticipantsHiveNode(
    peerIds: MutableList<String>,
    force: Boolean,
  ) {
    // HIVE: TODO Add room joined check

    val activePeerIds = this.activeParticipants.toSafeList().map { it.id }.toMutableList()
    val activePeerIdsSaved = activePeerIds.toList()
    val activePeers = this.activeParticipants
    val peerIdMap = mutableMapOf<String, Boolean>()
    peerIds.forEach { peerId -> peerIdMap[peerId] = true }

    /**
     * NOTE(roerohan): The following piece of code is to ensure that the positions in the active
     * peer map are not affected by the order of the peerIds sent by room node. If a participant is
     * present in the 3rd position in the map, and room node sends the peerId in the 2nd position,
     * the participant should remain in the 3rd position.
     */
    val replaceableIndices = arrayListOf<Int>()

    activePeerIds.forEachIndexed { index, activePeerId ->
      if (
        !peerIdMap.containsKey(activePeerId) ||
          this.joinedParticipants.none { it.id == activePeerId }
      ) {
        replaceableIndices.add(index)
      }
    }

    peerIds.forEach { peerId ->
      if (activePeers.toSafeList().none { it.id == peerId }) {
        if (activePeerIds.size < peerIds.size) {
          activePeerIds.add(peerId)
        } else {
          val f = replaceableIndices.removeFirstOrNull()
          if (f != null) {
            activePeerIds.add(f, peerId)
          }
        }
      }
    }

    if (meetingType == DyteMeetingType.WEBINAR) {
      activePeerIds.removeAll { !peerIdMap.containsKey(it) }
    } else {
      activePeerIds.removeAll { !peerIdMap.containsKey(it) && it != selfParticipant.id }
    }

    this.activeParticipants.clear()

    activePeerIds.forEach { peerId ->
      val p = this.joinedParticipants.find { it.id == peerId }
      if (p == null) {
        DyteLogger.info("updateActiveParticipants::participant_not_in_joined_list $peerId ")
        return
      }

      this.activeParticipants.add(p)
    }

    // Adding self to active
    if (meetingType == DyteMeetingType.WEBINAR) {
      // In webinar, only add if self is ON_STAGE
      if (selfParticipant.stageStatus == DyteStageStatus.ON_STAGE) {
        this.activeParticipants.add(selfParticipant)
      }
    } else {
      this.activeParticipants.add(selfParticipant)
    }

    //    this.updateWebinarStatusForJoinedParticipants(peerIds);

    updatePageCount()

    val activePeerIdsNew = this.activeParticipants.toSafeList().map { it.id }

    if (activePeerIdsSaved != activePeerIdsNew) {
      /*
       * TODO: check if we really need distinctBy operation because it might emit a different list
       *  than the final one saved. Our logic should form the correct final active list and that should be saved & emitted.
       * */
      emitEvent {
        it.onActiveParticipantsChanged(
          activeParticipants.distinctBy(DyteJoinedMeetingParticipant::id)
        )
      }
    }

    // NOTE(itzmanish): If force is true we want to activate the provided
    // peer ids set after deactivating the existing peers if there are any
    // Currently the force flag will only be provided if ICE reconnection on
    // receive transport happened.
    if (force) {
      currentActiveGridParticipantsHive = mutableSetOf()
    }

    val peerIdsSet = peerIds.toMutableSet()
    val peersToDeactivate = arrayListOf<String>()
    val peersToActivate = arrayListOf<String>()

    currentActiveGridParticipantsHive.forEach { peerId ->
      if (!peerIdsSet.contains(peerId)) {
        peersToDeactivate.add(peerId)
      }
    }

    peerIdsSet.forEach { peerId ->
      if (!currentActiveGridParticipantsHive.contains(peerId)) {
        peersToActivate.add(peerId)
      }
    }

    currentActiveGridParticipantsHive = peerIdsSet

    // NOTE(itzmanish): check if finalPeerList is different from the current active
    // grid participants if so then we need to consume new peers which were not in
    // the grid before.
    if (peersToActivate.size == 0 && peersToDeactivate.size == 0) {
      DyteLogger.info(
        "ParticipantController::updateActiveParticipants::current_selected_peers_are_same"
      )
      return
    }

    // TODO : fix this
    if (isHiveMeeting) {
      peersToDeactivate.forEach { peerId ->
        (roomNodeController as HiveController).deactivatePeer(peerId)

        val peer = joinedParticipants.find { it.id == peerId }
        if (peer != null) videoUtils?.destroyView(peer)
      }

      peersToActivate.forEach { peerId ->
        (roomNodeController as HiveController).activatePeer(peerId)
      }
    }
  }

  /*
   * TODO: add support for ISO-8601 date. Take input as epoch in millis and convert it to ISO-8601
   *  date string.
   * */
  @OptIn(ExperimentalSerializationApi::class)
  override suspend fun broadcastMessage(type: String, data: Map<String, *>) {
    DyteLogger.info("ParticipantController::broadcastMessage::")
    val payload = buildJsonObject {
      data.entries.forEach {
        val value =
          when (it.value) {
            is String -> {
              JsonPrimitive(it.value.toString())
            }
            is Boolean -> {
              JsonPrimitive(it.value as Boolean)
            }
            is Number -> {
              JsonPrimitive(it.value as Number)
            }
            is UInt -> {
              JsonPrimitive(it.value as UInt)
            }
            else -> {
              JsonPrimitive("not supported format ${it.value}")
            }
          }
        put(it.key, value)
      }
    }

    if (isHiveMeeting) {
      (roomNodeController as HiveController).broadcastMessage(type, payload)
    } else {
      val message = buildJsonObject {
        put("type", type)
        put("payload", payload)
      }
      socketController.sendMessage(OutboundMeetingEventType.ROOM_MESSAGE, message)
    }
  }

  override suspend fun refreshGridParticipantsHive(force: Boolean) {
    if (this.getCurrentPageNumber() != 0) {
      return
    }

    // Add only ON_STAGE participants to active list (covers webinar)
    val joinedPeers =
      this.joinedParticipants.filter { it.stageStatus == DyteStageStatus.ON_STAGE }.map { it.id }

    val selectedPeers = this.selectedPeers.peers

    if (getCurrentPageNumber() != 0) {
      DyteLogger.info(
        "ParticipantController::refreshGridParticipants::not_refreshing_grid_in_not_active_grid_mode"
      )
      return
    }

    // FIXME(itzmanish): I am only considering preset v1 for now.
    // Need to be fixed by adding maxVideoCount on the config.
    val maxPeerOnScreen = participantInfo.presetInfo.gridInfo.maxParticipantsPerPage - 1

    val finalPeerListSet = setOf<String>()
    finalPeerListSet.plus(
      selectedPeers.filter { it != selfParticipant.id && joinedPeers.contains(it) }
    )

    var finalPeerList = finalPeerListSet.toMutableList()

    val peersToAdd = maxPeerOnScreen - finalPeerListSet.size

    if (peersToAdd >= 0) {
      DyteLogger.info("ParticipantController::refreshGridParticipants::adding_peers")
      val peersToAddFromJoined =
        joinedPeers
          .filter { id -> !finalPeerListSet.contains(id) && id != selfParticipant.id }
          .take(peersToAdd)
      finalPeerList.addAll(peersToAddFromJoined)
    } else {
      finalPeerList = finalPeerList.take(maxPeerOnScreen).toMutableList()
    }

    this.updateActiveParticipantsHiveNode(finalPeerList, force)
  }

  override fun onSelectedPeersDiff(entries: List<PeerWithPriority>) {
    if (this.getCurrentPageNumber() != 0) return

    this.updateActiveParticipantsWithPriorities(entries.toMutableList(), true)
  }

  override suspend fun setPage(newPageNumber: Int) {
    DyteLogger.info(
      "ParticipantController::setPage::$newPageNumber::current::${getCurrentPageNumber()}"
    )
    if (newPageNumber < 0) {
      throw IllegalArgumentException("page number cant be less than 0")
    }

    if (isHiveMeeting) {
      if (newPageNumber == 0) {
        val peerIds = currentActiveGridParticipantsHive.toList()
        processNewPage(peerIds, newPageNumber, gridPagesInfo.currentPageNumber)

        return
      }

      val compulsoryPeers = this.selectedPeers.compulsoryPeers
      val filteredCompulsoryPeers =
        compulsoryPeers.filter { peerId -> this.joinedParticipants.any { it.id == peerId } }
      val pinnedPeers = filteredCompulsoryPeers.filter { it == pinnedParticipantId }
      val joinedPeerIds = this.joinedParticipants.map { it.id }

      val factor =
        (participantInfo.presetInfo.gridInfo.maxParticipantsPerPage -
          filteredCompulsoryPeers.size -
          pinnedPeers.size)

      val start = (newPageNumber - 1) * factor

      var end = newPageNumber * factor

      if (end > joinedPeerIds.size) end = joinedPeerIds.size

      val returnPeerIds =
        filteredCompulsoryPeers + pinnedPeers + joinedPeerIds.slice(start until end)

      processNewPage(returnPeerIds, newPageNumber, gridPagesInfo.currentPageNumber)

      /*val activePeers = getParticipantsArrayFrom(returnPeerIds)
      activeParticipants.clear()
      activeParticipants.addAll(activePeers.toList())

      if (roomNodeController is HiveController) {
        activeParticipants.forEach { roomNodeController.activatePeer(it.id) }
      }

      updatePageCount(newPageNumber)
      emitEvent {
        it.onActiveParticipantsChanged(
          activeParticipants.distinctBy(DyteJoinedMeetingParticipant::id)
        )
      }*/
      return
    }

    requestedPageNumber = newPageNumber
    val content = HashMap<String, JsonElement>()
    content["pageNum"] = JsonPrimitive(newPageNumber)
    val prevPageResponse =
      socketController.sendMessageParsed<WebSocketGetPageModel>(
        OutboundMeetingEventType.GET_PAGE,
        JsonObject(content),
      )
    if (prevPageResponse.peerIds.isNotEmpty()) {
      processNewPage(prevPageResponse.peerIds, newPageNumber, gridPagesInfo.currentPageNumber)
    }
  }

  private fun processNewPage(peerIds: List<String>, pageNumber: Int, previousPageNumber: Int) {
    val newParticipant = getParticipantsArrayFrom(peerIds).toMutableList()

    var activeParticipantsChanged = false

    if (pageNumber == 0) {
      if (meetingType == DyteMeetingType.WEBINAR) {
        if (selfParticipant.stageStatus == DyteStageStatus.ON_STAGE) {
          activeParticipantsChanged = true
          newParticipant.add(selfParticipant)
        }
      } else {
        activeParticipantsChanged = true
        newParticipant.add(selfParticipant)
      }
    }

    if (pageNumber > 0) {
      activeParticipants.removeAll { it.id == selfParticipant.id }
    }

    if (previousPageNumber != pageNumber) {
      // For new page we have to recreate active Participant and respect the one which is send by
      // server
      activeParticipants.clear()
      activeParticipants.addAll(newParticipant.toList())
      activeParticipantsChanged = true
    } else {
      // Find new elements to add
      val participantsToAdd =
        getDiff(newParticipant, activeParticipants.toSafeList()).toMutableList()
      // Find old elements to Remove
      val participantsToRemove = getDiff(activeParticipants.toSafeList(), newParticipant)

      if (participantsToAdd.isNotEmpty() || participantsToRemove.isNotEmpty()) {
        // Means Either Add or Remove , And both Too, then we have to just trigger a callback
        activeParticipantsChanged = true
      }

      // search for element to remove
      participantsToRemove.forEach { toRemove ->
        var index = 0
        while (index < activeParticipants.size) {
          // Find index of element in active array to remove
          if (activeParticipants.get(index).id == toRemove.id) break
          index += 1
        }
        activeParticipants.removeAt(index)
        videoUtils?.destroyView(toRemove)

        // Insert an element at the remove index if there is new participant is there to add.
        val elementToAdd = participantsToAdd.firstOrNull()
        elementToAdd?.let {
          // Replace this element inside the array
          activeParticipants.add(index, it)
          participantsToAdd.remove(it)
        }
      }
      // Check if we have elements still left to add
      participantsToAdd.forEach { activeParticipants.add(it) }

      if (roomNodeController is HiveController) {
        currentActiveGridParticipantsHive =
          this.activeParticipants.toSafeList().map { it.id }.toMutableSet()

        scope.launch {
          participantsToAdd.forEach { roomNodeController.activatePeer(it.id) }

          participantsToRemove.forEach { roomNodeController.deactivatePeer(it.id) }
        }
      }
    }

    updatePageCount(pageNumber)
    if (activeParticipantsChanged) {
      emitEvent {
        it.onActiveParticipantsChanged(
          activeParticipants.distinctBy(DyteJoinedMeetingParticipant::id)
        )
      }
    }
  }

  private fun getParticipantsArrayFrom(peerIds: List<String>): List<DyteJoinedMeetingParticipant> {
    val result = arrayListOf<DyteJoinedMeetingParticipant>()
    for (peerId in peerIds) {
      val peer = joinedParticipants.find { it.id == peerId } ?: continue

      // DO-NOT change stageStatus here in case of a HIVE WEBINAR
      if (isHiveMeeting && meetingType == DyteMeetingType.WEBINAR) {
        result.add(peer)
        continue
      }

      peer._stageStatus = DyteStageStatus.ON_STAGE
      result.add(peer)
    }

    return result
  }

  private fun getDiff(
    a: List<DyteJoinedMeetingParticipant>,
    b: List<DyteJoinedMeetingParticipant>,
  ): List<DyteJoinedMeetingParticipant> {
    val result = arrayListOf<DyteJoinedMeetingParticipant>()
    a.forEach { participant ->
      // check if this exist in Active participant array.
      val peer = b.find { it.id == participant.id }
      if (peer == null) {
        result.add(participant)
      }
    }
    return result
  }

  override fun onPeerScreenShareStarted(participant: DyteJoinedMeetingParticipant) {
    // DyteParticipantEventsListener
    emitEvent {
      it.onScreenShareStarted(participant)
      it.onScreenSharesUpdated()
    }

    // DyteParticipantUpdateListener
    dispatchParticipantUpdate(participant.id) { it.onScreenShareStarted() }

    flutterNotifier.triggerEvent(DyteEventType.OnScreenShareUpdate(screenshareParticipants))
  }

  override fun onPeerScreenSharedEnded(participant: DyteJoinedMeetingParticipant) {
    val screenSharingParticipant =
      joinedParticipants.find { it.id == participant.id }
        ?: kotlin.run {
          DyteLogger.warn(
            "ParticipantController::onPeerScreenSharedEnded::screenSharingParticipant with peerId ${participant.id} not found"
          )
          return
        }

    emitEvent {
      it.onScreenShareEnded(participant)
      it.onScreenSharesUpdated()
    }

    dispatchParticipantUpdate(screenSharingParticipant.id) { it.onScreenShareEnded() }

    flutterNotifier.triggerEvent(DyteEventType.OnScreenShareUpdate(screenshareParticipants))

    videoUtils?.destroyScreenShare(screenSharingParticipant)
  }

  override fun getGridPagesInfo(): GridInfo {
    return gridPagesInfo
  }

  override fun addParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener,
  ) {
    val existingListeners: ArrayList<DyteParticipantUpdateListener>? =
      participantUpdateListeners[dyteMeetingParticipant.id]
    val updatedListeners = arrayListOf<DyteParticipantUpdateListener>()
    if (existingListeners == null) {
      updatedListeners.add(dyteParticipantUpdateListener)
    } else {
      updatedListeners.addAll(existingListeners)
      if (existingListeners.contains(dyteParticipantUpdateListener).not()) {
        updatedListeners.add(dyteParticipantUpdateListener)
      }
    }
    participantUpdateListeners[dyteMeetingParticipant.id] = updatedListeners
  }

  override fun removeParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener,
  ) {
    val existingListeners = participantUpdateListeners[dyteMeetingParticipant.id]
    existingListeners?.remove(dyteParticipantUpdateListener)
  }

  override fun removeParticipantUpdateListeners(dyteMeetingParticipant: DyteMeetingParticipant) {
    participantUpdateListeners[dyteMeetingParticipant.id] = arrayListOf()
  }

  override suspend fun handleSocketServiceRoomJoined() {
    if (meetingType != DyteMeetingType.CHAT) {
      return
    }
    withContext(scope.coroutineContext) { refreshAllParticipants() }
  }

  override fun updatePeerStageStatus(
    peerId: String,
    updatedStageStatus: DyteStageStatus,
  ): UpdatePeerStageStatusResult? {
    return runBlocking(scope.coroutineContext) {
      val peerToUpdate = joinedParticipants.find { it.id == peerId } ?: return@runBlocking null
      val oldStageStatus = peerToUpdate.stageStatus
      if (oldStageStatus == updatedStageStatus) {
        return@runBlocking null
      }

      peerToUpdate._stageStatus = updatedStageStatus
      dispatchParticipantUpdate(peerId) { it.onUpdate(peerToUpdate) }
      UpdatePeerStageStatusResult(oldStageStatus, peerToUpdate)
    }
  }

  fun onAudioUpdate(participant: DyteMeetingParticipant) {
    dispatchParticipantUpdate(participant.id) { it.onAudioUpdate(participant.audioEnabled) }
  }

  fun onVideoUpdate(participant: DyteMeetingParticipant) {
    dispatchParticipantUpdate(participant.id) { it.onVideoUpdate(participant.videoEnabled) }
  }

  override fun emitEvent(event: (listener: DyteParticipantEventsListener) -> Unit) {
    if (selfParticipant.roomJoined.not()) {
      // do not emit join room event if user is not joined in
      return
    }
    super.emitEvent(event)
    super.emitEvent {
      it.onUpdate(DyteParticipants(this@ParticipantController, hostController, waitlistController))
    }
  }

  fun triggerOnUpdate() {
    super.emitEvent {
      it.onUpdate(DyteParticipants(this@ParticipantController, hostController, waitlistController))
    }
  }

  fun enableCache() {
    DyteLogger.info("ParticipantController::enableCache::")
    videoUtils?.enableCache()
  }

  fun disableCache() {
    DyteLogger.info("ParticipantController::enableCache::")
    videoUtils?.disableCache()
  }

  private fun dispatchParticipantUpdate(
    participantId: String,
    updateCallback: (DyteParticipantUpdateListener) -> Unit,
  ) {
    scope.launch {
      val updateListeners = participantUpdateListeners[participantId] ?: return@launch
      updateListeners.forEach { mainScope.launch { updateCallback(it) } }
    }
  }

  private suspend fun handleSocketServiceEvent(event: Int, payload: ByteArray?) {
    when (event) {
      SocketServiceUtils.RoomEvent.JOIN_ROOM.id -> {
        if (payload == null) {
          return
        }

        val peerInfoResponse =
          try {
            PeerInfoResponse.ADAPTER.decode(payload)
          } catch (e: Exception) {
            DyteLogger.info(
              "ParticipantController::handle JOIN_ROOM event::failed to decode PeerInfoResponse"
            )
            return
          }

        val joinedPeer =
          peerInfoResponse.peer
            ?: kotlin.run {
              DyteLogger.info(
                "ParticipantController::handle JOIN_ROOM event::received peer is null"
              )
              return
            }

        if (joinedPeer.user_id == selfParticipant.userId) {
          return
        }

        refreshAllParticipants()
      }
    }
  }

  private suspend fun refreshAllParticipants() {
    if (socketService == null) {
      DyteLogger.info("ParticipantController::refreshAllParticipants::socketService not provided")
      return
    }

    val responsePayload =
      try {
        socketService.requestResponse(
          SocketServiceUtils.RoomEvent.GET_ALL_ADDED_PARTICIPANTS.id,
          null,
        )
      } catch (e: Exception) {
        DyteLogger.error(
          "ParticipantController::refreshAllParticipants::socket request failed ${e.message}"
        )
        return
      }

    if (responsePayload == null) {
      DyteLogger.warn("ParticipantController::refreshAllParticipants::socket response is null")
      return
    }

    val getAllAddedParticipantsResponse =
      try {
        GetAllAddedParticipantsResponse.ADAPTER.decode(responsePayload)
      } catch (e: Exception) {
        DyteLogger.warn(
          "ParticipantController::refreshAllParticipants::failed to decode getAllChatChannelResponse"
        )
        return
      }

    _allParticipants.clear()
    getAllAddedParticipantsResponse.participants.forEach {
      _allParticipants.add(it.toDyteParticipant())
    }

    emitEvent { it.onAllParticipantsUpdated(_allParticipants) }
  }

  /**
   * Fetches the stageStatus of a participant from SocketService and returns it as
   * [WebSocketStageStatus].
   */
  private suspend fun getPeerWebSocketStageStatus(peerId: String): WebSocketStageStatus? {
    val roomPeerInfo = roomSocketHandler.getPeerInfo(peerId)
    return roomPeerInfo?.stageStatus?.toWebSocketStageStatus()
  }

  companion object {
    private fun AddedParticipant.toDyteParticipant(): DyteParticipant {
      return DyteParticipant(
        userId = id,
        name = name,
        picture = picture,
        customParticipantId = custom_participant_id,
      )
    }

    private fun DyteStageStatus.toWebSocketStageStatus(): WebSocketStageStatus {
      return when (this) {
        DyteStageStatus.ON_STAGE -> WebSocketStageStatus.ON_STAGE
        DyteStageStatus.REQUESTED_TO_JOIN_STAGE -> WebSocketStageStatus.REQUESTED_TO_JOIN_STAGE
        DyteStageStatus.ACCEPTED_TO_JOIN_STAGE -> WebSocketStageStatus.ACCEPTED_TO_JOIN_STAGE
        else -> WebSocketStageStatus.OFF_STAGE
      }
    }
  }
}

interface IParticipantController {

  val allParticipants: List<DyteParticipant>
  var currentActiveGridParticipantsHive: MutableSet<String>

  fun getCurrentPageNumber(): Int

  fun handleRoomState(webSocketRoomStateModel: WebSocketRoomStateModel)

  fun handleRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel)

  fun onParticipantVideoMuted(participant: DyteJoinedMeetingParticipant)

  fun onParticipantVideoUnmuted(
    participant: DyteJoinedMeetingParticipant,
    videoTrack: VideoStreamTrack,
  )

  fun onPeerScreenShareStarted(participant: DyteJoinedMeetingParticipant)

  fun onPeerScreenSharedEnded(participant: DyteJoinedMeetingParticipant)

  fun onSelectedPeersDiff(entries: List<PeerWithPriority>)

  suspend fun setPage(newPageNumber: Int)

  fun getVideoView(dyteMeetingParticipant: DyteJoinedMeetingParticipant): VideoView?

  fun getScreenShareView(dyteMeetingParticipant: DyteJoinedMeetingParticipant): VideoView

  fun getGridPagesInfo(): GridInfo

  fun updateActiveParticipantsWithPriorities(
    peersWithPriority: MutableList<PeerWithPriority>,
    refresh: Boolean = false,
  )

  suspend fun refreshGridParticipantsHive(force: Boolean = false)

  suspend fun updateActiveParticipantsHiveNode(peerIds: MutableList<String>, force: Boolean = false)

  suspend fun broadcastMessage(type: String, data: Map<String, *>)

  fun addParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener,
  )

  fun removeParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener,
  )

  fun removeParticipantUpdateListeners(dyteMeetingParticipant: DyteMeetingParticipant)

  suspend fun handleSocketServiceRoomJoined()

  fun updatePeerStageStatus(
    peerId: String,
    updatedStageStatus: DyteStageStatus,
  ): UpdatePeerStageStatusResult?

  data class UpdatePeerStageStatusResult(
    val oldStageStatus: DyteStageStatus,
    val updatedPeer: DyteJoinedMeetingParticipant,
  )
}

data class GridInfo(
  val pageCount: Int,
  val currentPageNumber: Int,
  val isNextPagePossible: Boolean,
  val isPreviousPagePossible: Boolean,
) {
  fun toMap(): Map<String, Any?> {
    val args = HashMap<String, Any?>()
    args["pageCount"] = pageCount
    args["currentPageNumber"] = currentPageNumber
    args["isNextPagePossible"] = isNextPagePossible
    args["isPreviousPagePossible"] = isPreviousPagePossible
    return args
  }
}

data class PeerWithPriority(val peerId: String, val priority: Int)
