package io.dyte.core.network.info

import io.dyte.core.controllers.DyteStageStatus
import io.dyte.core.models.DyteMediaPermission
import io.dyte.core.models.DyteMediaPermission.ALLOWED
import io.dyte.core.models.DyteMediaPermission.NOT_ALLOWED
import io.dyte.core.network.models.BorderRadiusType
import io.dyte.core.network.models.BorderRadiusType.ROUNDED
import io.dyte.core.network.models.BorderWidthType
import io.dyte.core.network.models.BorderWidthType.THIN
import io.dyte.core.network.models.ParticipantChatPreset.ChatChannelPreset
import io.dyte.core.network.models.ParticipantChatPreset.ChatMessagePreset
import io.dyte.core.network.models.ParticipantPreset
import io.dyte.core.network.models.UserDataWrapper
import io.dyte.core.network.models.UserDataWrapperV1
import io.dyte.core.network.models.UserPresetDataWrapper
import io.dyte.core.network.models.UserPresetDesignColors
import io.dyte.core.network.models.WaitingRoomType

data class ParticipantInfo
internal constructor(
  val id: String,
  val name: String,
  val email: String?,
  val picture: String?,
  val loggedIn: Boolean,
  val clientSpecificId: String,
  val organizationId: String,
  val scope: String?,
  val hiddenParticipant: Boolean,
  val isRecorder: Boolean,
  val presetInfo: PresetInfo
) {
  companion object {
    fun getParticipantInfoFromV1(
      userData: UserDataWrapperV1,
      userPresetData: UserPresetDataWrapper
    ): ParticipantInfo {
      val presetInfo = PresetInfo.getPresetInfoFromV1(userPresetData)
      return ParticipantInfo(
        id = userData.user.id,
        name = userData.user.name ?: "",
        email = null,
        picture = userData.user.picture,
        loggedIn = false,
        clientSpecificId = userData.user.clientSpecificId,
        organizationId = userData.user.organizationId,
        scope = null,
        hiddenParticipant = false,
        isRecorder = false,
        presetInfo = presetInfo
      )
    }

    fun getParticipantInfoFromV2(userData: UserDataWrapper): ParticipantInfo {
      val preset = PresetInfo.getPresetInfoFromV2(userData.data.preset)
      val participant = userData.data.participant
      return ParticipantInfo(
        participant.id,
        participant.name ?: "",
        null,
        userData.data.participant.picture,
        false,
        participant.clientSpecificId,
        participant.organizationId,
        null,
        false,
        false,
        preset
      )
    }
  }
}

data class PresetInfo
internal constructor(
  val viewType: String,
  val name: String,
  val waitingRoomType: WaitingRoomType,
  val gridInfo: GridInfo,
  val permissions: SelfPermissions,
  val ui: DyteDesignToken
) {
  companion object {
    fun getPresetInfoFromV1(userPresetData: UserPresetDataWrapper): PresetInfo {
      userPresetData.data?.preset?.let { preset ->
        val selfPermissions = SelfPermissions.getParticipantPermissionsFromV1(userPresetData)
        return PresetInfo(
          preset.permissions?.viewType ?: "",
          preset.presetName ?: "",
          WaitingRoomType.valueOf(preset.permissions?.waitingRoomType?.name ?: ""),
          GridInfo(preset.theme?.grid?.multi?.maxVideoCount ?: 6),
          selfPermissions,
          DyteDesignToken.getDesignTokenFromV1(userPresetData)
        )
      }
        ?: run { throw IllegalArgumentException("not enough data to build preset info from v1") }
    }

    fun getPresetInfoFromV2(participantPreset: ParticipantPreset): PresetInfo {
      val selfPermissions = SelfPermissions.getParticipantPermissionsFromV2(participantPreset)
      return PresetInfo(
        participantPreset.config.viewType,
        participantPreset.name,
        WaitingRoomType.valueOf(participantPreset.permissions.waitingRoomType),
        GridInfo(participantPreset.config.streamsConfig.mobile),
        selfPermissions,
        DyteDesignToken.getDesignTokenFromV2(participantPreset)
      )
    }
  }
}

data class GridInfo internal constructor(val maxParticipantsPerPage: Int)

data class SelfPermissions
internal constructor(
  val miscellaneous: MiscellaneousPermissions,
  val host: HostPermissions,
  val plugins: PluginPermissions,
  val polls: PollPermissions,
  val media: MediaPermissions,
  val chat: ChatPermissions,
  val waitingRoom: WaitingRoomPermissions,
  val liveStream: LiveStreamPermissions,
  val chatChannel: ChatChannelPermissions? = null,
  val chatMessage: ChatMessagePermissions? = null
) {
  companion object {
    fun getParticipantPermissionsFromV1(userPresetData: UserPresetDataWrapper): SelfPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        val miscPermissions =
          MiscellaneousPermissions(
            permissions.canEditDisplayName ?: false,
            permissions.hiddenParticipant ?: false,
            permissions.isRecorder ?: false
          )
        val hostPermissions = HostPermissions.getHostPermissionsFromV1(userPresetData)
        val pluginPermissions = PluginPermissions.getPluginPermissionsFromV1(userPresetData)
        val pollPermissions = PollPermissions.getPollPermissionsFromV1(userPresetData)
        val mediaPermissions = MediaPermissions.getMediaPermissionsFromV1(userPresetData)
        val chatPermissions = ChatPermissions.getChatPermissionsFromV1(userPresetData)
        val waitingRoomPermission = WaitingRoomPermissions.getPermissionsFromV1(userPresetData)
        val liveStreamPermissions =
          LiveStreamPermissions.getLiveStreamPermissionsFromV1(userPresetData)
        return SelfPermissions(
          miscPermissions,
          hostPermissions,
          pluginPermissions,
          pollPermissions,
          mediaPermissions,
          chatPermissions,
          waitingRoomPermission,
          liveStreamPermissions
        )
      }
        ?: run {
          throw IllegalArgumentException("not enough data to build participant permissions from v1")
        }
    }

    fun getParticipantPermissionsFromV2(participantPreset: ParticipantPreset): SelfPermissions {
      val miscPermissions =
        MiscellaneousPermissions(
          participantPreset.permissions.canEditDisplayName,
          participantPreset.permissions.hiddenParticipant,
          participantPreset.permissions.isRecorder
        )
      val hostPermissions = HostPermissions.getHostPermissionsFromV2(participantPreset)
      val pluginPermissions = PluginPermissions.getPluginPermissionsFromV2(participantPreset)
      val pollPermissions = PollPermissions.getPollPermissionsFromV2(participantPreset)
      val mediaPermissions = MediaPermissions.getMediaPermissionsFromV2(participantPreset)
      val chatPermissions = ChatPermissions.getChatPermissionsFromV2(participantPreset)
      val waitingRoomPermission = WaitingRoomPermissions.getPermissionsFromV2(participantPreset)
      val liveStreamPermissions =
        LiveStreamPermissions.getLiveStreamPermissionsFromV2(participantPreset)
      val chatChannelPermissions =
        ChatChannelPermissions.getChatChannelPermissionsFromV2(participantPreset)
      val chatMessagePermissions =
        ChatMessagePermissions.getChatMessagePermissionsFromV2(participantPreset)
      return SelfPermissions(
        miscPermissions,
        hostPermissions,
        pluginPermissions,
        pollPermissions,
        mediaPermissions,
        chatPermissions,
        waitingRoomPermission,
        liveStreamPermissions,
        chatChannelPermissions,
        chatMessagePermissions
      )
    }
  }

  fun toMap(): Map<String, Any?> {
    return hashMapOf(
      "miscellaneous" to miscellaneous.toMap(),
      "media" to media.toMap(),
      "plugins" to plugins.toMap(),
      "polls" to polls.toMap(),
      "chat" to chat.toMap(),
      "host" to host.toMap(),
      "waitingRoom" to waitingRoom.toMap(),
      "livestream" to liveStream.toMap()
    )
  }
  // We need to pass stageStatus because its not static.
  fun canPublishVideo(stageStatus: DyteStageStatus): Boolean {
    return media.video.permission == ALLOWED ||
      stageStatus == DyteStageStatus.ON_STAGE ||
      stageStatus == DyteStageStatus.ACCEPTED_TO_JOIN_STAGE
  }

  fun canPublishAudio(stageStatus: DyteStageStatus): Boolean {
    return media.audioPermission == ALLOWED ||
      stageStatus == DyteStageStatus.ON_STAGE ||
      stageStatus == DyteStageStatus.ACCEPTED_TO_JOIN_STAGE
  }

  fun canCreatePoll(): Boolean {
    return polls.canCreate
  }

  fun canViewPoll(): Boolean {
    return polls.canView
  }

  fun canVoteOnPoll(): Boolean {
    return polls.canVote
  }

  fun closePlugins(): Boolean {
    return plugins.canClose
  }

  fun canStartPlugins(): Boolean {
    return plugins.canLaunch
  }

  fun canRecord(): Boolean {
    return host.canTriggerRecording
  }

  fun canSendChatText(): Boolean {
    return chat.canSendText
  }

  fun canSendChatFile(): Boolean {
    return chat.canSendFiles
  }

  fun canDoParticipantHostControls(): Boolean {
    return host.canMuteAudio ||
      host.canMuteVideo ||
      host.canKickParticipant ||
      host.canPinParticipant
  }
}

data class LiveStreamPermissions(val canLiveStream: Boolean) {
  companion object {
    fun getLiveStreamPermissionsFromV2(
      participantPreset: ParticipantPreset
    ): LiveStreamPermissions {
      return LiveStreamPermissions(participantPreset.permissions.canLiveStream)
    }

    fun getLiveStreamPermissionsFromV1(
      @Suppress("UNUSED_PARAMETER") // Exists for parity with similar methods
      userPresetData: UserPresetDataWrapper,
    ): LiveStreamPermissions {
      return LiveStreamPermissions(false)
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf(
      "canLivestream" to canLiveStream,
    )
  }
}

data class MiscellaneousPermissions
internal constructor(
  val canEditDisplayName: Boolean,
  val isHiddenParticipant: Boolean,
  val isRecorder: Boolean
) {
  fun toMap(): Map<String, Any> {
    return hashMapOf(
      "canEditDisplayName" to canEditDisplayName,
      "isHiddenParticipant" to isHiddenParticipant,
      "isRecorder" to isRecorder
    )
  }
}

data class HostPermissions
internal constructor(
  val canKickParticipant: Boolean,
  val canMuteAudio: Boolean,
  val canMuteVideo: Boolean,
  val canPinParticipant: Boolean,
  val canTriggerRecording: Boolean,
  val canAcceptRequests: Boolean
) {
  companion object {
    fun getHostPermissionsFromV1(userPresetData: UserPresetDataWrapper): HostPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return HostPermissions(
          canMuteAudio = permissions.canDisableParticipantAudio ?: false,
          canMuteVideo = permissions.canDisableParticipantVideo ?: false,
          canKickParticipant = permissions.kickParticipant ?: false,
          canPinParticipant = permissions.pinParticipant ?: false,
          canTriggerRecording = permissions.canRecord ?: false,
          canAcceptRequests = permissions.acceptWaitingRequests ?: false
        )
      }
        ?: run {
          throw IllegalArgumentException(
            "not sufficient data to build host permissions from v1 api"
          )
        }
    }

    // TODO : fix canPresent, canAcceptPresentRequest,
    fun getHostPermissionsFromV2(participantPreset: ParticipantPreset): HostPermissions {
      return HostPermissions(
        canMuteAudio = participantPreset.permissions.disableParticipantAudio,
        canMuteVideo = participantPreset.permissions.disableParticipantVideo,
        canKickParticipant = participantPreset.permissions.kickParticipant,
        canPinParticipant = participantPreset.permissions.pinParticipant,
        canTriggerRecording = participantPreset.permissions.canRecord,
        canAcceptRequests = participantPreset.permissions.canAcceptProductionRequests
      )
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf(
      "canMuteAudio" to canMuteAudio,
      "canMuteVideo" to canMuteVideo,
      "canKickParticipant" to canKickParticipant,
      "canPinParticipant" to canPinParticipant,
      "canTriggerRecording" to canTriggerRecording,
    )
  }
}

data class PluginPermissions
internal constructor(
  val canClose: Boolean,
  val canLaunch: Boolean,
  private val canEditAcl: Boolean
) {
  companion object {
    fun getPluginPermissionsFromV1(userPresetData: UserPresetDataWrapper): PluginPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return PluginPermissions(
          permissions.plugins?.canClose ?: false,
          permissions.plugins?.canStart ?: false,
          permissions.plugins?.canEditAcl ?: false
        )
      }
        ?: run {
          throw IllegalArgumentException(
            "not sufficient data to build plugins permissions from v1 api"
          )
        }
    }

    fun getPluginPermissionsFromV2(participantPreset: ParticipantPreset): PluginPermissions {
      return PluginPermissions(
        participantPreset.permissions.plugins.canClose,
        participantPreset.permissions.plugins.canStart,
        false
      )
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf("canClose" to canClose, "canLaunch" to canLaunch)
  }
}

data class PollPermissions
internal constructor(val canCreate: Boolean, val canVote: Boolean, val canView: Boolean) {
  companion object {
    fun getPollPermissionsFromV1(userPresetData: UserPresetDataWrapper): PollPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return PollPermissions(
          permissions.polls?.canCreate ?: false,
          permissions.polls?.canVote ?: false,
          permissions.polls?.canView ?: false
        )
      }
        ?: run {
          throw IllegalArgumentException(
            "not sufficient data to build polls permissions from v1 api"
          )
        }
    }

    fun getPollPermissionsFromV2(participantPreset: ParticipantPreset): PollPermissions {
      return PollPermissions(
        participantPreset.permissions.polls.canCreate,
        participantPreset.permissions.polls.canVote,
        participantPreset.permissions.polls.canView
      )
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf<String, Any>(
      "canCreate" to canCreate,
      "canView" to canView,
      "canVote" to canVote,
    )
  }
}

data class MediaPermissions
internal constructor(
  val video: VideoPermissions,
  val audioPermission: DyteMediaPermission,
) {

  val canPublishVideo = video.permission == ALLOWED
  val canPublishAudio = audioPermission == ALLOWED

  companion object {
    fun getMediaPermissionsFromV1(userPresetData: UserPresetDataWrapper): MediaPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        val audioPermission =
          if (permissions.produce?.audio == true) {
            ALLOWED
          } else {
            NOT_ALLOWED
          }
        return MediaPermissions(
          VideoPermissions.getVideoPermissionsFromV1(userPresetData),
          audioPermission
        )
      }
        ?: run {
          throw IllegalArgumentException(
            "not sufficient data to build media permissions from v1 api"
          )
        }
    }

    fun getMediaPermissionsFromV2(participantPreset: ParticipantPreset): MediaPermissions {
      return MediaPermissions(
        VideoPermissions.getVideoPermissionsFromV2(participantPreset),
        DyteMediaPermission.valueOf(participantPreset.permissions.media.audio.canProduce)
      )
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf(
      "audioPermission" to audioPermission.name,
      "video" to video.toMap(),
    )
  }
}

data class VideoPermissions
internal constructor(val permission: DyteMediaPermission, val quality: String, val frameRate: Int) {
  companion object {
    fun getVideoPermissionsFromV1(userPresetData: UserPresetDataWrapper): VideoPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        val videoPermission =
          if (permissions.produce?.video?.allow == true) {
            ALLOWED
          } else {
            NOT_ALLOWED
          }
        return VideoPermissions(
          videoPermission,
          permissions.produce?.video?.quality ?: "720p",
          permissions.produce?.video?.frameRate ?: 30
        )
      }
        ?: run {
          throw IllegalArgumentException(
            "not sufficient data to build video permissions from v1 api"
          )
        }
    }

    // TODO : fix quality and framerate
    fun getVideoPermissionsFromV2(participantPreset: ParticipantPreset): VideoPermissions {
      return VideoPermissions(
        DyteMediaPermission.valueOf(participantPreset.permissions.media.video.canProduce),
        "720p",
        30
      )
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf(
      "permission" to permission.name,
      "quality" to quality,
      "frameRate" to frameRate,
    )
  }
}

data class ChatPermissions
internal constructor(val canSendText: Boolean, val canSendFiles: Boolean) {
  companion object {
    fun getChatPermissionsFromV1(userPresetData: UserPresetDataWrapper): ChatPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return ChatPermissions(
          permissions.chat?.chatPublicProps?.text ?: false,
          permissions.chat?.chatPublicProps?.files ?: false
        )
      }
        ?: run {
          throw IllegalArgumentException("not sufficent data to build chat permissions from v1 api")
        }
    }

    fun getChatPermissionsFromV2(participantPreset: ParticipantPreset): ChatPermissions {
      return ChatPermissions(
        participantPreset.permissions.chat.public.text,
        participantPreset.permissions.chat.public.files
      )
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf(
      "canSendText" to canSendText,
      "canSendFiles" to canSendFiles,
    )
  }
}

data class WaitingRoomPermissions
internal constructor(val canAcceptRequests: Boolean, val behaviour: WaitingRoomType) {
  companion object {
    fun getPermissionsFromV1(userPresetData: UserPresetDataWrapper): WaitingRoomPermissions {
      userPresetData.data?.preset?.permissions?.let { permissions ->
        return WaitingRoomPermissions(
          permissions.acceptWaitingRequests ?: false,
          permissions.waitingRoomType
        )
      }
        ?: run {
          throw IllegalArgumentException(
            "not sufficent data to build waiting permissions from v1 api"
          )
        }
    }

    fun getPermissionsFromV2(participantPreset: ParticipantPreset): WaitingRoomPermissions {
      return WaitingRoomPermissions(
        participantPreset.permissions.acceptWaitingRequests,
        WaitingRoomType.valueOf(participantPreset.permissions.waitingRoomType)
      )
    }
  }

  fun toMap(): Map<String, Any> {
    return hashMapOf(
      "canAcceptRequests" to canAcceptRequests,
      "behaviour" to behaviour.name,
    )
  }
}

data class DyteDesignToken(
  val borderRadius: BorderRadiusType,
  val borderWidth: BorderWidthType,
  val colors: UserPresetDesignColors
) {
  companion object {
    fun getDesignTokenFromV1(
      @Suppress("UNUSED_PARAMETER") // Exists for parity with similar methods
      userPresetData: UserPresetDataWrapper,
    ): DyteDesignToken {
      return DyteDesignToken(
        borderRadius = ROUNDED,
        borderWidth = THIN,
        colors =
          UserPresetDesignColors(
            brand =
              mapOf(
                "300" to "#497CFD",
                "400" to "#356EFD",
                "500" to "#2160FD",
                "600" to "#0D51FD",
                "700" to "#0246FD",
              ),
            background =
              mapOf(
                "600" to "#666666",
                "700" to "#4C4C4C",
                "800" to "#333333",
                "900" to "#1A1A1A",
                "1000" to "#080808",
              ),
            text = "#FFFFFFFF",
            textOnBrand = "#FF111111",
            videoBg = "#1A1A1A",
            success = "#62A504",
            warning = "#FFCD07",
            danger = "#FF2D2D"
          )
      )
    }

    fun getDesignTokenFromV2(participantPreset: ParticipantPreset): DyteDesignToken {
      return DyteDesignToken(
        borderRadius = participantPreset.ui.designTokens.borderRadius,
        borderWidth = participantPreset.ui.designTokens.borderWidth,
        colors = participantPreset.ui.designTokens.colors
      )
    }
  }
}

data class ChatChannelPermissions
internal constructor(
  val createPermission: ChatChannelPermission,
  val deletePermission: ChatChannelPermission,
  val updatePermission: ChatChannelPermission,
  val canReadAll: Boolean
) {
  enum class ChatChannelPermission {
    NONE,
    PRIVATE,
    PUBLIC,
    ALL;

    internal companion object {
      fun fromPreset(
        channelPresetPermission: ChatChannelPreset.ChannelPermission
      ): ChatChannelPermission {
        return when (channelPresetPermission) {
          ChatChannelPreset.ChannelPermission.NONE -> NONE
          ChatChannelPreset.ChannelPermission.PRIVATE -> PRIVATE
          ChatChannelPreset.ChannelPermission.PUBLIC -> PUBLIC
          ChatChannelPreset.ChannelPermission.ALL -> ALL
        }
      }
    }
  }

  internal companion object {
    fun getChatChannelPermissionsFromV2(
      participantPreset: ParticipantPreset
    ): ChatChannelPermissions? {
      val channelPreset = participantPreset.permissions.chat.channel ?: return null
      return ChatChannelPermissions(
        createPermission = ChatChannelPermission.fromPreset(channelPreset.canCreate),
        deletePermission = ChatChannelPermission.fromPreset(channelPreset.canDelete),
        updatePermission = ChatChannelPermission.fromPreset(channelPreset.canUpdate),
        canReadAll = channelPreset.canReadAll
      )
    }
  }
}

data class ChatMessagePermissions
internal constructor(
  val deletePermission: ChatMessagePermission,
  val editPermission: ChatMessagePermission,
  val deleteCutoffTimeSeconds: Long,
  val editCutoffTimeSeconds: Long
) {
  enum class ChatMessagePermission {
    NONE,
    SELF,
    ALL;

    internal companion object {
      fun fromPreset(
        messagePresetPermission: ChatMessagePreset.MessagePermission
      ): ChatMessagePermission {
        return when (messagePresetPermission) {
          ChatMessagePreset.MessagePermission.NONE -> NONE
          ChatMessagePreset.MessagePermission.SELF -> SELF
          ChatMessagePreset.MessagePermission.ALL -> ALL
        }
      }
    }
  }

  internal companion object {
    fun getChatMessagePermissionsFromV2(
      participantPreset: ParticipantPreset
    ): ChatMessagePermissions? {
      val messagePreset = participantPreset.permissions.chat.message ?: return null
      return ChatMessagePermissions(
        deletePermission = ChatMessagePermission.fromPreset(messagePreset.canDelete),
        editPermission = ChatMessagePermission.fromPreset(messagePreset.canEdit),
        deleteCutoffTimeSeconds = messagePreset.deleteCutoffTimeSeconds,
        editCutoffTimeSeconds = messagePreset.editCutoffTimeSeconds
      )
    }
  }
}
