package io.dyte.core.controllers

import io.dyte.core.DyteError
import io.dyte.core.PresetEditorErrorCodes.No_Active_Recording_Found
import io.dyte.core.PresetEditorErrorCodes.Record_Permission_Not_Given
import io.dyte.core.PresetEditorErrorCodes.Recording_Start_Exception
import io.dyte.core.PresetEditorErrorCodes.Recording_Stop_Exception
import io.dyte.core.Result
import io.dyte.core.controllers.DyteRecordingState.IDLE
import io.dyte.core.controllers.DyteRecordingState.RECORDING
import io.dyte.core.controllers.DyteRecordingState.STARTING
import io.dyte.core.controllers.DyteRecordingState.STOPPING
import io.dyte.core.events.EventEmitter
import io.dyte.core.listeners.DyteRecordingEventsListener
import io.dyte.core.network.ApiClient
import io.dyte.core.network.info.RecordingInfo
import io.dyte.core.observability.DyteLogger
import kotlinx.coroutines.CoroutineScope

internal abstract class RecordingController(
  private val apiClient: ApiClient,
  private val canRecord: Boolean,
  protected val scope: CoroutineScope,
) : EventEmitter<DyteRecordingEventsListener>(), IRecordingController {
  private lateinit var recordingInfo: RecordingInfo
  private var recordingState: DyteRecordingState = IDLE

  suspend fun init() {
    setupEvents()
    try {
      recordingInfo = apiClient.getActiveRecording()
      updateState(RECORDING)
      emitEvent { it.onMeetingRecordingStarted() }
    } catch (e: Exception) {
      // e.printStackTrace()
    }
  }

  abstract fun setupEvents()

  override suspend fun start(): Result<Unit, DyteError> {
    if (canRecord.not()) {
      return Result.Failure(DyteError(Record_Permission_Not_Given))
    }

    return try {
      updateState(STARTING)
      recordingInfo = apiClient.startRecording()
      Result.Success(Unit)
    } catch (e: Exception) {
      DyteLogger.error("RecordingController::start", e)
      updateState(IDLE)
      emitEvent { it.onMeetingRecordingStopError(e) }
      Result.Failure(DyteError(Recording_Start_Exception, e.message ?: "Unknown exception occurs"))
    }
  }

  override suspend fun stop(): Result<Unit, DyteError> {
    if (canRecord.not()) {
      return Result.Failure(DyteError(Record_Permission_Not_Given))
    }
    if (recordingState != RECORDING) {
      return Result.Failure(DyteError(No_Active_Recording_Found))
    }
    try {
      if (::recordingInfo.isInitialized.not()) {
        recordingInfo = apiClient.getActiveRecording()
      }
      return if (recordingState == RECORDING) {
        updateState(STOPPING)
        apiClient.stopRecording(recordingInfo)
        updateState(IDLE)
        Result.Success(Unit)
      } else {
        Result.Failure(DyteError(No_Active_Recording_Found))
      }
    } catch (e: Exception) {
      DyteLogger.error("RecordingController::stop", e)
      updateState(IDLE)
      emitEvent { it.onMeetingRecordingStopError(e) }
      return Result.Failure(
        DyteError(Recording_Stop_Exception, e.message ?: "Unknown exception occurs")
      )
    }
  }

  override fun getState(): DyteRecordingState {
    return recordingState
  }

  private fun updateState(newState: DyteRecordingState) {
    recordingState = newState
    emitEvent { it.onMeetingRecordingStateUpdated(recordingState) }
  }

  protected fun onRecordingPeerJoined() {
    updateState(RECORDING)
    emitEvent { it.onMeetingRecordingStarted() }
  }

  protected fun onRecordingPeerLeft() {
    updateState(IDLE)
    emitEvent { it.onMeetingRecordingEnded() }
  }
}

interface IRecordingController {
  suspend fun start(): Result<Unit, DyteError>

  suspend fun stop(): Result<Unit, DyteError>

  fun getState(): DyteRecordingState
}

enum class DyteRecordingState {
  IDLE,
  STARTING,
  RECORDING,
  STOPPING;

  companion object {
    fun fromMap(stage: String): DyteRecordingState {
      return when (stage) {
        "idle" -> IDLE
        "starting" -> STARTING
        "recording" -> RECORDING
        "stopping" -> STOPPING
        else -> IDLE
      }
    }
  }

  fun toMap(): Map<String, String> {
    val strState =
      when (this) {
        IDLE -> "idle"
        STARTING -> "starting"
        RECORDING -> "recording"
        STOPPING -> "stopping"
      }

    val map: HashMap<String, String> = HashMap()

    map["state"] = strState

    return map
  }
}
