package io.dyte.core.chat

import io.dyte.core.Result
import io.dyte.core.chat.ChatFileUploader.UploadFileResult
import io.dyte.core.chat.models.ChatChannel
import io.dyte.core.chat.models.GetMessagesResult
import io.dyte.core.events.EventEmitter
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.platform.PlatformFile
import io.dyte.core.platform.Uri
import io.dyte.core.plugins.PluginChatSender
import kotlinx.coroutines.CoroutineScope

/*
 * TODO: Need to setup removeOnMessageListener somewhere.
 * Ideal way would be to listen to `roomJoined` event to set listeners and
 * `roomLeft` event to remove listeners.
 * */
internal abstract class BaseChatController(
  private val platformUtilsProvider: IDytePlatformUtilsProvider,
  private val canSendChatText: Boolean,
  private val canSendChatFile: Boolean,
  private val chatFileUploader: ChatFileUploader,
  protected val scope: CoroutineScope
) : EventEmitter<DyteChatEventsListener>(), IChatController, PluginChatSender {

  private lateinit var roomUUID: String

  /** TODO: Remove this */
  override fun setRoomUUID(uuid: String) {
    roomUUID = uuid
    chatFileUploader.setRoomUUID(uuid)
  }

  override val messages = arrayListOf<DyteChatMessage>()

  override val channels: MutableList<ChatChannel> = mutableListOf()

  /**
   * Sets a message listener on the ChatSocketHandler for messages coming from the socket service.
   *
   * TODO: Need to figure out a way to remove listener from ChatSocketHandler. Ideal way would be to
   *   listen to `roomJoined` event to set listeners and `roomLeft` event to remove listeners.
   */
  override suspend fun loadChatMessages() {
    try {
      val newMessages = this.getChatMessages()
      handleMultipleMessages(newMessages)
    } catch (e: Exception) {
      DyteLogger.error("ChatController::loadChatMessages::error", e)
    }
  }

  protected fun handleMultipleMessages(newMessages: List<DyteChatMessage>) {
    messages.clear()
    messages.addAll(newMessages)
    if (newMessages.isNotEmpty()) {
      emitEvent { it.onChatUpdates(messages) }
    }
  }

  protected fun handleMessage(newMessage: DyteChatMessage) {
    messages.add(newMessage)
    emitEvent {
      it.onNewChatMessage(newMessage)
      it.onChatUpdates(messages)
    }
  }

  protected abstract suspend fun getChatMessages(): List<DyteChatMessage>

  override suspend fun sendMessage(message: String) {
    DyteLogger.info("DyteChat.sendTextMessage")
    if (!canSendChatText) {
      DyteLogger.error("sendTextMessage::public_chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send text message")
    }

    if (message.isEmpty()) {
      DyteLogger.error("sendTextMessage::message_can_not_be_empty")
      throw IllegalArgumentException("message can not be empty")
    }
    sendTextMessageSocket(message)
  }

  protected abstract suspend fun sendTextMessageSocket(message: String)

  override suspend fun sendFileMessage(fileUri: Uri) {
    DyteLogger.info("DyteChat.sendFileMessage")
    if (!canSendChatFile) {
      DyteLogger.info("sendFileMessage::chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(fileUri)
        ?: run {
          DyteLogger.info("sendFileMessage::chat_read_file_failed")
          throw Exception("Requested file doesn't exist or is missing read permission")
        }

    return sendFileMessageInternal(file)
  }

  override suspend fun sendFileMessage(filePath: String) {
    DyteLogger.info("DyteChat.sendFileMessage")
    if (!canSendChatFile) {
      DyteLogger.info("sendFileMessage::chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(filePath)
        ?: run {
          DyteLogger.info("sendFileMessage::chat_read_file_failed")
          throw Exception("Requested file doesn't exist or is missing read permission")
        }

    return sendFileMessageInternal(file)
  }

  override suspend fun sendImageMessage(imageUri: Uri) {
    DyteLogger.info("DyteChat.sendImageMessage")

    if (!canSendChatFile) {
      DyteLogger.error("sendImageMessage::permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(imageUri)
        ?: run {
          DyteLogger.error("sendImageMessage::required_argument_image_can_not_be_empty")
          throw Exception("Requested image doesn't exist or is missing read permission")
        }

    return sendImageMessageInternal(file)
  }

  override suspend fun sendImageMessage(imagePath: String) {
    DyteLogger.info("DyteChat.sendImageMessage")
    if (!canSendChatFile) {
      DyteLogger.error("sendImageMessage::permission_denied")
      throw UnsupportedOperationException("not allowed to send image message")
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(imagePath)
        ?: throw Exception("Requested image doesn't exist or is missing read permission")

    return sendImageMessageInternal(file)
  }

  private suspend fun sendImageMessageInternal(file: PlatformFile) {
    if (file.mimeType !in ACCEPTED_IMAGE_TYPES) {
      DyteLogger.error("sendImageMessage::image_type_not_supported")
      throw UnsupportedOperationException("Unsupported image file found -> ${file.mimeType}")
    }

    when (val uploadImageResult = chatFileUploader.uploadFile(file)) {
      is UploadFileResult.Uploaded -> {
        this.sendImageMessageSocket(uploadImageResult.getLocation)
      }
      is UploadFileResult.UploadFailed -> {
        DyteLogger.error("sendImageMessage::${uploadImageResult.reason}")
        throw Exception("Failed to upload image")
      }
    }
  }

  protected abstract suspend fun sendImageMessageSocket(path: String)

  private suspend fun sendFileMessageInternal(file: PlatformFile) {
    when (val uploadFileResult = chatFileUploader.uploadFile(file)) {
      is UploadFileResult.Uploaded -> {
        sendFileMessageSocket(uploadFileResult.getLocation, file.name, file.size)
      }
      is UploadFileResult.UploadFailed -> {
        DyteLogger.error("sendFileMessage::failure_upload_file")
        throw Exception("Failed to upload file")
      }
    }
  }

  protected abstract suspend fun sendFileMessageSocket(
    fileUrl: String,
    fileName: String,
    fileSize: Long
  )

  override suspend fun sendTextMessageFromPlugin(message: String) {
    sendMessage(message)
  }

  protected fun parseTime(t: Long?): String {
    return platformUtilsProvider
      .getPlatformUtils()
      .getUserDisplayableTime(t?.times(1000) ?: error("Message time is null"))
  }

  companion object {
    private val ACCEPTED_IMAGE_TYPES = listOf("image/jpeg", "image/png")
  }
}

interface IChatController {
  val messages: List<DyteChatMessage>

  val channels: List<ChatChannel>

  fun setRoomUUID(uuid: String)

  suspend fun loadChatMessages()

  suspend fun sendMessage(message: String)

  suspend fun sendImageMessage(imageUri: Uri)

  suspend fun sendImageMessage(imagePath: String)

  suspend fun sendFileMessage(fileUri: Uri)

  suspend fun sendFileMessage(filePath: String)

  suspend fun getMessagesPaginated(
    timestamp: Long,
    size: Int,
    offset: Int,
    reversed: Boolean,
    channelId: String?
  ): Result<GetMessagesResult, String>
}

enum class MessageType(val type: Int) {
  TEXT(0),
  IMAGE(1),
  FILE(2)
}
