package io.dyte.core.media

import io.dyte.core.observability.ILoggerController
import io.dyte.webrtc.*
import kotlinx.coroutines.runBlocking


open class DyteCommonMediaUtils : IDyteCommonMediaUtils {
  private lateinit var logger: ILoggerController
  private var platform: String = "android";

  init {
    try {
      // WebRtc.initialize()
    } catch (e:Exception) {
      e.printStackTrace()
    }
  }

  private var audioTrack: AudioStreamTrack? = null
  private var videoTrack: VideoStreamTrack? = null

  private var videoMediaStream: MediaStream? = null
  private var audioMediaStream: MediaStream? = null

  private var cameraFacingUser = true

  override fun getVideoStream(): MediaStream? {
    if(videoMediaStream != null && videoMediaStream?.videoTracks?.first()?.readyState is MediaStreamTrackState.Live ) {
      logger.traceLog("DyteCommonMediaUtils | getVideoStream | returning existing video stream")
      return videoMediaStream!!
    }

    logger.traceLog("DyteCommonMediaUtils | getVideoStream | creating new video stream")

    val camExists = runBlocking {
      MediaDevices.enumerateDevices().isNotEmpty()
    }

    if (camExists) {
      runBlocking {
        try {
          if(platform == "iOS") {
            videoMediaStream = MediaDevices.getUserMedia {
              video {
                facingMode(if (cameraFacingUser) FacingMode.User else FacingMode.Environment)
              }
            }
          } else {
            videoMediaStream = MediaDevices.getUserMedia {
              video {
                facingMode(if (cameraFacingUser) FacingMode.User else FacingMode.Environment)
                height(640)
                width(480)
                frameRate(30.0)
              }
            }
          }
        } catch (e: Exception) {
          logger.traceLog("DyteCommonMediaUtils | getVideoStream | error while creating video stream: ${e.message}")
        }
      }
    }

    logger.traceLog("DyteCommonMediaUtils | getVideoStream | returning new video stream")

    return videoMediaStream
  }

  override fun getAudioStream(): MediaStream? {
    if(audioMediaStream != null && audioMediaStream?.audioTracks?.first()?.readyState is MediaStreamTrackState.Live) {
      logger.traceLog("DyteCommonMediaUtils | getAudioStream | returning existing audio stream")
      return audioMediaStream!!
    }

    logger.traceLog("DyteCommonMediaUtils | getAudioStream | creating new audio stream")
    runBlocking {
      try {
        audioMediaStream = MediaDevices.getUserMedia {
          audio(true)
        }
      } catch (e: Exception) {
        logger.traceLog("DyteCommonMediaUtils | getAudioStream | error while creating audio stream: ${e.message}")
      }
    }
    logger.traceLog("DyteCommonMediaUtils | getAudioStream | returning new audio stream")

    return audioMediaStream
  }

  override fun createAudioTrack(): AudioStreamTrack? {
    audioTrack = getAudioStream()?.audioTracks?.first()
    return audioTrack
  }

  override fun createVideoTrack(): VideoStreamTrack? {
    videoTrack = getVideoStream()?.videoTracks?.first()
    return videoTrack
  }

  override fun stopVideo() {
    videoTrack?.enabled = false
    logger.traceLog("DyteCommonMediaUtils | video stopped")
  }

  override fun resumeVideo() {
    videoTrack?.enabled = true
    logger.traceLog("DyteCommonMediaUtils | video resumed")
  }

  override fun switchCamera() {
    runBlocking {
      if (videoTrack?.enabled == true) {
        logger.traceLog("DyteCommonMediaUtils | switchCamera | switching camera")
        videoTrack?.switchCamera()
        cameraFacingUser = !cameraFacingUser
      } else {
        logger.traceLog("DyteCommonMediaUtils | switchCamera | video track is not enabled, not switching camera")
      }
    }
  }

  override fun dispose() {
    // WebRtc.dispose()
    videoMediaStream?.release()
    audioMediaStream?.release()

    logger.traceLog("DyteCommonMediaUtils | audio/video streams disposed")
  }

  override fun onCameraClosed() {
    videoTrack = null
  }

  override fun getVideoTrack(): VideoStreamTrack? {
    return if (videoTrack == null || videoTrack?.readyState is MediaStreamTrackState.Ended)
      createVideoTrack()
    else videoTrack
  }

  override fun setLogger(logger: ILoggerController) {
    this.logger = logger
  }
  override fun setPlatform(platform: String) {
    this.platform = platform;
  }
}

interface IDyteCommonMediaUtils {

  fun getVideoStream(): MediaStream?

  fun getAudioStream(): MediaStream?
  /**
   * Switch camera
   *
   * toggles between front camera and back camera
   */
  fun switchCamera()

  /**
   * Create audio track
   *
   * @return instance of AudioTrack from webrtc
   */
  fun createAudioTrack(): AudioStreamTrack?

  /**
   * Create video track
   *
   * @return instance of videoTrack from webrtc
   */
  fun createVideoTrack(): VideoStreamTrack?

  fun stopVideo()
  fun resumeVideo()

  fun dispose()

  fun onCameraClosed()

  // EglContext for Android can be retrieved by doing: WebRtc.rootEglBase.eglBaseContext

  fun getVideoTrack(): VideoStreamTrack?

  fun setLogger(logger: ILoggerController)
  fun setPlatform(platform: String)
}