package io.dyte.core.media

import io.dyte.callstats.CallStats
import io.dyte.callstats.media.ConsumerFacade
import io.dyte.callstats.media.ProducerFacade
import io.dyte.callstats.media.TransportFacade
import io.dyte.callstats.models.AuthPayload
import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteScreenShareMeetingParticipant
import io.dyte.core.network.BaseApiService
import io.dyte.core.network.models.IceServerData
import io.dyte.core.observability.ILoggerController
import io.dyte.core.platform.IDyteMediaSoupUtils
import io.dyte.core.platform.IDyteMediaUtils
import io.dyte.core.socket.events.payloadmodel.inbound.ConsumerAppData
import io.dyte.core.socket.events.payloadmodel.inbound.ConsumerRtpParameters
import io.dyte.core.socket.events.payloadmodel.inbound.Encodings
import io.dyte.core.socket.events.payloadmodel.inbound.Rtcp
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerModel
import io.dyte.core.socket.events.payloadmodel.outbound.Codec
import io.dyte.core.socket.events.payloadmodel.outbound.CodecParameters
import io.dyte.core.socket.events.payloadmodel.outbound.CodecRtcpFeedback
import io.dyte.core.socket.events.payloadmodel.outbound.Fingerprint
import io.dyte.core.socket.events.payloadmodel.outbound.HeaderExtension
import io.dyte.core.socket.events.payloadmodel.outbound.RouterCapabilitiesModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.media.Consumer
import io.dyte.media.Device
import io.dyte.media.DtlsFingerprint
import io.dyte.media.DtlsParameters
import io.dyte.media.DtlsRole
import io.dyte.media.EmitData
import io.dyte.media.IceCandidate
import io.dyte.media.IceCandidateType
import io.dyte.media.IceParameters
import io.dyte.media.Producer
import io.dyte.media.Protocol
import io.dyte.media.TcpType
import io.dyte.media.Transport
import io.dyte.media.utils.IMediaClientLogger
import io.dyte.media.utils.LocalRtcpParameters
import io.dyte.media.utils.LocalRtpCodecParameters
import io.dyte.media.utils.LocalRtpParameters
import io.dyte.media.utils.RTCRtpMediaType
import io.dyte.media.utils.RtcpFeedback
import io.dyte.media.utils.RtpCapabilities
import io.dyte.media.utils.RtpHeaderDirection
import io.dyte.media.utils.RtpHeaderExtension
import io.dyte.media.utils.RtpHeaderExtensionParameters
import io.dyte.webrtc.AudioStreamTrack
import io.dyte.webrtc.CommonRtpEncodingParameters
import io.dyte.webrtc.IceServer
import io.dyte.webrtc.MediaStreamTrackState
import io.dyte.webrtc.VideoStreamTrack
import io.dyte.webrtc.readyState
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonPrimitive

internal class DyteMediaSoup(private val callStatsClient: CallStats) : IDyteMediaSoupUtils {
  private lateinit var controllerContainer: IControllerContainer
  private lateinit var mediaUtils: IDyteMediaUtils

  private lateinit var logger: ILoggerController

  private var sendTransport: Transport? = null
  private var receiveTransport: Transport? = null

  private var onCameraStreamKilled = false
  private val jsonParser = BaseApiService.json
  private val consumersToParticipants = HashMap<String, DyteJoinedMeetingParticipant>()
  private val consumers = arrayListOf<Consumer>()

  private var micProducer: Producer? = null
  private var cameraProducer: Producer? = null

  private lateinit var mediaSoupDevice: Device

  private var localVideoTrack: VideoStreamTrack? = null
  private var localAudioTrack: AudioStreamTrack? = null

  private val consumerMutex = Mutex()

  @OptIn(ExperimentalCoroutinesApi::class)
  private val serialScope = CoroutineScope(newSingleThreadContext("DyteMediaSoup"))

  override fun init(controllerContainer: IControllerContainer) {
    logger = controllerContainer.loggerController
    controllerContainer.platformUtilsProvider.getMediaUtils().setLogger(logger)
    controllerContainer.platformUtilsProvider.getMediaUtils().setPlatform(controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName())
    this.controllerContainer = controllerContainer
    mediaUtils = controllerContainer.platformUtilsProvider.getMediaUtils()

    startCallStats()

    if (controllerContainer.permissionController.isPermissionGrated(CAMERA) && controllerContainer.presetController.canPublishVideo()) {
      localVideoTrack =
        mediaUtils.getVideoTrack()
      controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
      controllerContainer.selfController.getSelf()._videoEnabled = true
    }

    if (controllerContainer.permissionController.isPermissionGrated(MICROPHONE) && controllerContainer.presetController.canPublishAudio()) {
      localAudioTrack =
        mediaUtils.createAudioTrack()
      controllerContainer.selfController.getSelf()._audioEnabled = true
    }
  }

  private fun startCallStats() {
    callStatsClient.authenticate(AuthPayload(this.controllerContainer.metaController.getPeerId()))

    /*
    * Disabling Pre call tests temporarily for version 1 of CallStats library.
    * */
    /*callStatsClient.sendPreCallTestBeginEvent(object : TestObserver {
      override fun onDone(data: Any) {
        println("DyteClient | DyteMediaSoup | sendPreCallTestBegin onDone: $data")
      }

      override fun onError(ex: Exception) {
        println("DyteClient | DyteMediaSoup | sendPreCallTestBegin onError: ${ex.message}")
      }

      override fun onFailure(reason: String, lastUpdatedResults: Any?) {
        println("DyteClient | DyteMediaSoup | sendPreCallTestBegin onFailure: $reason")
      }
    })*/

    callStatsClient.startPingStats(interval = 7000)
  }

  override suspend fun loadRouterRtpCapabilities(routerRtpCapabilities: RouterCapabilitiesModel) {
    val mediasoupLoggger = MediasoupLoggger(controllerContainer.loggerController)

    mediaSoupDevice = Device(mediasoupLoggger)
    logger.traceLog("DyteMediaSoup | Mediasoup Device created")

    val localCodecs: MutableList<LocalRtpCodecParameters> = mutableListOf()
    val localHeaderExtensions: MutableList<RtpHeaderExtension> = mutableListOf()

    routerRtpCapabilities.codecs?.forEach {
      val localRtcpFeedback: MutableList<RtcpFeedback> = mutableListOf()
      it.rtcpFeedback?.forEach { fb ->
        localRtcpFeedback.add(
          RtcpFeedback(
            type = fb.type!!,
            parameter = fb.parameter!!
          )
        )
      }

      localCodecs.add(
        LocalRtpCodecParameters(
          kind = RTCRtpMediaType.fromString(it.kind!!),
          mimeType = it.mimeType!!,
          preferredPayloadType = it.preferredPayloadType,
          clockRate = it.clockRate!!,
          numChannels = it.channels,
          parameters = mutableMapOf(), //it.parameters as MutableMap<String, String?>,
          rtcpFeedback = localRtcpFeedback
        )
      )
    }

    routerRtpCapabilities.headerExtensions?.forEach {
      localHeaderExtensions.add(
        RtpHeaderExtension(
          kind = RTCRtpMediaType.fromString(it.kind!!),
          uri = it.uri,
          preferredId = it.preferredId,
          preferredEncrypt = it.preferredEncrypt,
          direction = RtpHeaderDirection.fromString(it.direction!!)
        )
      )
    }

    val newRtp = RtpCapabilities(
      codecs = localCodecs,
      headerExtensions = localHeaderExtensions
    )

    localHeaderExtensions.forEachIndexed { index, rtpHeaderExtension ->
      if (rtpHeaderExtension.uri == "urn:3gpp:video-orientation"){
        localHeaderExtensions[index].uri = "do not respect !urn:3gpp:video-orientation!"
      }
    }

    mediaSoupDevice.load(newRtp)
    logger.traceLog("DyteMediaSoup | Mediasoup Device loaded with RTP capabilities")
  }

  override suspend fun createWebRtcTransportRecv(
    model: WebRtcCreateTransportModel,
    iceServers: List<IceServerData>
  ) {
    val id = requireNotNull(model.id)

    val iceServersList = arrayListOf<IceServer>()
    iceServers.filter { it.username?.isNotBlank() ?: false }.forEach {
      val iceServer = IceServer(listOf(it.url), it.username!!, it.credential!!)
      iceServersList.add(iceServer)
    }

    val localIceParameters = IceParameters(
      usernameFragment = model.iceParameters!!.usernameFragment!!,
      password = model.iceParameters!!.password!!,
      iceLite = model.iceParameters!!.iceLite!!
    )

    val localIceCandidate: MutableList<IceCandidate> = mutableListOf()

    model.iceCandidates?.forEach {
      localIceCandidate.add(
        IceCandidate(
          foundation = it.foundation,
          ip = it.ip!!,
          port = it.port!!,
          priority = it.priority!!,
          protocol = Protocol.fromString(it.protocol!!),
          type = IceCandidateType.fromString(it.type!!),
          tcpType = if (it.tcpType != null) TcpType.fromString(it.tcpType!!) else null,
          //Check
          raddr = null,
          rport = null,
          transport = if (it.tcpType != null) "tcp" else "udp"
        )
      )
    }

    val localFingerprints: MutableList<DtlsFingerprint> = mutableListOf()

    model.dtlsParameters?.fingerprints?.forEach {
      localFingerprints.add(
        DtlsFingerprint(
          value = it.value!!,
          algorithm = it.algorithm!!
        )
      )
    }

    val localDtlsParameters = DtlsParameters(
      role = DtlsRole.fromString(model.dtlsParameters!!.role!!),
      fingerprints = localFingerprints
    )

    logger.traceLog("DyteMediaSoup | creating Mediasoup Receive Transport")
    receiveTransport = mediaSoupDevice.createRecvTransport(
      id = id,
      iceParameters = localIceParameters,
      iceCandidates = localIceCandidate,
      dtlsParameters = localDtlsParameters,
      iceServers = iceServersList
    )
    logger.traceLog("DyteMediaSoup | Mediasoup Receive Transport created")

    receiveTransport?.let {
      callStatsClient.registerConsumingTransport(
        TransportFacade(
          isHive = false,
          mediasoupTransport = it
        )
      )
    }

    handleReceiveTransport()
  }

  override suspend fun createWebRtcTransportProd(
    model: WebRtcCreateTransportModel,
    iceServers: List<IceServerData>
  ) {
    val id = requireNotNull(model.id)

    val iceServersList = arrayListOf<IceServer>()
    iceServers.filter { it.username?.isNotBlank() ?: false }.forEach {
      val iceServer = IceServer(listOf(it.url), it.username!!, it.credential!!)
      iceServersList.add(iceServer)
    }

    val localIceParameters = IceParameters(
      usernameFragment = model.iceParameters!!.usernameFragment!!,
      password = model.iceParameters!!.password!!,
      iceLite = model.iceParameters!!.iceLite!!
    )

    val localIceCandidate: MutableList<IceCandidate> = mutableListOf()

    model.iceCandidates?.forEach {
      localIceCandidate.add(
        IceCandidate(
          foundation = it.foundation,
          ip = it.ip!!,
          port = it.port!!,
          priority = it.priority!!,
          protocol = Protocol.fromString(it.protocol!!),
          type = IceCandidateType.fromString(it.type!!),
          tcpType = if (it.tcpType != null) TcpType.fromString(it.tcpType!!) else null,
          //Check
          raddr = null,
          rport = null,
          transport = if (it.tcpType != null) "tcp" else "udp"
        )
      )
    }

    val localFingerprints: MutableList<DtlsFingerprint> = mutableListOf()

    model.dtlsParameters?.fingerprints?.forEach {
      localFingerprints.add(
        DtlsFingerprint(
          value = it.value!!,
          algorithm = it.algorithm!!
        )
      )
    }

    val localDtlsParameters = DtlsParameters(
      role = DtlsRole.fromString(model.dtlsParameters!!.role!!),
      fingerprints = localFingerprints
    )

    logger.traceLog("DyteMediaSoup | creating Mediasoup Send Transport")
    sendTransport = mediaSoupDevice.createSendTransport(
        id = id,
        iceParameters = localIceParameters,
        iceCandidates = localIceCandidate,
        dtlsParameters = localDtlsParameters,
        iceServers = iceServersList
    )
    logger.traceLog("DyteMediaSoup | Mediasoup Send Transport created")

    sendTransport?.let {
      callStatsClient.registerProducingTransport(
        TransportFacade(
          isHive = false,
          mediasoupTransport = it
        )
      )
    }

    handleProduceTransport()
    produceMedia()
  }

  override suspend fun produceMedia() {
    val videoEnableByDeveloper = controllerContainer.metaController.isVideoEnabled()

    if (controllerContainer.presetController.canPublishVideo() && videoEnableByDeveloper) {
      logger.traceLog("DyteMediaSoup | publishing video")
      connectCameraTransport()
    }

    val audioEnableByDeveloper = controllerContainer.metaController.isAudioEnabled()

    if (controllerContainer.presetController.canPublishAudio() && audioEnableByDeveloper) {
      logger.traceLog("DyteMediaSoup | publishing audio")
      connectAudioTransport("")
    }
  }
  override fun handleNewConsumer(
    webSocketConsumerModel: WebSocketConsumerModel,
    onDone: () -> Unit
  ) {
    serialScope.launch {
      consumerMutex.withLock {
        _handleNewConsumer(webSocketConsumerModel, onDone)
      }
    }
  }
  suspend fun _handleNewConsumer(
    webSocketConsumerModel: WebSocketConsumerModel,
    onDone: () -> Unit
  ) {
    controllerContainer.loggerController.traceLog("createConsumer::initializing_consumer")
      val participant =
        controllerContainer.participantController.meetingRoomParticipants.joined.find { it.id == webSocketConsumerModel.peerId }

      participant?.let {
        val id = requireNotNull(webSocketConsumerModel.id)
        val producerId = requireNotNull(webSocketConsumerModel.producerId)
        val kind = requireNotNull(webSocketConsumerModel.kind)
        val rtpParameters = webSocketConsumerModel.rtpParameters
        val enc = arrayListOf<CommonRtpEncodingParameters>()

        rtpParameters!!.encodings?.forEach {
          val a = CommonRtpEncodingParameters()
          a.ssrc = it.ssrc!!.toLong()
          enc.add(a)
        }

        val localRtpParameters = LocalRtpParameters(
          rtpParameters.mid!!,
          rtpParameters.codecs!!.map {
            LocalRtpCodecParameters(
              if (it.mimeType!!.contains("video")) RTCRtpMediaType.RTCRtpMediaTypeVideo else RTCRtpMediaType.RTCRtpMediaTypeAudio,
              it.mimeType!!,
              it.preferredPayloadType,
              it.clockRate,
              it.channels,
              mutableMapOf(
                "apt" to JsonPrimitive(it.parameters!!.apt),
                "profileId" to JsonPrimitive(it.parameters!!.profileId),
                "profileLevelId" to JsonPrimitive(it.parameters!!.profileLevelId),
                "packetizationMode" to JsonPrimitive(it.parameters!!.packetizationMode),
                "levelAsymmetryAllowed" to JsonPrimitive(it.parameters!!.levelAsymmetryAllowed)
              ),
              it.rtcpFeedback!!.map { feedback -> RtcpFeedback(feedback.type, feedback.parameter) }
                .toMutableList(),
              it.payloadType
            )
          }.toMutableList(),
          rtpParameters.headerExtensions!!.map {
            RtpHeaderExtensionParameters(
              it.uri,
              it.id,
              it.encrypt
            )
          }.toMutableList(),
          enc,
          LocalRtcpParameters(
            rtpParameters.rtcp!!.mux,
            rtpParameters.rtcp.cname!!,
            rtpParameters.rtcp.reducedSize!!
          )
        )


        logger.traceLog("createConsumer::Creating consumer with ID - $id | kind $kind")
        val consumer = receiveTransport!!.consume(
          id = id,
          producerId = producerId,
          kind = RTCRtpMediaType.fromString(kind),
          rtpParameters = localRtpParameters,
          appData = mapOf(
            "peerId" to webSocketConsumerModel.appData!!.peerId,
            "screenShare" to webSocketConsumerModel.appData!!.screenShare!!
          ),
          peerId = webSocketConsumerModel.peerId!!
        )

        consumers.add(consumer)
        consumersToParticipants[id] = participant

        controllerContainer.loggerController.traceLog("createConsumer::initialized_consumer")
        callStatsClient.registerConsumer(
          ConsumerFacade(
            isHive = false,
            mediasoupConsumer = consumer
          )
        )
        onDone.invoke()
      }
  }

  override fun handleCloseConsumer(webSocketConsumerModel: WebSocketConsumerClosedModel) {
    webSocketConsumerModel.id?.let { consumerId ->
      logger.traceLog("DyteMediaSoup | Closing consumer $consumerId")

      val consumer = consumers.find { it.id == consumerId }
      if (consumer != null) {

        val appData = ConsumerAppData(
          peerId = consumer.appData["peerId"] as String,
          screenShare = consumer.appData["screenShare"] as? Boolean ?: false
        )

        if (appData.screenShare == true) {
          val joinedParticipant = requireNotNull(consumersToParticipants[webSocketConsumerModel.id])

          val screenShareParticipant =
            controllerContainer.participantController.meetingRoomParticipants.screenshares.find { it.id == joinedParticipant.id }
          screenShareParticipant?.let {
            it._screenShareTrack = null
            controllerContainer.participantController.onPeerScreenSharedEnded(it)
          }
        }
        consumers.remove(consumer)
        consumersToParticipants.remove(consumerId)

        logger.traceLog("DyteMediaSoup | Consumer $consumerId closed successfully")
      } else {
        logger.traceError("DyteMediaSoup | Consumer $consumerId not found")
      }
    }
  }

  override fun resumeConsumer(id: String) {
    serialScope.launch {
      consumerMutex.withLock {
        _resumeConsumer(id)
      }
    }
  }

  fun _resumeConsumer(id: String) {
    println("DyteMobileClient | DyteMediaSoup _resumeConsumer $id")
      val consumer = consumers.find { it.id == id }
      if (consumer != null) {
        val participant = requireNotNull(consumersToParticipants[id])

        val isAudioTrack = consumer.track is AudioStreamTrack

        if (isAudioTrack) {
//          participant._audioEnabled = (consumer.track as AudioStreamTrack).enabled
//          controllerContainer.eventController.triggerEvent(
//            DyteEventType.OnPeerAudioUpdate(participant)
//          )
        } else {
          if (consumer.appData["screenShare"] as? Boolean == true) {
            controllerContainer.participantController.onPeerScreenShareStarted(
              DyteScreenShareMeetingParticipant.get(participant),
              consumer.track as VideoStreamTrack
            )
          } else {
            participant._videoTrack = consumer.track as VideoStreamTrack
            participant._videoEnabled = (consumer.track as VideoStreamTrack).enabled
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnPeerVideoUpdate(participant)
            )
          }
        }
      } else {
        controllerContainer.loggerController.traceError("DyetMediaSoup | Consumer $id not found")
      }
  }

  override suspend fun connectCameraTransport() {
    if (controllerContainer.presetController.canPublishVideo()
      && controllerContainer.permissionController.isPermissionGrated(CAMERA)
    ) {
      enableCamImpl()
    } else {
      controllerContainer.loggerController.traceError("DyteMediaSoup | Cannot publish video")
    }
  }

  override suspend fun connectAudioTransport(id: String) {
    if (controllerContainer.presetController.canPublishAudio()
      && controllerContainer.permissionController.isPermissionGrated(MICROPHONE)
    ) {
      enableMicImpl()
    } else {
      controllerContainer.loggerController.traceError("DyteMediaSoup | Cannot publish audio")
    }
  }

  private suspend fun enableCamImpl() {
    try {
      if (cameraProducer != null) {
        logger.traceError("DyteMediaSoup | Camera producer already exists")
        return
      }
      if (!mediaSoupDevice.loaded) {
        logger.traceError("DyteMediaSoup | Device not loaded")
        return
      }
      if (!mediaSoupDevice.canProduce(RTCRtpMediaType.RTCRtpMediaTypeVideo)) {
        logger.traceError("DyteMediaSoup | Device cannot produce video")
        return
      }
      if (!controllerContainer.selfController.getSelf()._videoEnabled) {
        logger.traceError("DyteMediaSoup | Not creating camera producer as video is disabled")
        return
      }
      if (localVideoTrack == null) {
        localVideoTrack =
          mediaUtils.getVideoTrack()
        localVideoTrack?.enabled = true
      }
      logger.traceLog("DyteMediaSoup | Creating camera producer")
      createCameraProducer()
    } catch (e: Exception) {
      e.printStackTrace()
    }
  }

  private suspend fun enableMicImpl() {
    try {
      if (micProducer != null) {
        logger.traceLog("DyteMediaSoup | Mic producer already exists")
        return
      }
      if (!mediaSoupDevice.loaded) {
        logger.traceLog("DyteMediaSoup | Device not loaded")
        return
      }
      if (!mediaSoupDevice.canProduce(RTCRtpMediaType.RTCRtpMediaTypeAudio)) {
        logger.traceLog("DyteMediaSoup | Device cannot produce audio")
        return
      }
      if (localAudioTrack == null) {
        localAudioTrack =
          mediaUtils.createAudioTrack()
        localAudioTrack?.enabled = true
      }
      if (controllerContainer.selfController.getSelf().audioEnabled) {
        logger.traceLog("DyteMediaSoup | Creating mic producer")
        createMicProducer()
      } else {
        logger.traceLog("DyteMediaSoup | Not creating mic producer as audio is disabled")
      }
    } catch (e: Exception) {
      e.printStackTrace()
      if (localAudioTrack != null) {
        localAudioTrack?.enabled = false
      }
    }
  }

  private suspend fun createMicProducer() {
    if (sendTransport == null || localAudioTrack == null) {
      logger.traceError("DyteMediaSoup | Send transport or local audio track is null")
      return
    }
    if (controllerContainer.presetController.canPublishAudio().not()) {
      logger.traceError("DyteMediaSoup | Cannot publish audio")
      return
    }
    if (micProducer == null) {
      val data = ConsumerAppData(false, controllerContainer.metaController.getPeerId())

      micProducer = sendTransport!!.produce(
        track = localAudioTrack!!,
        stream = mediaUtils.getAudioStream()!!,
        //codec = mediaSoupDevice.rtpCapabilities()?.codecs?.first{ it -> it.mimeType.lowercase() == "audio/opus"},
        appData = mapOf("peerId" to data.peerId),
        source = ""
      )
      
      logger.traceLog("DyteMediaSoup | Mic producer created successfully")
      micProducer?.let {
        callStatsClient.registerProducer(
          ProducerFacade(
            isHive = false,
            mediasoupProducer = it
          )
        )
      }
    } else {
      logger.traceLog("DyteMediaSoup | Mic producer already exists")
    }
  }


  private suspend fun createCameraProducer() {
    if (cameraProducer != null && cameraProducer?.closed == false) {
      logger.traceError("DyteMediaSoup | Camera producer already exists and is open")
      return
    }
    if (sendTransport == null || localVideoTrack == null) {
      logger.traceError("DyteMediaSoup | Send transport or local video track is null")
      return
    }
    if (controllerContainer.presetController.canPublishVideo().not()) {
      logger.traceError("DyteMediaSoup | Cannot publish video")
      return
    }
    val data = ConsumerAppData(false, controllerContainer.metaController.getPeerId())

    if (localVideoTrack?.readyState is MediaStreamTrackState.Ended) {
      localVideoTrack = mediaUtils.getVideoTrack()
    }
    controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack

    cameraProducer = sendTransport!!.produce(
      track = localVideoTrack!!,
      stream = mediaUtils.getVideoStream()!!,
      codec = mediaSoupDevice.rtpCapabilities()?.codecs?.first { it.mimeType.lowercase() == "video/vp8" },
      appData = mapOf("peerId" to data.peerId), //BaseApiService.json.encodeToString(data) as Map<String, Any>,
      source = ""
    )

    logger.traceLog("DyteMediaSoup | Camera producer created successfully")

    cameraProducer?.let {
      callStatsClient.registerProducer(
        ProducerFacade(
          isHive = false,
          mediasoupProducer = it
        )
      )
    }
  }

  override suspend fun muteSelfAudio(): Boolean {
    logger.traceLog("DyteMediaSoup | Muting self audio")
    callStatsClient.sendAudioToggleEvent(false)
    localAudioTrack?.enabled = false
    return true
  }

  override suspend fun unmuteSelfAudio(): Boolean {
    logger.traceLog("DyteMediaSoup | Unmuting self audio")
    createMicProducer()
    callStatsClient.sendAudioToggleEvent(true)
    localAudioTrack?.enabled = true
    return true
  }

  override suspend fun muteSelfVideo(): Boolean {
    logger.traceLog("DyteMediaSoup | Muting self video")
    callStatsClient.sendVideoToggleEvent(false)
    cameraProducer?.let { producer ->
      producer.close()
      controllerContainer.mediaSoupController.videoProduceId?.let { producerId ->
        controllerContainer.mediaSoupController.onProducerClose(producerId)
      }
    }
    localVideoTrack = null
    return true
  }

  override suspend fun unmuteSelfVideo(): Boolean {
    logger.traceLog("DyteMediaSoup | Unmuting self video")
    callStatsClient.sendVideoToggleEvent(true)
    if (onCameraStreamKilled || localVideoTrack == null || localVideoTrack?.readyState is MediaStreamTrackState.Ended) {
      localVideoTrack =
        mediaUtils.getVideoTrack()
    }

    localVideoTrack?.enabled = true

    createCameraProducer()
    /*
    serialScope.launch {
      cameraProducer?.resume()
      controllerContainer.mediaSoupController.onProducerResume(controllerContainer.mediaSoupController.videoProduceId!!)
    }
    */

    controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
    return true
  }

  override suspend fun leaveCall() {
    logger.traceLog("DyteMediaSoup | Disposing media to leave call")

    try {
      serialScope.launch {
        val localConsumers = ArrayList(consumers)
        localConsumers.forEach { consumer ->
          if (!consumer.closed) {
            consumersToParticipants.remove(consumer.id)
            consumer.close()
            consumers.remove(consumer)
          }
        }

        logger.traceLog("DyteMediaSoup | Consumers closed")
      }

      mediaUtils.dispose()
      receiveTransport?.close()
      sendTransport?.close()
      receiveTransport = null
      sendTransport = null

      cameraProducer = null
      micProducer = null

      localAudioTrack = null
      localVideoTrack = null
      callStatsClient.stopPingStats()
      callStatsClient.sendDisconnectEvent()
      logger.traceLog("DyteMediaSoup | Transports closed")
    } catch (e: Exception) {
      e.printStackTrace()
    }
  }

  override fun getSelfTrack(): Pair<Any?, VideoStreamTrack?> {
    return Pair(localAudioTrack, localVideoTrack)
  }

  override fun switchCamera() {
    if (localVideoTrack?.enabled == true) {
      mediaUtils.switchCamera()
    }
  }

  override fun handleResumeConsumer(id: String) {
    TODO("Not yet implemented")
  }

  override fun handlePauseConsumer(id: String) {
    TODO("Not yet implemented")
  }

  override fun onCameraStreamKilled() {
    onCameraStreamKilled = true
  }

  private fun handleReceiveTransport() {
    receiveTransport!!.observer.onEach {
      when (it.name) {
        "connect" -> {
          logger.traceLog("recvTransport::transport_status::connect")

          val mediasoupDtlsParameters = it.data!!["dtlsParameters"] as DtlsParameters
          val listenerFingerpints: MutableList<Fingerprint> = mutableListOf()

          mediasoupDtlsParameters.fingerprints.forEach { fingerprint ->
            listenerFingerpints.add(
              Fingerprint(
                algorithm = fingerprint.algorithm,
                value = fingerprint.value
              )
            )
          }

          val localDtlsParameters =
            io.dyte.core.socket.events.payloadmodel.outbound.DtlsParameters(
              role = DtlsRole.value(mediasoupDtlsParameters.role),
              fingerprints = listenerFingerpints
            )

          controllerContainer.mediaSoupController.onReceiveTransportConnected(
            receiveTransport!!.id,
            Json.encodeToString(localDtlsParameters)
          )
        }

        "connectionstatechanged" -> {
          logger.traceLog("recvTransport::transport_status_changed::${it.data?.get("state")}")
          controllerContainer.connectionController.onReceiveTransportStateChanged(
            it.data?.get("state").toString()
          )
        }

        "close" -> {
          logger.traceLog("DyteMediaSoup | Receive transport onClose")
          consumers.forEach { consumer ->
            callStatsClient.registerConsumerListenerOnTransportClose(
              ConsumerFacade(
                isHive = false,
                mediasoupConsumer = consumer
              )
            )
          }
        }
      }
    }.launchIn(serialScope)
  }

  private fun handleProduceTransport() {
    sendTransport!!.observer.onEach {
      when (it.name) {
        "connect" -> {
          logger.traceLog("sendTransport::ice_state_connected")

          val mediasoupDtlsParameters = it.data!!["dtlsParameters"] as DtlsParameters
          val listenerFingerpints: MutableList<Fingerprint> = mutableListOf()

          mediasoupDtlsParameters.fingerprints.forEach { fingerprint ->
            listenerFingerpints.add(
              Fingerprint(
                algorithm = fingerprint.algorithm,
                value = fingerprint.value
              )
            )
          }

          val localDtlsParameters =
            io.dyte.core.socket.events.payloadmodel.outbound.DtlsParameters(
              role = DtlsRole.value(mediasoupDtlsParameters.role),
              fingerprints = listenerFingerpints
            )

          controllerContainer.mediaSoupController.onSendTransportConnected(
            sendTransport!!.id,
            Json.encodeToString(localDtlsParameters)
          )
        }

        "connectionstatechanged" -> {
          logger.traceLog("sendTransport::transport_status_changed::${it.data?.get("state")}")
          controllerContainer.connectionController.onSendTransportStateChanged(
            it.data?.get("state").toString()
          )
        }

        "produce" -> {
          logger.traceLog("DyteMediaSoup | Send transport onProduce")

          val mediasoupRtpParameters = it.data?.get("rtpParameters") as LocalRtpParameters

          val localCodecs: MutableList<Codec> = mutableListOf()
          val localHeaderExtensions: MutableList<HeaderExtension> =
            mutableListOf()
          val localEncodings: MutableList<Encodings> = mutableListOf()

          val localRtcp = Rtcp(
            cname = mediasoupRtpParameters.rtcp?.cname,
            reducedSize = mediasoupRtpParameters.rtcp?.reducedSize,
            mux = mediasoupRtpParameters.rtcp?.mux
          )

          var localkind: String

          mediasoupRtpParameters.codecs.forEach { parameters ->
            val localRtcpFeedback: MutableList<CodecRtcpFeedback> = mutableListOf()

            parameters.rtcpFeedback.forEach { fb ->
              localRtcpFeedback.add(
                CodecRtcpFeedback(
                  type = fb.type,
                  parameter = fb.parameter
                )
              )
            }

            localkind = RTCRtpMediaType.value(parameters.kind)

            localCodecs.add(
              Codec(
                kind = RTCRtpMediaType.value(parameters.kind),
                mimeType = parameters.mimeType,
                payloadType = parameters.payloadType,
                clockRate = parameters.clockRate,
                channels = if (localkind == "audio") 2 else parameters.numChannels,
                rtcpFeedback = localRtcpFeedback,
                parameters = CodecParameters(
                  xGoogleStartBitrate = parameters.parameters["x-google-start-bitrate"] as Int?,
                  apt = parameters.parameters["apt"] as Int?,
                  profileId = parameters.parameters["profileId"] as Int?,
                  packetizationMode = parameters.parameters["packetization-mode"] as Int?,
                  levelAsymmetryAllowed = parameters.parameters["level-asymmetry-allowed"] as Int?,
                  profileLevelId = parameters.parameters["profile-level-id"] as String?
                ),
                preferredPayloadType = parameters.preferredPayloadType
              )
            )
          }

          mediasoupRtpParameters.headerExtension.forEach { parameters ->
            localHeaderExtensions.add(
              HeaderExtension(
                uri = parameters.uri,
                id = parameters.id,
                encrypt = parameters.encrypted,
                // parameters = ??
              )
            )
          }

          mediasoupRtpParameters.encodings?.forEach { params ->
            localEncodings.add(
              Encodings(
                ssrc = params.ssrc?.toInt(),
                // scalabilityMode = ??
              )
            )
          }

          val produceRtpParameters = ConsumerRtpParameters(
            codecs = localCodecs,
            headerExtensions = localHeaderExtensions,
            encodings = localEncodings,
            rtcp = localRtcp,
            mid = mediasoupRtpParameters.mid
          )

          val produceRtpParamsString = Json.encodeToString(produceRtpParameters)

          val appDataString = Json.encodeToString(
            ConsumerAppData.serializer(),
            ConsumerAppData(
              peerId = (it.data?.get("appData") as Map<String, Any>)["peerId"].toString()
            )
          )

          val producerId = controllerContainer.mediaSoupController.onProduce(
            sendTransport!!.id,
            it.data?.get("kind").toString().lowercase(), produceRtpParamsString, appDataString
          )

          sendTransport!!.externalObserver.emit(EmitData("returnProduce", mapOf("producerId" to producerId)))
        }

        "close" -> {
          logger.traceLog("DyteMediaSoup | Send transport onClose")
          micProducer?.let { producer ->
            callStatsClient.registerProducerListenerOnTransportClose(
              ProducerFacade(
                isHive = false,
                mediasoupProducer = producer
              )
            )
          }
          cameraProducer?.let { producer ->
            callStatsClient.registerProducerListenerOnTransportClose(
              ProducerFacade(
                isHive = false,
                mediasoupProducer = producer
              )
            )
          }
        }
        "newproducer" -> {}
        "producerClose" -> {
          // We should receive a transport here now when producer.close() is called
        }
      }
    }.launchIn(serialScope)
  }

  override fun handlePreReconnection() {
    consumers.clear()
    consumersToParticipants.clear()
  }
}

class MediasoupLoggger(private val logger: ILoggerController): IMediaClientLogger {
  override fun traceError(message: String) {
    logger.traceError(message)
  }

  override fun traceLog(message: String) {
    logger.traceLog(message)
  }

  override fun traceWarning(message: String) {
    logger.traceWarning(message)
  }
}