package io.dyte.core.waitingroom

import io.dyte.core.Result
import io.dyte.core.controllers.ParticipantController
import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.feat.DyteWaitlistedParticipant
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.host.IHostController
import io.dyte.core.listeners.DyteWaitlistEventsListener
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.observability.DyteLogger
import kotlinx.coroutines.CoroutineScope

internal abstract class BaseWaitlistController(
  protected val internalEventsEmitter: EventEmitter<InternalEvents>,
  protected val selfParticipant: DyteSelfParticipant,
  protected val scope: CoroutineScope,
  private val participantController: ParticipantController,
  private val hostController: IHostController,
) : EventEmitter<DyteWaitlistEventsListener>(), IWaitlistController, InternalEvents {
  override val waitlistedParticipants = arrayListOf<DyteWaitlistedParticipant>()

  protected val selfPermissions = selfParticipant.permissions
  protected val logger: DyteLogger = DyteLogger

  override suspend fun acceptAllWaitingRequests() {
    if (selfPermissions.waitingRoom.canAcceptRequests.not()) {
      // note(swapnil): disabling throwing exception as this will cause SDK to crash
      // throw UnsupportedOperationException("not permitted to accept all waitlisted request")

      logger.warn("WaitlistController::acceptAllWaitingRequests::self cannot accept requests")
      return
    }

    if (waitlistedParticipants.isEmpty()) {
      logger.info("WaitlistController::acceptAllWaitingRequests::waiting room empty")
      return
    }

    val result = acceptAllWaitingRequestsInternal(waitlistedParticipants.toList())
    when (result) {
      is Result.Success -> {
        for (acceptedParticipant in result.value) {
          waitlistedParticipants.remove(acceptedParticipant)
          emitEvent { it.onWaitListParticipantAccepted(acceptedParticipant) }
        }
      }
      is Result.Failure -> {
        logger.error("WaitlistController::acceptAllWaitingRequests", result.value)
      }
    }
  }

  protected abstract suspend fun acceptAllWaitingRequestsInternal(
    waitlist: List<DyteWaitlistedParticipant>
  ): Result<List<DyteWaitlistedParticipant>, Exception>

  override suspend fun acceptWaitlistedRequest(dyteMeetingParticipant: DyteWaitlistedParticipant) {
    if (selfPermissions.waitingRoom.canAcceptRequests.not()) {
      // note(swapnil): disabling throwing exception as this will cause SDK to crash
      // throw UnsupportedOperationException("not permitted to accept waitlisted request")

      logger.warn("WaitlistController::acceptWaitlistedRequest::self cannot accept request")
      return
    }

    val result = acceptWaitlistedRequestInternal(dyteMeetingParticipant)
    if (result is Result.Failure) {
      logger.error(
        "WaitlistController::acceptWaitlistedRequest::failed for ${dyteMeetingParticipant.id}",
        result.value,
      )
      return
    }

    waitlistedParticipants.remove(dyteMeetingParticipant)
    emitEvent { it.onWaitListParticipantAccepted(dyteMeetingParticipant) }
  }

  protected abstract suspend fun acceptWaitlistedRequestInternal(
    waitlistedParticipant: DyteWaitlistedParticipant
  ): Result<Unit, Exception>

  override suspend fun rejectWaitlistedRequest(dyteMeetingParticipant: DyteWaitlistedParticipant) {
    if (selfPermissions.waitingRoom.canAcceptRequests.not()) {
      // note(swapnil): disabling throwing exception as this will cause SDK to crash
      // throw UnsupportedOperationException("not permitted to reject waitlisted request")

      logger.warn("WaitlistController::rejectWaitlistedRequest::self cannot reject request")
      return
    }

    val result = rejectWaitlistedRequestInternal(dyteMeetingParticipant)
    if (result is Result.Failure) {
      logger.error(
        "WaitlistController::rejectWaitlistedRequest::failed for ${dyteMeetingParticipant.id}",
        result.value,
      )
      return
    }

    waitlistedParticipants.remove(dyteMeetingParticipant)
    emitEvent { it.onWaitListParticipantRejected(dyteMeetingParticipant) }
  }

  protected abstract suspend fun rejectWaitlistedRequestInternal(
    waitlistedParticipant: DyteWaitlistedParticipant
  ): Result<Unit, Exception>

  override fun emitEvent(event: (listener: DyteWaitlistEventsListener) -> Unit) {
    super.emitEvent(event)
    participantController.emitEvent {
      it.onUpdate(DyteRoomParticipants(participantController, hostController, this))
    }
  }

  protected fun addParticipantToWaitlist(id: String, userId: String?, displayName: String?) {
    val participant =
      DyteWaitlistedParticipant(
        id,
        userId ?: "",
        displayName ?: "",
        null,
        false,
        null,
        ParticipantFlags(
          recorder = false,
          hiddenParticipant = false,
          webinarHiddenParticipant = false,
        ),
        "",
        participantController,
        this@BaseWaitlistController,
        scope,
      )
    if (waitlistedParticipants.none { it.id == participant.id }) {
      waitlistedParticipants.add(participant)
      emitEvent { it.onWaitListParticipantJoined(participant) }
    }
  }

  protected fun removeParticipantFromWaitlist(peerId: String) {
    val waitlistedParticipant = waitlistedParticipants.find { it.id == peerId } ?: return
    waitlistedParticipants.remove(waitlistedParticipant)
    emitEvent { it.onWaitListParticipantClosed(waitlistedParticipant) }
  }

  protected fun refreshWaitlistOnPeerJoined(peerId: String) {
    val waitlistedParticipant = waitlistedParticipants.find { it.id == peerId } ?: return
    waitlistedParticipants.remove(waitlistedParticipant)
    emitEvent { it.onWaitListParticipantAccepted(waitlistedParticipant) }
  }
}

internal interface IWaitlistController {
  val waitlistedParticipants: List<DyteWaitlistedParticipant>

  suspend fun acceptAllWaitingRequests()

  suspend fun acceptWaitlistedRequest(dyteMeetingParticipant: DyteWaitlistedParticipant)

  suspend fun rejectWaitlistedRequest(dyteMeetingParticipant: DyteWaitlistedParticipant)
}
