package io.dyte.core.plugins

import io.dyte.core.Result
import io.dyte.core.network.IApiClient
import io.dyte.core.observability.DyteLogger
import io.dyte.core.utils.JsonUtils
import io.ktor.http.URLBuilder
import io.ktor.http.set
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.put
import kotlinx.serialization.json.putJsonObject

// TODO(swapnil): Make PluginModel abstract and implement for RoomNode, SocketService for
// formatSendData
internal class PluginModel(
  private val id: String,
  private val baseUrl: String,
  private val dyteBaseUrl: String,
  private val pluginsApi: IApiClient,
  private val pluginSocketServer: String,
) {
  private val logger: DyteLogger = DyteLogger

  var enabledBy: String? = null
    private set

  var authToken: String? = null
    private set

  var urlToLoad: String? = null
    private set

  var active: Boolean = false
    private set

  /** Activates plugin locally for self. */
  suspend fun enableLocal(enabledBy: String): Result<Unit, Exception> {
    val active = this.active // mutex.withLock { this.active }

    if (active) {
      logger.info("PluginModel::enableLocal::success already active")
      return Result.Success(Unit) // Expected: Result.Failure(Exception("Plugin already active"))
    }

    try {
      val pluginAuthToken = pluginsApi.authorizePlugin(id).data.token
      val pluginUrl = URLBuilder(baseUrl)
      pluginUrl.set {
        parameters["auth"] = pluginAuthToken
        parameters["pluginId"] = id
        parameters["backend"] = dyteBaseUrl
        encodedParameters["parent"] = "*"
      }

      if (!this.active) {
        authToken = pluginAuthToken
        urlToLoad = pluginUrl.buildString()
        this.enabledBy = enabledBy
        this.active = true
      }
      logger.info("PluginModel::enableLocal::success")
      return Result.Success(Unit)
    } catch (e: Exception) {
      logger.error("PluginModel::enableLocal::failure", e)
      return Result.Failure(e)
    }
  }

  /** Deactivates plugin locally for self. */
  fun disableLocal(): Result<Unit, Exception> {
    if (active) {
      enabledBy = null
      authToken = null
      urlToLoad = null
      active = false
    }
    logger.info("PluginModel::disableLocal::success")
    return Result.Success(Unit)
  }

  fun formatSendDataPayload(eventName: String, data: Any?): PluginMessage? {
    val dataJsonElement: JsonElement =
      JsonUtils.serializeAny(data)
        ?: run {
          logger.info("PluginModel::formatSendDataPayload::invalid data for $eventName")
          return null
        }

    val sendDataPayloadJsonObject = buildJsonObject {
      put(KEY_SEND_DATA_EVENT_NAME, eventName)
      put(KEY_SEND_DATA_DATA, dataJsonElement)
    }

    val sendDataJsonObject =
      if (pluginSocketServer == "socket-service") {
        createSocketServiceSendDataJsonObject(sendDataPayloadJsonObject)
      } else {
        createRoomNodeSendDataJsonObject(sendDataPayloadJsonObject)
      }

    return PluginMessage(sendDataJsonObject)
  }

  private fun createRoomNodeSendDataJsonObject(dataJsonObject: JsonObject): JsonObject {
    return buildJsonObject {
      put("type", TYPE_PLUGIN_INTERNAL_DATA)
      putJsonObject("payload") {
        put("pluginId", id)
        put("data", dataJsonObject)
      }
    }
  }

  private fun createSocketServiceSendDataJsonObject(payloadJsonObject: JsonObject): JsonObject {
    return buildJsonObject {
      put("type", LocalPluginEvent.SEND_DATA.id)
      put("uuid", "")
      put("payload", payloadJsonObject)
    }
  }

  companion object {
    private const val KEY_SEND_DATA_EVENT_NAME = "eventName"
    private const val KEY_SEND_DATA_DATA = "data"
    private const val TYPE_PLUGIN_INTERNAL_DATA = "websocket/plugin-internal-data"
  }
}
