package io.dyte.core.feat

import io.dyte.core.Result
import io.dyte.core.chat.BaseChatController
import io.dyte.core.chat.channel.ChatChannelController
import io.dyte.core.chat.models.ChatChannel
import io.dyte.core.chat.models.ChatChannel.ChannelVisibility
import io.dyte.core.chat.models.ChatChannelUpdateParams
import io.dyte.core.chat.models.GetMessagesResult
import io.dyte.core.events.PublicAPIEmitter
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteParticipant
import io.dyte.core.platform.Uri
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext

// TODO : add pin/unpin support
// TODO: add pinned message type.
// TODO : getMessagesByUser
// TODO : getMessageByType
class DyteChat
internal constructor(
  internal val chatController: BaseChatController,
  internal val chatChannelController: ChatChannelController?,
  private val serialScope: CoroutineScope,
) : PublicAPIEmitter<DyteChatEventsListener>() {

  override val emitterSuperClass = chatController

  val messages: List<DyteChatMessage>
    get() = runBlocking(serialScope.coroutineContext) { chatController.messages }

  val privateChatMessages: List<DyteChatMessage>
    get() = runBlocking(serialScope.coroutineContext) { chatController.privateChatMessages }

  val channels: List<ChatChannel>
    get() =
      runBlocking(serialScope.coroutineContext) { chatChannelController?.channels ?: emptyList() }

  fun getPrivateChatMessages(participant: DyteJoinedMeetingParticipant): List<DyteChatMessage> {
    return chatController.getPrivateChatMessages(participant)
  }

  fun toMap(): List<Map<String, Any?>> {
    return messages.map { encodeDyteChatMessage(it) }
  }

  /**
   * Sends a string type chat message in the given room
   *
   * @param message
   */
  @Throws(Exception::class)
  fun sendTextMessage(message: String) {
    sendTextMessage(message, listOf())
  }

  @Throws(Exception::class)
  fun sendTextMessage(message: String, peerIds: List<String>) {
    serialScope.launch { chatController.sendMessageToPeers(message, peerIds) }
  }

  @Deprecated(
    message =
      "Sending a file message no longer needs fileName parameter. This method will be removed in next version of the SDK.",
    replaceWith = ReplaceWith("sendFileMessage(fileUri: Uri)"),
  )
  /**
   * Send file message
   *
   * @param filePath Location of file on device storage
   * @param fileName Name of the file, displayed on chat screen
   */
  @Throws(Exception::class)
  fun sendFileMessage(filePath: String, @Suppress("UNUSED_PARAMETER") fileName: String) {
    serialScope.launch { chatController.sendFileMessageToPeers(filePath, listOf()) }
  }

  @Deprecated(
    message =
      "Sending an image message no longer needs fileName parameter. This method will be removed in next version of the SDK.",
    replaceWith = ReplaceWith("sendImageMessage(imageUri: Uri)"),
  )
  /**
   * Send chat image message
   *
   * @param filePath Location of image file on device storage
   * @param fileName Name of the image file, displayed on chat screen
   */
  @Suppress("UNUSED_PARAMETER")
  @Throws(Exception::class)
  fun sendImageMessage(filePath: String, fileName: String) {
    sendImageMessage(filePath, listOf())
  }

  fun sendImageMessage(filePath: String, fileName: String, peerIds: List<String>) {
    serialScope.launch { chatController.sendImageMessageToPeers(filePath, peerIds) }
  }

  /**
   * Send chat image message
   *
   * @param imageUri uri of image
   */
  fun sendImageMessage(imageUri: Uri) {
    sendImageMessage(imageUri, listOf())
  }

  fun sendImageMessage(imageUri: Uri, peerIds: List<String>) {
    serialScope.launch { chatController.sendImageMessageToPeers(imageUri, peerIds) }
  }
  /**
   * Send chat image message
   *
   * @param imagePath Location of image file on device storage
   */
  fun sendImageMessage(imagePath: String) {
    sendImageMessage(imagePath, listOf())
  }

  fun sendImageMessage(imagePath: String, peerIds: List<String>) {
    serialScope.launch { chatController.sendImageMessageToPeers(imagePath, peerIds) }
  }
  /**
   * Send chat file message
   *
   * @param fileUri uri of file
   */
  fun sendFileMessage(fileUri: Uri) {
    sendFileMessage(fileUri, listOf())
  }

  fun sendFileMessage(fileUri: Uri, peerIds: List<String>) {
    serialScope.launch { chatController.sendFileMessageToPeers(fileUri, peerIds) }
  }
  /**
   * Send chat file message
   *
   * @param filePath Location of file on device storage
   */
  fun sendFileMessage(filePath: String) {
    sendFileMessage(filePath, listOf())
  }

  fun sendFileMessage(filePath: String, peerIds: List<String>) {
    serialScope.launch { chatController.sendFileMessageToPeers(filePath, peerIds) }
  }

  /** Creates a chat channel with specified name and userIds as members. */
  fun createChannel(
    name: String,
    memberIds: List<String>,
    displayPictureUrl: String?,
    visibility: ChannelVisibility,
    isDirectMessage: Boolean,
    onResult: (channel: ChatChannel?, errorString: String?) -> Unit,
  ) {
    if (chatChannelController == null) {
      onResult(null, "meeting not of type CHAT")
      return
    }

    serialScope.launch {
      val result =
        chatChannelController.createChannel(
          name,
          memberIds,
          displayPictureUrl,
          visibility,
          isDirectMessage,
        )
      withContext(Dispatchers.Main) {
        when (result) {
          is Result.Success -> {
            onResult(result.value, null)
          }
          is Result.Failure -> {
            onResult(null, result.value)
          }
        }
      }
    }
  }

  /** Updates a chat channel */
  fun updateChannel(
    channelId: String,
    channelUpdateParams: ChatChannelUpdateParams,
    onResult: (channel: ChatChannel?, errorString: String?) -> Unit,
  ) {
    if (chatChannelController == null) {
      onResult(null, "meeting not of type CHAT")
      return
    }

    serialScope.launch {
      val result = chatChannelController.updateChannel(channelId, channelUpdateParams)
      withContext(Dispatchers.Main) {
        when (result) {
          is Result.Success -> {
            onResult(result.value, null)
          }
          is Result.Failure -> {
            onResult(null, result.value)
          }
        }
      }
    }
  }

  /** Returns a list of members added to a channel */
  fun getChannelMembers(
    channelId: String,
    onResult: (channelMembers: List<DyteParticipant>?, errorString: String?) -> Unit,
  ) {
    if (chatChannelController == null) {
      onResult(null, "meeting not of type CHAT")
      return
    }

    serialScope.launch {
      val result = chatChannelController.getChannelMembers(channelId)
      withContext(Dispatchers.Main) {
        when (result) {
          is Result.Success -> {
            onResult(result.value, null)
          }
          is Result.Failure -> {
            onResult(null, result.value)
          }
        }
      }
    }
  }

  /** Gets chat messages in a paginated manner */
  public fun getMessages(
    timestamp: Long,
    size: Int,
    offset: Int = 0,
    reversed: Boolean,
    channelId: String? = null,
    onResult: (messagesResult: GetMessagesResult?, errorString: String?) -> Unit,
  ) {
    serialScope.launch {
      val result = chatController.getMessagesPaginated(timestamp, size, offset, reversed, channelId)
      withContext(Dispatchers.Main) {
        when (result) {
          is Result.Success -> {
            onResult(result.value, null)
          }
          is Result.Failure -> {
            onResult(null, result.value)
          }
        }
      }
    }
  }

  /** Sends a TEXT type chat message to Channel */
  fun sendTextMessageToChannel(channelId: String, message: String) {
    if (chatChannelController == null) {
      return
    }

    serialScope.launch { chatChannelController.sendTextMessageToChannel(channelId, message) }
  }

  /** Sends a IMAGE type chat message to Channel */
  fun sendImageMessageToChannel(channelId: String, imageUri: Uri) {
    if (chatChannelController == null) {
      return
    }

    serialScope.launch { chatChannelController.sendImageMessageToChannel(channelId, imageUri) }
  }

  /** Sends a FILE type chat message to Channel */
  fun sendFileMessageToChannel(channelId: String, fileUri: Uri) {
    if (chatChannelController == null) {
      return
    }

    serialScope.launch { chatChannelController.sendFileMessageToChannel(channelId, fileUri) }
  }
}
