package io.dyte.core.events

import kotlinx.atomicfu.locks.reentrantLock
import kotlinx.atomicfu.locks.withLock

/** A thread-safe event multicaster to dispatch callbacks for specific events. */
/*
 * TODO 1: Integrate removed check in multicast() to avoid giving callback to a removed listener
 * TODO 2: Experiment with WeakReference to hold listener references. If this works, it will improve
 *  the DevX for our iOS clients
 * */
internal open class EventMulticaster<EventKeyType, EventListenerType> {
  private val listenerMap: MutableMap<EventKeyType, MutableSet<EventListenerType>> = HashMap()

  private val listenersLock = reentrantLock()

  open fun addListener(eventKey: EventKeyType, listener: EventListenerType) {
    listenersLock.withLock {
      val eventKeyListeners = listenerMap.getOrPut(eventKey) { mutableSetOf() }
      eventKeyListeners.add(listener)
    }
  }

  open fun removeListener(eventKey: EventKeyType, listener: EventListenerType) {
    listenersLock.withLock { listenerMap[eventKey]?.remove(listener) }
  }

  open fun clearListeners(eventKey: EventKeyType) {
    listenersLock.withLock { listenerMap.remove(eventKey) }
  }

  open fun clearAll() {
    listenersLock.withLock { listenerMap.clear() }
  }

  open fun multicastEvent(eventKey: EventKeyType, eventBlock: (EventListenerType) -> Unit) {
    val eventKeyListeners = listenersLock.withLock { listenerMap[eventKey]?.toList() } ?: return
    eventKeyListeners.forEach {
      try {
        eventBlock(it)
      } catch (_: Exception) {
        // Catch and ignore the exception to continue notifying remaining listeners
      }
    }
  }

  /**
   * Copies the listeners of the `sourceEventKey` if they exist into `destinationEventKey`. NOTE:
   * Currently if destinationEventKey already exists the MultiCaster then its existing listeners are
   * replaced.
   */
  open fun copyListeners(sourceEventKey: EventKeyType, destinationEventKey: EventKeyType) {
    listenersLock.withLock {
      val eventKeyListeners = listenerMap[sourceEventKey]
      if (!eventKeyListeners.isNullOrEmpty()) {
        listenerMap[destinationEventKey] = eventKeyListeners.toMutableSet()
      }
    }
  }
}
